/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium.dtls;

import java.net.InetSocketAddress;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import javax.crypto.Cipher;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import org.eclipse.californium.elements.util.DatagramReader;
import org.eclipse.californium.elements.util.DatagramWriter;
import org.eclipse.californium.elements.util.StringUtil;
import org.eclipse.californium.scandium.dtls.AlertMessage;
import org.eclipse.californium.scandium.dtls.ApplicationMessage;
import org.eclipse.californium.scandium.dtls.ChangeCipherSpecMessage;
import org.eclipse.californium.scandium.dtls.ConnectionId;
import org.eclipse.californium.scandium.dtls.ConnectionIdGenerator;
import org.eclipse.californium.scandium.dtls.ContentType;
import org.eclipse.californium.scandium.dtls.DTLSConnectionState;
import org.eclipse.californium.scandium.dtls.DTLSMessage;
import org.eclipse.californium.scandium.dtls.DTLSSession;
import org.eclipse.californium.scandium.dtls.HandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeMessage;
import org.eclipse.californium.scandium.dtls.HandshakeParameter;
import org.eclipse.californium.scandium.dtls.HandshakeType;
import org.eclipse.californium.scandium.dtls.ProtocolVersion;
import org.eclipse.californium.scandium.dtls.cipher.AeadBlockCipher;
import org.eclipse.californium.scandium.dtls.cipher.CipherSuite;
import org.eclipse.californium.scandium.dtls.cipher.InvalidMacException;
import org.eclipse.californium.scandium.util.ByteArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Record {
    private static final Logger LOGGER = LoggerFactory.getLogger(Record.class.getCanonicalName());
    public static final int CONTENT_TYPE_BITS = 8;
    public static final int VERSION_BITS = 8;
    public static final int EPOCH_BITS = 16;
    public static final int SEQUENCE_NUMBER_BITS = 48;
    public static final int LENGTH_BITS = 16;
    public static final int CID_LENGTH_BITS = 8;
    public static final int RECORD_HEADER_BITS = 104;
    private static final long MAX_SEQUENCE_NO = 0xFFFFFFFFFFFFL;
    private ContentType type;
    private final ProtocolVersion version;
    private final int epoch;
    private long sequenceNumber;
    private DTLSMessage fragment;
    private byte[] fragmentBytes;
    private ConnectionId connectionId;
    private int padding;
    private DTLSSession session;
    private InetSocketAddress peerAddress;

    Record(ContentType type, ProtocolVersion version, int epoch, long sequenceNumber, ConnectionId connectionId, byte[] fragmentBytes, InetSocketAddress peerAddress) {
        this(version, epoch, sequenceNumber);
        if (type == null) {
            throw new NullPointerException("Type must not be null");
        }
        if (fragmentBytes == null) {
            throw new NullPointerException("Fragment bytes must not be null");
        }
        if (peerAddress == null) {
            throw new NullPointerException("Peer address must not be null");
        }
        this.type = type;
        this.connectionId = connectionId;
        this.fragmentBytes = fragmentBytes;
        this.peerAddress = peerAddress;
    }

    public Record(ContentType type, int epoch, long sequenceNumber, DTLSMessage fragment, DTLSSession session, boolean cid, int pad) throws GeneralSecurityException {
        this(new ProtocolVersion(), epoch, sequenceNumber);
        if (fragment == null) {
            throw new NullPointerException("Fragment must not be null");
        }
        if (session == null) {
            throw new NullPointerException("Session must not be null");
        }
        this.setType(type);
        this.session = session;
        this.peerAddress = session.getPeer();
        if (cid) {
            this.connectionId = session.getWriteConnectionId();
            this.padding = pad;
        }
        this.setEncodedFragment(fragment);
        if (this.fragmentBytes == null) {
            throw new IllegalArgumentException("Fragment missing encoded bytes!");
        }
    }

    public Record(ContentType type, long sequenceNumber, DTLSMessage fragment, InetSocketAddress peerAddress) {
        this(new ProtocolVersion(), 0, sequenceNumber);
        if (fragment == null) {
            throw new NullPointerException("Fragment must not be null");
        }
        if (peerAddress == null) {
            throw new NullPointerException("Peer address must not be null");
        }
        this.setType(type);
        this.peerAddress = peerAddress;
        this.fragment = fragment;
        this.fragmentBytes = fragment.toByteArray();
        if (this.fragmentBytes == null) {
            throw new IllegalArgumentException("Fragment missing encoded bytes!");
        }
    }

    private Record(ProtocolVersion version, int epoch, long sequenceNumber) {
        if (sequenceNumber > 0xFFFFFFFFFFFFL) {
            throw new IllegalArgumentException("Sequence number must be 48 bits only! " + sequenceNumber);
        }
        if (sequenceNumber < 0L) {
            throw new IllegalArgumentException("Sequence number must not be less than 0! " + sequenceNumber);
        }
        if (epoch < 0) {
            throw new IllegalArgumentException("Epoch must not be less than 0! " + epoch);
        }
        if (version == null) {
            throw new NullPointerException("Version must not be null");
        }
        this.version = version;
        this.epoch = epoch;
        this.sequenceNumber = sequenceNumber;
    }

    public byte[] toByteArray() {
        DatagramWriter writer = new DatagramWriter();
        if (this.useConnectionId()) {
            writer.write(ContentType.TLS12_CID.getCode(), 8);
        } else {
            writer.write(this.type.getCode(), 8);
        }
        writer.write(this.version.getMajor(), 8);
        writer.write(this.version.getMinor(), 8);
        writer.write(this.epoch, 16);
        writer.writeLong(this.sequenceNumber, 48);
        if (this.useConnectionId()) {
            writer.writeBytes(this.connectionId.getBytes());
        }
        writer.write(this.fragmentBytes.length, 16);
        writer.writeBytes(this.fragmentBytes);
        return writer.toByteArray();
    }

    public static List<Record> fromByteArray(byte[] byteArray, InetSocketAddress peerAddress, ConnectionIdGenerator cidGenerator) {
        if (byteArray == null) {
            throw new NullPointerException("Byte array must not be null");
        }
        if (peerAddress == null) {
            throw new NullPointerException("Peer address must not be null");
        }
        ArrayList<Record> records = new ArrayList<Record>();
        DatagramReader reader = new DatagramReader(byteArray);
        while (reader.bytesAvailable()) {
            if (reader.bitsLeft() < 104) {
                LOGGER.debug("Received truncated DTLS record(s). Discarding ...");
                return records;
            }
            int type = reader.read(8);
            int major = reader.read(8);
            int minor = reader.read(8);
            ProtocolVersion version = new ProtocolVersion(major, minor);
            int epoch = reader.read(16);
            long sequenceNumber = reader.readLong(48);
            ConnectionId connectionId = null;
            if (type == ContentType.TLS12_CID.getCode()) {
                if (cidGenerator == null) {
                    LOGGER.debug("Received TLS_CID record, but cid is not supported. Discarding ...");
                    continue;
                }
                if (cidGenerator.useConnectionId()) {
                    connectionId = cidGenerator.read(reader);
                    if (connectionId == null) {
                        LOGGER.debug("Received TLS_CID record, but cid is not matching. Discarding ...");
                        continue;
                    }
                } else {
                    LOGGER.debug("Received TLS_CID record, but cid is not used. Discarding ...");
                    continue;
                }
            }
            int length = reader.read(16);
            if (reader.bitsLeft() < length) {
                LOGGER.debug("Received truncated DTLS record(s). Discarding ...");
                return records;
            }
            byte[] fragmentBytes = reader.readBytes(length);
            ContentType contentType = ContentType.getTypeByValue(type);
            if (contentType == null) {
                LOGGER.debug("Received DTLS record of unsupported type [{}]. Discarding ...", (Object)type);
                continue;
            }
            records.add(new Record(contentType, version, epoch, sequenceNumber, connectionId, fragmentBytes, peerAddress));
        }
        return records;
    }

    private byte[] encryptFragment(byte[] plaintextFragment) throws GeneralSecurityException {
        if (this.session == null || this.epoch == 0) {
            return plaintextFragment;
        }
        byte[] encryptedFragment = plaintextFragment;
        CipherSuite cipherSuite = this.session.getWriteState().getCipherSuite();
        LOGGER.trace("Encrypting record fragment using current write state{}{}", (Object)StringUtil.lineSeparator(), (Object)this.session.getWriteState());
        switch (cipherSuite.getCipherType()) {
            case NULL: {
                break;
            }
            case AEAD: {
                encryptedFragment = this.encryptAEAD(plaintextFragment);
                break;
            }
            case BLOCK: {
                encryptedFragment = this.encryptBlockCipher(plaintextFragment);
                break;
            }
            case STREAM: {
                break;
            }
        }
        return encryptedFragment;
    }

    private byte[] decryptFragment(byte[] ciphertextFragment, DTLSConnectionState currentReadState) throws GeneralSecurityException {
        if (currentReadState == null) {
            return ciphertextFragment;
        }
        byte[] result = ciphertextFragment;
        CipherSuite cipherSuite = currentReadState.getCipherSuite();
        LOGGER.trace("Decrypting record fragment using current read state{}{}", (Object)StringUtil.lineSeparator(), (Object)currentReadState);
        switch (cipherSuite.getCipherType()) {
            case NULL: {
                break;
            }
            case AEAD: {
                result = this.decryptAEAD(ciphertextFragment, currentReadState);
                break;
            }
            case BLOCK: {
                result = this.decryptBlockCipher(ciphertextFragment, currentReadState);
                break;
            }
            case STREAM: {
                break;
            }
        }
        return result;
    }

    private byte[] encryptBlockCipher(byte[] compressedFragment) throws GeneralSecurityException {
        int smallestMultipleOfBlocksize;
        if (this.session == null) {
            throw new IllegalStateException("DTLS session must be set on record");
        }
        if (compressedFragment == null) {
            throw new NullPointerException("Compressed fragment must not be null");
        }
        DTLSConnectionState writeState = this.session.getWriteState();
        DatagramWriter plaintext = new DatagramWriter();
        plaintext.writeBytes(compressedFragment);
        plaintext.writeBytes(this.getBlockCipherMac(writeState, compressedFragment));
        int ciphertextLength = compressedFragment.length + writeState.getCipherSuite().getMacLength() + 1;
        for (smallestMultipleOfBlocksize = writeState.getRecordIvLength(); smallestMultipleOfBlocksize <= ciphertextLength; smallestMultipleOfBlocksize += writeState.getRecordIvLength()) {
        }
        int paddingLength = smallestMultipleOfBlocksize % ciphertextLength;
        byte[] padding = new byte[paddingLength + 1];
        Arrays.fill(padding, (byte)paddingLength);
        plaintext.writeBytes(padding);
        Cipher blockCipher = writeState.getCipherSuite().getCipher();
        blockCipher.init(1, writeState.getEncryptionKey());
        DatagramWriter result = new DatagramWriter();
        result.writeBytes(blockCipher.getIV());
        result.writeBytes(blockCipher.doFinal(plaintext.toByteArray()));
        return result.toByteArray();
    }

    private byte[] decryptBlockCipher(byte[] ciphertextFragment, DTLSConnectionState currentReadState) throws GeneralSecurityException {
        if (currentReadState == null) {
            throw new NullPointerException("Current read state must not be null");
        }
        if (ciphertextFragment == null) {
            throw new NullPointerException("Ciphertext must not be null");
        }
        DatagramReader reader = new DatagramReader(ciphertextFragment);
        byte[] iv = reader.readBytes(currentReadState.getRecordIvLength());
        Cipher blockCipher = currentReadState.getCipherSuite().getCipher();
        blockCipher.init(2, (Key)currentReadState.getEncryptionKey(), new IvParameterSpec(iv));
        byte[] plaintext = blockCipher.doFinal(reader.readBytesLeft());
        byte paddingLength = plaintext[plaintext.length - 1];
        int fragmentLength = plaintext.length - 1 - paddingLength - currentReadState.getCipherSuite().getMacLength();
        reader = new DatagramReader(plaintext);
        byte[] content = reader.readBytes(fragmentLength);
        byte[] macFromMessage = reader.readBytes(currentReadState.getCipherSuite().getMacLength());
        byte[] mac = this.getBlockCipherMac(currentReadState, content);
        if (Arrays.equals(macFromMessage, mac)) {
            return content;
        }
        throw new InvalidMacException(mac, macFromMessage);
    }

    private byte[] getBlockCipherMac(DTLSConnectionState conState, byte[] content) throws GeneralSecurityException {
        Mac hmac = Mac.getInstance(conState.getCipherSuite().getMacName());
        hmac.init(conState.getMacKey());
        DatagramWriter mac = new DatagramWriter();
        mac.writeBytes(this.generateAdditionalData(content.length));
        mac.writeBytes(content);
        return hmac.doFinal(mac.toByteArray());
    }

    protected byte[] encryptAEAD(byte[] byteArray) throws GeneralSecurityException {
        DTLSConnectionState writeState = this.session.getWriteState();
        byte[] iv = writeState.getIv().getIV();
        byte[] explicitNonce = this.generateExplicitNonce();
        byte[] nonce = ByteArrayUtils.concatenate(iv, explicitNonce);
        byte[] additionalData = this.generateAdditionalData(byteArray.length);
        SecretKey key = writeState.getEncryptionKey();
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("encrypt: {} bytes", (Object)byteArray.length);
            LOGGER.trace("nonce: {}", (Object)StringUtil.byteArray2HexString(nonce));
            LOGGER.trace("adata: {}", (Object)StringUtil.byteArray2HexString(additionalData));
        }
        byte[] encryptedFragment = AeadBlockCipher.encrypt(writeState.getCipherSuite(), key, nonce, additionalData, byteArray);
        encryptedFragment = ByteArrayUtils.concatenate(explicitNonce, encryptedFragment);
        LOGGER.trace("==> {} bytes", (Object)encryptedFragment.length);
        return encryptedFragment;
    }

    protected byte[] decryptAEAD(byte[] byteArray, DTLSConnectionState currentReadState) throws GeneralSecurityException {
        byte[] explicitNonce;
        if (currentReadState == null) {
            throw new NullPointerException("Current read state must not be null");
        }
        if (byteArray == null) {
            throw new NullPointerException("Ciphertext must not be null");
        }
        CipherSuite cipherSuite = currentReadState.getCipherSuite();
        byte[] iv = currentReadState.getIv().getIV();
        SecretKey key = currentReadState.getEncryptionKey();
        int applicationDataLength = byteArray.length - cipherSuite.getRecordIvLength() - cipherSuite.getMacLength();
        byte[] additionalData = this.generateAdditionalData(applicationDataLength);
        DatagramReader reader = new DatagramReader(byteArray);
        byte[] explicitNonceUsed = reader.readBytes(cipherSuite.getRecordIvLength());
        byte[] nonce = ByteArrayUtils.concatenate(iv, explicitNonceUsed);
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("decrypt: {} bytes", (Object)applicationDataLength);
            LOGGER.trace("nonce: {}", (Object)StringUtil.byteArray2HexString(nonce));
            LOGGER.trace("adata: {}", (Object)StringUtil.byteArray2HexString(additionalData));
        }
        if (LOGGER.isDebugEnabled() && "AES/CCM".equals(cipherSuite.getTransformation()) && !Arrays.equals(explicitNonce = this.generateExplicitNonce(), explicitNonceUsed)) {
            StringBuilder b = new StringBuilder("The explicit nonce used by the sender does not match the values provided in the DTLS record");
            b.append(StringUtil.lineSeparator()).append("Used    : ").append(StringUtil.byteArray2HexString(explicitNonceUsed));
            b.append(StringUtil.lineSeparator()).append("Expected: ").append(StringUtil.byteArray2HexString(explicitNonce));
            LOGGER.debug(b.toString());
        }
        return AeadBlockCipher.decrypt(cipherSuite, key, nonce, additionalData, reader.readBytesLeft());
    }

    private byte[] generateExplicitNonce() {
        DatagramWriter writer = new DatagramWriter();
        writer.write(this.epoch, 16);
        writer.writeLong(this.sequenceNumber, 48);
        return writer.toByteArray();
    }

    private byte[] generateAdditionalData(int length) {
        DatagramWriter writer = new DatagramWriter();
        writer.write(this.epoch, 16);
        writer.writeLong(this.sequenceNumber, 48);
        if (this.useConnectionId()) {
            writer.write(ContentType.TLS12_CID.getCode(), 8);
        } else {
            writer.write(this.type.getCode(), 8);
        }
        writer.write(this.version.getMajor(), 8);
        writer.write(this.version.getMinor(), 8);
        if (this.useConnectionId()) {
            writer.writeBytes(this.connectionId.getBytes());
            writer.write(this.connectionId.length(), 8);
        }
        writer.write(length, 16);
        return writer.toByteArray();
    }

    public boolean isNewClientHello() {
        if (this.fragmentBytes == null) {
            throw new IllegalStateException("Fragment bytes must not be null!");
        }
        if (0 < this.epoch || this.type != ContentType.HANDSHAKE || 0 == this.fragmentBytes.length) {
            return false;
        }
        HandshakeType handshakeType = HandshakeType.getTypeByCode(this.fragmentBytes[0]);
        return handshakeType == HandshakeType.CLIENT_HELLO;
    }

    public ContentType getType() {
        return this.type;
    }

    public ProtocolVersion getVersion() {
        return this.version;
    }

    public int getEpoch() {
        return this.epoch;
    }

    public long getSequenceNumber() {
        return this.sequenceNumber;
    }

    public void updateSequenceNumber(long sequenceNumber) throws GeneralSecurityException {
        if (sequenceNumber > 0xFFFFFFFFFFFFL) {
            throw new IllegalArgumentException("Sequence number must have max 48 bits");
        }
        if (sequenceNumber < 0L) {
            throw new IllegalArgumentException("Sequence number must not be smaller than 0! " + sequenceNumber);
        }
        if (this.fragment == null) {
            throw new IllegalStateException("Fragment must not be null!");
        }
        if (this.sequenceNumber != sequenceNumber) {
            this.sequenceNumber = sequenceNumber;
            if (this.session != null && this.session.getWriteState() != null && this.epoch > 0) {
                this.setEncodedFragment(this.fragment);
            }
        }
    }

    public int getFragmentLength() {
        return this.fragmentBytes.length;
    }

    public void setSession(DTLSSession session) {
        this.session = session;
    }

    public InetSocketAddress getPeerAddress() {
        if (this.peerAddress != null) {
            return this.peerAddress;
        }
        throw new IllegalStateException("Record does not have a peer address");
    }

    public ConnectionId getConnectionId() {
        return this.connectionId;
    }

    public byte[] getFragmentBytes() {
        return this.fragmentBytes;
    }

    public DTLSMessage getFragment() throws GeneralSecurityException, HandshakeException {
        if (this.session != null) {
            return this.getFragment(this.session.getReadState());
        }
        return this.getFragment(null);
    }

    public DTLSMessage getFragment(DTLSConnectionState currentReadState) throws GeneralSecurityException, HandshakeException {
        if (this.fragment == null) {
            ContentType actualType = this.type;
            byte[] decryptedMessage = this.decryptFragment(this.fragmentBytes, currentReadState);
            if (ContentType.TLS12_CID == this.type) {
                int index;
                for (index = decryptedMessage.length - 1; index >= 0 && decryptedMessage[index] == 0; --index) {
                }
                if (index < 0) {
                    throw new GeneralSecurityException("no inner type!");
                }
                byte typeCode = decryptedMessage[index];
                actualType = ContentType.getTypeByValue(typeCode);
                if (actualType == null) {
                    throw new GeneralSecurityException("unknown inner type! " + typeCode);
                }
                decryptedMessage = Arrays.copyOf(decryptedMessage, index);
            }
            switch (actualType) {
                case ALERT: {
                    this.fragment = AlertMessage.fromByteArray(decryptedMessage, this.getPeerAddress());
                    break;
                }
                case APPLICATION_DATA: {
                    this.fragment = ApplicationMessage.fromByteArray(decryptedMessage, this.getPeerAddress());
                    break;
                }
                case CHANGE_CIPHER_SPEC: {
                    this.fragment = ChangeCipherSpecMessage.fromByteArray(decryptedMessage, this.getPeerAddress());
                    break;
                }
                case HANDSHAKE: {
                    this.fragment = this.decryptHandshakeMessage(decryptedMessage);
                    break;
                }
                default: {
                    LOGGER.warn("Cannot decrypt message of unsupported type [{}]", (Object)this.type);
                }
            }
            this.type = actualType;
        }
        return this.fragment;
    }

    private DTLSMessage decryptHandshakeMessage(byte[] decryptedMessage) throws GeneralSecurityException, HandshakeException {
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Decrypting HANDSHAKE message ciphertext{}{}", (Object)StringUtil.lineSeparator(), (Object)StringUtil.byteArray2HexString(decryptedMessage));
        }
        HandshakeParameter parameter = null;
        if (this.session != null) {
            parameter = this.session.getParameter();
        } else {
            LOGGER.debug("Parsing message without a session");
        }
        if (LOGGER.isDebugEnabled()) {
            StringBuilder msg = new StringBuilder("Parsing HANDSHAKE message plaintext [{}]");
            if (LOGGER.isTraceEnabled()) {
                msg.append(":").append(StringUtil.lineSeparator()).append(StringUtil.byteArray2HexString(decryptedMessage));
            }
            LOGGER.debug(msg.toString(), (Object)parameter);
        }
        return HandshakeMessage.fromByteArray(decryptedMessage, parameter, this.getPeerAddress());
    }

    private void setEncodedFragment(DTLSMessage fragment) throws GeneralSecurityException {
        byte[] byteArray = fragment.toByteArray();
        if (this.useConnectionId()) {
            int index = byteArray.length;
            byteArray = Arrays.copyOf(byteArray, index + 1 + this.padding);
            byteArray[index] = (byte)this.type.getCode();
        }
        this.fragmentBytes = this.encryptFragment(byteArray);
        this.fragment = fragment;
    }

    private boolean useConnectionId() {
        return this.connectionId != null && !this.connectionId.isEmpty();
    }

    private void setType(ContentType type) {
        if (type == null) {
            throw new NullPointerException("Type must not be null");
        }
        switch (type) {
            case ALERT: 
            case APPLICATION_DATA: 
            case CHANGE_CIPHER_SPEC: 
            case HANDSHAKE: {
                this.type = type;
                break;
            }
            default: {
                throw new IllegalArgumentException("Not supported content type: " + (Object)((Object)type));
            }
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("==[ DTLS Record ]==============================================");
        sb.append(StringUtil.lineSeparator()).append("Content Type: ").append(this.type.toString());
        sb.append(StringUtil.lineSeparator()).append("Peer address: ").append(this.getPeerAddress());
        sb.append(StringUtil.lineSeparator()).append("Version: ").append(this.version.getMajor()).append(", ").append(this.version.getMinor());
        sb.append(StringUtil.lineSeparator()).append("Epoch: ").append(this.epoch);
        sb.append(StringUtil.lineSeparator()).append("Sequence Number: ").append(this.sequenceNumber);
        if (this.connectionId != null) {
            sb.append(StringUtil.lineSeparator()).append("connection id: ").append(this.connectionId.getAsString());
        }
        sb.append(StringUtil.lineSeparator()).append("Length: ").append(this.fragmentBytes.length);
        sb.append(StringUtil.lineSeparator()).append("Fragment:");
        if (this.fragment != null) {
            sb.append(StringUtil.lineSeparator()).append(this.fragment);
        } else {
            sb.append(StringUtil.lineSeparator()).append("fragment is not decrypted yet");
        }
        sb.append(StringUtil.lineSeparator()).append("===============================================================");
        return sb.toString();
    }
}

