/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.cloud.resources;

import java.io.StringWriter;
import java.security.Principal;
import java.security.cert.CertificateEncodingException;
import java.security.cert.X509Certificate;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.concurrent.atomic.AtomicBoolean;
import org.eclipse.californium.cloud.option.TimeOption;
import org.eclipse.californium.cloud.resources.ProtectedCoapResource;
import org.eclipse.californium.cloud.util.DeviceProvisioningConsumer;
import org.eclipse.californium.cloud.util.PrincipalInfo;
import org.eclipse.californium.cloud.util.PrincipalInfoProvider;
import org.eclipse.californium.cloud.util.ResultConsumer;
import org.eclipse.californium.core.coap.CoAP;
import org.eclipse.californium.core.coap.Response;
import org.eclipse.californium.core.network.Exchange;
import org.eclipse.californium.core.server.resources.CoapExchange;
import org.eclipse.californium.elements.auth.X509CertPath;
import org.eclipse.californium.elements.util.PemUtil;
import org.eclipse.californium.elements.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Provisioning
extends ProtectedCoapResource {
    private static final Logger LOGGER = LoggerFactory.getLogger(Provisioning.class);
    private static final SimpleDateFormat ISO_DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
    public static final String RESOURCE_NAME = "prov";
    private final int[] CONTENT_TYPES = new int[]{0};
    private final DeviceProvisioningConsumer devices;

    public Provisioning(DeviceProvisioningConsumer devices) {
        super(RESOURCE_NAME, PrincipalInfo.Type.CA, PrincipalInfo.Type.PROVISIONING);
        Arrays.sort(this.CONTENT_TYPES);
        this.getAttributes().setTitle("Device provisioning resource.");
        this.getAttributes().addContentTypes(this.CONTENT_TYPES);
        this.devices = devices;
    }

    @Override
    public void handlePOST(final CoapExchange exchange) {
        int format = exchange.getRequestOptions().getContentFormat();
        if (format != -1 && Arrays.binarySearch(this.CONTENT_TYPES, format) < 0) {
            exchange.respond(CoAP.ResponseCode.NOT_ACCEPTABLE);
            return;
        }
        Principal principal = this.getPrincipal(exchange);
        PrincipalInfo info = this.getPrincipalInfo(exchange);
        if (info.type == PrincipalInfo.Type.DEVICE && (info = Provisioning.getDeviceInfoCa(principal)) == null) {
            LOGGER.info("CA not available.");
            exchange.respond(CoAP.ResponseCode.FORBIDDEN, "CA not available.");
            return;
        }
        if (this.allowed(info.type)) {
            String payload = exchange.getRequestText();
            if (payload.isEmpty()) {
                exchange.respond(CoAP.ResponseCode.BAD_REQUEST, "Missing provisioning payload.");
            } else {
                try {
                    final TimeOption timeOption = TimeOption.getMessageTime(exchange.advanced().getRequest());
                    long time = timeOption.getLongValue();
                    payload = "# added " + Provisioning.format(time) + " by " + info.name + StringUtil.lineSeparator() + payload;
                    if (info.type == PrincipalInfo.Type.CA) {
                        X509CertPath certPath = (X509CertPath)principal;
                        try {
                            StringWriter writer = new StringWriter();
                            byte[] data = certPath.getTarget().getEncoded();
                            writer.write(StringUtil.lineSeparator());
                            writer.write(".x509");
                            writer.write(61);
                            writer.write(StringUtil.lineSeparator());
                            PemUtil.write("CERTIFICATE", data, writer);
                            payload = payload + writer.toString();
                        }
                        catch (CertificateEncodingException certificateEncodingException) {
                            // empty catch block
                        }
                    }
                    LOGGER.debug("{}", (Object)payload);
                    this.devices.add(info, time, payload, new ResultConsumer(){
                        private final AtomicBoolean done = new AtomicBoolean();

                        @Override
                        public void results(ResultConsumer.ResultCode code, String message) {
                            if (this.done.compareAndSet(false, true)) {
                                CoAP.ResponseCode responseCode = CoAP.ResponseCode.INTERNAL_SERVER_ERROR;
                                switch (code) {
                                    case SUCCESS: {
                                        responseCode = CoAP.ResponseCode.CHANGED;
                                        break;
                                    }
                                    case PROVISIONING_ERROR: {
                                        responseCode = CoAP.ResponseCode.CONFLICT;
                                        break;
                                    }
                                }
                                Response response = new Response(responseCode);
                                response.setPayload(message);
                                response.getOptions().setContentFormat(0);
                                TimeOption responseTimeOption = timeOption.adjust();
                                if (responseTimeOption != null) {
                                    response.getOptions().addOtherOption(responseTimeOption);
                                }
                                exchange.respond(response);
                                LOGGER.info("{}", (Object)response);
                            }
                        }
                    });
                }
                catch (Throwable t) {
                    exchange.respond(CoAP.ResponseCode.INTERNAL_SERVER_ERROR, "Provisioning failed, " + t.getMessage());
                }
            }
        } else {
            LOGGER.info("No permission to added device credentials.");
            exchange.respond(CoAP.ResponseCode.FORBIDDEN, "No provisioning permission.");
        }
    }

    public static PrincipalInfo getDeviceInfoCa(Principal principal) {
        if (!(principal instanceof X509CertPath)) {
            LOGGER.warn("Principal is not X.509 based! {}", (Object)principal.getClass().getSimpleName());
            return null;
        }
        X509CertPath x509 = (X509CertPath)principal;
        PrincipalInfoProvider provider = x509.getExtendedInfo().get("provider", PrincipalInfoProvider.class);
        if (provider == null) {
            LOGGER.warn("Principal has no device-info-provider assigned!");
            return null;
        }
        X509Certificate anchor = x509.getAnchor();
        if (anchor == null) {
            LOGGER.warn("Principal has no CA.");
            return null;
        }
        return provider.getPrincipalInfo(X509CertPath.fromCertificatesChain(anchor));
    }

    @Override
    protected CoAP.ResponseCode checkPermission(Exchange exchange) {
        PrincipalInfo info = this.getPrincipalInfo(exchange);
        if (info == null) {
            return CoAP.ResponseCode.UNAUTHORIZED;
        }
        if (info.type == PrincipalInfo.Type.DEVICE ? !(this.getPrincipal(exchange) instanceof X509CertPath) : !this.allowed(info.type)) {
            return CoAP.ResponseCode.FORBIDDEN;
        }
        return this.checkOperationPermission(info, exchange, exchange.getRequest().getCode().write);
    }

    public static String format(long millis) {
        return ISO_DATE_FORMAT.format(new Date(millis));
    }
}

