/*
 *  ******************************************************************************
 *  *
 *  *
 *  * This program and the accompanying materials are made available under the
 *  * terms of the Apache License, Version 2.0 which is available at
 *  * https://www.apache.org/licenses/LICENSE-2.0.
 *  *
 *  *  See the NOTICE file distributed with this work for additional
 *  *  information regarding copyright ownership.
 *  * Unless required by applicable law or agreed to in writing, software
 *  * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 *  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 *  * License for the specific language governing permissions and limitations
 *  * under the License.
 *  *
 *  * SPDX-License-Identifier: Apache-2.0
 *  *****************************************************************************
 */

package org.datavec.api.transform.metadata;

import lombok.Data;
import lombok.EqualsAndHashCode;
import org.datavec.api.transform.ColumnType;
import org.datavec.api.writable.Writable;
import org.nd4j.shade.jackson.annotation.JsonProperty;

@Data
@EqualsAndHashCode(callSuper = true)
public class IntegerMetaData extends BaseColumnMetaData {

    //min/max are nullable: null -> no restriction on min/max values
    private final Integer minAllowedValue;
    private final Integer maxAllowedValue;

    public IntegerMetaData(String name) {
        this(name, null, null);
    }

    /**
     * @param min Min allowed value. If null: no restriction on min value in this column
     * @param max Max allowed value. If null: no restriction on max value in this column
     */
    public IntegerMetaData(@JsonProperty("name") String name, @JsonProperty("minAllowedValue") Integer min,
                    @JsonProperty("maxAllowedValue") Integer max) {
        super(name);
        this.minAllowedValue = min;
        this.maxAllowedValue = max;
    }

    @Override
    public ColumnType getColumnType() {
        return ColumnType.Integer;
    }

    @Override
    public boolean isValid(Writable writable) {
        int value;
        try {
            value = Integer.parseInt(writable.toString());
        } catch (NumberFormatException e) {
            return false;
        }

        if (minAllowedValue != null && value < minAllowedValue)
            return false;
        if (maxAllowedValue != null && value > maxAllowedValue)
            return false;
        return true;
    }

    /**
     * Is the given object valid for this column,
     * given the column type and any
     * restrictions given by the
     * ColumnMetaData object?
     *
     * @param input object to check
     * @return true if value, false if invalid
     */
    @Override
    public boolean isValid(Object input) {
        int value;
        try {
            value = Integer.parseInt(input.toString());
        } catch (NumberFormatException e) {
            return false;
        }

        if (minAllowedValue != null && value < minAllowedValue)
            return false;
        if (maxAllowedValue != null && value > maxAllowedValue)
            return false;
        return true;
    }

    @Override
    public IntegerMetaData clone() {
        return new IntegerMetaData(name, minAllowedValue, maxAllowedValue);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("IntegerMetaData(name=\"").append(name).append("\",");
        if (minAllowedValue != null)
            sb.append("minAllowed=").append(minAllowedValue);
        if (maxAllowedValue != null) {
            if (minAllowedValue != null)
                sb.append(",");
            sb.append("maxAllowed=").append(maxAllowedValue);
        }
        sb.append(")");
        return sb.toString();
    }
}
