/**********************************************************************
Copyright (c) 2006 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.rdbms.datasource.proxool;

import java.util.Properties;

import javax.sql.DataSource;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.OMFContext;
import org.datanucleus.PersistenceConfiguration;
import org.datanucleus.store.StoreManager;
import org.datanucleus.store.rdbms.datasource.DatastoreDriverNotFoundException;
import org.datanucleus.store.rdbms.datasource.DatastorePoolException;
import org.datanucleus.store.rdbms.datasource.DataNucleusDataSourceFactory;
import org.datanucleus.util.ClassUtils;

/**
 * Plugin for the creation of a Proxool connection pool.
 * Note that all Proxool classes are named explicitly in the code to avoid loading
 * them at class initialisation. (see http://proxool.sourceforge.net/)
 */
public class ProxoolDataSourceFactory implements DataNucleusDataSourceFactory
{
    /** Number of the pool being created (using in the Proxool alias). */
    private static int poolNumber = 0;

    /**
     * Method to make a Proxool DataSource for use internally.
     * @param omfCtx OMFContext
     * @return The DataSource
     * @throws Exception Thrown if an error occurs during creation
     */
    public DataSource makePooledDataSource(OMFContext omfCtx)
    {
        PersistenceConfiguration conf = omfCtx.getPersistenceConfiguration();
        StoreManager storeMgr = omfCtx.getStoreManager();
        String dbDriver = storeMgr.getConnectionDriverName();
        String dbURL = storeMgr.getConnectionURL();
        String dbUser = storeMgr.getConnectionUserName();
        if (dbUser == null)
        {
            dbUser = ""; // Some RDBMS (e.g Postgresql) don't like null usernames
        }
        String dbPassword = storeMgr.getConnectionPassword();
        if (dbPassword == null)
        {
            dbPassword = ""; // Some RDBMS (e.g Postgresql) don't like null passwords
        }
        ClassLoaderResolver clr = omfCtx.getClassLoaderResolver(null);

        // Load the database driver
        try
        {
            Class.forName(dbDriver);
        }
        catch (ClassNotFoundException cnfe)
        {
            try
            {
                clr.classForName(dbDriver);
            }
            catch (RuntimeException e)
            {
                // JDBC driver not found
                throw new DatastoreDriverNotFoundException(dbDriver);
            }
        }

        // Check the presence of commons-logging
        ClassUtils.assertClassForJarExistsInClasspath(clr, 
            "org.apache.commons.logging.Log", "commons-logging.jar");
        ClassUtils.assertClassForJarExistsInClasspath(clr, 
            "org.logicalcobwebs.proxool.ProxoolDriver", "proxool.jar");

        // Create a Proxool pool with alias "datanucleus{poolNumber}"
        String alias = "datanucleus" + poolNumber;
        try
        {
            // Apply any properties
            Properties props = new Properties();
            if (conf.hasProperty("datanucleus.connectionPool.maxConnections"))
            {
                int value = conf.getIntProperty("datanucleus.connectionPool.maxConnections");
                if (value > 0)
                {
                    props.put("proxool.maximum-connection-count", "" + value);
                }
                else
                {
                    props.put("proxool.maximum-connection-count", "10");
                }
            }
            else
            {
                props.put("proxool.maximum-connection-count", "10");
            }
            if (conf.hasProperty("datanucleus.connectionPool.testSQL"))
            {
                String value = conf.getStringProperty("datanucleus.connectionPool.testSQL");
                props.put("proxool.house-keeping-test-sql", value);
            }
            else
            {
                props.put("proxool.house-keeping-test-sql", "SELECT 1");
            }

            // Add the user/password to the properties
            props.setProperty("user", dbUser);
            props.setProperty("password", dbPassword);

            String url = "proxool." + alias + ":" + dbDriver + ":" + dbURL;
            poolNumber++;
            org.logicalcobwebs.proxool.ProxoolFacade.registerConnectionPool(url, props);
        }
        catch (org.logicalcobwebs.proxool.ProxoolException pe)
        {
            pe.printStackTrace();
            throw new DatastorePoolException("Proxool", dbDriver, dbURL, pe);
        }

        DataSource ds = new org.logicalcobwebs.proxool.ProxoolDataSource(alias);

        return ds;
    }
}