/*
 * Decompiled with CFR 0.152.
 */
package org.conscrypt;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.security.Principal;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLPeerUnverifiedException;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSessionContext;
import javax.security.cert.X509Certificate;
import org.conscrypt.AbstractSessionContext;
import org.conscrypt.ActiveSession;
import org.conscrypt.ClientSessionContext;
import org.conscrypt.NativeCrypto;
import org.conscrypt.NativeRef;
import org.conscrypt.OpenSSLX509Certificate;
import org.conscrypt.SSLUtils;
import org.conscrypt.SslWrapper;

abstract class SslSessionWrapper {
    private static final Logger logger = Logger.getLogger(SslSessionWrapper.class.getName());

    SslSessionWrapper() {
    }

    static SslSessionWrapper newInstance(NativeRef.SSL_SESSION ref, ActiveSession activeSession) throws SSLPeerUnverifiedException {
        AbstractSessionContext context = (AbstractSessionContext)activeSession.getSessionContext();
        if (context instanceof ClientSessionContext) {
            return new Impl(context, ref, activeSession.getPeerHost(), activeSession.getPeerPort(), activeSession.getPeerCertificates(), SslSessionWrapper.getOcspResponse(activeSession), activeSession.getPeerSignedCertificateTimestamp());
        }
        return new Impl(context, ref, null, -1, null, null, null);
    }

    private static byte[] getOcspResponse(ActiveSession activeSession) {
        List<byte[]> ocspResponseList = activeSession.getStatusResponses();
        if (ocspResponseList.size() >= 1) {
            return ocspResponseList.get(0);
        }
        return null;
    }

    static SslSessionWrapper newInstance(AbstractSessionContext context, byte[] data, String host, int port) {
        ByteBuffer buf = ByteBuffer.wrap(data);
        try {
            int type = buf.getInt();
            if (!SSLUtils.SessionType.isSupportedType(type)) {
                throw new IOException("Unexpected type ID: " + type);
            }
            int length = buf.getInt();
            SslSessionWrapper.checkRemaining(buf, length);
            byte[] sessionData = new byte[length];
            buf.get(sessionData);
            int count = buf.getInt();
            SslSessionWrapper.checkRemaining(buf, count);
            java.security.cert.X509Certificate[] peerCerts = new java.security.cert.X509Certificate[count];
            for (int i = 0; i < count; ++i) {
                length = buf.getInt();
                SslSessionWrapper.checkRemaining(buf, length);
                byte[] certData = new byte[length];
                buf.get(certData);
                try {
                    peerCerts[i] = OpenSSLX509Certificate.fromX509Der(certData);
                    continue;
                }
                catch (Exception e) {
                    throw new IOException("Can not read certificate " + i + "/" + count);
                }
            }
            byte[] ocspData = null;
            if (type >= SSLUtils.SessionType.OPEN_SSL_WITH_OCSP.value) {
                int countOcspResponses = buf.getInt();
                SslSessionWrapper.checkRemaining(buf, countOcspResponses);
                if (countOcspResponses >= 1) {
                    int ocspLength = buf.getInt();
                    SslSessionWrapper.checkRemaining(buf, ocspLength);
                    ocspData = new byte[ocspLength];
                    buf.get(ocspData);
                    for (int i = 1; i < countOcspResponses; ++i) {
                        ocspLength = buf.getInt();
                        SslSessionWrapper.checkRemaining(buf, ocspLength);
                        buf.position(buf.position() + ocspLength);
                    }
                }
            }
            byte[] tlsSctData = null;
            if (type == SSLUtils.SessionType.OPEN_SSL_WITH_TLS_SCT.value) {
                int tlsSctDataLength = buf.getInt();
                SslSessionWrapper.checkRemaining(buf, tlsSctDataLength);
                if (tlsSctDataLength > 0) {
                    tlsSctData = new byte[tlsSctDataLength];
                    buf.get(tlsSctData);
                }
            }
            if (buf.remaining() != 0) {
                SslSessionWrapper.log((Throwable)((Object)new AssertionError((Object)"Read entire session, but data still remains; rejecting")));
                return null;
            }
            NativeRef.SSL_SESSION ref = new NativeRef.SSL_SESSION(NativeCrypto.d2i_SSL_SESSION(sessionData));
            return new Impl(context, ref, host, port, peerCerts, ocspData, tlsSctData);
        }
        catch (IOException e) {
            SslSessionWrapper.log(e);
            return null;
        }
        catch (BufferUnderflowException e) {
            SslSessionWrapper.log(e);
            return null;
        }
    }

    abstract byte[] getId();

    abstract boolean isValid();

    abstract void offerToResume(SslWrapper var1) throws SSLException;

    abstract String getCipherSuite();

    abstract String getProtocol();

    abstract String getPeerHost();

    abstract int getPeerPort();

    abstract byte[] getPeerOcspStapledResponse();

    abstract byte[] getPeerSignedCertificateTimestamp();

    abstract byte[] toBytes();

    abstract SSLSession toSSLSession();

    private static void log(Throwable t) {
        logger.log(Level.INFO, "Error inflating SSL session: {0}", t.getMessage() != null ? t.getMessage() : t.getClass().getName());
    }

    private static void checkRemaining(ByteBuffer buf, int length) throws IOException {
        if (length < 0) {
            throw new IOException("Length is negative: " + length);
        }
        if (length > buf.remaining()) {
            throw new IOException("Length of blob is longer than available: " + length + " > " + buf.remaining());
        }
    }

    private static final class Impl
    extends SslSessionWrapper {
        private final NativeRef.SSL_SESSION ref;
        private final AbstractSessionContext context;
        private final String host;
        private final int port;
        private final String protocol;
        private final String cipherSuite;
        private final java.security.cert.X509Certificate[] peerCertificates;
        private final byte[] peerOcspStapledResponse;
        private final byte[] peerSignedCertificateTimestamp;

        private Impl(AbstractSessionContext context, NativeRef.SSL_SESSION ref, String host, int port, java.security.cert.X509Certificate[] peerCertificates, byte[] peerOcspStapledResponse, byte[] peerSignedCertificateTimestamp) {
            this.context = context;
            this.host = host;
            this.port = port;
            this.peerCertificates = peerCertificates;
            this.peerOcspStapledResponse = peerOcspStapledResponse;
            this.peerSignedCertificateTimestamp = peerSignedCertificateTimestamp;
            this.protocol = NativeCrypto.SSL_SESSION_get_version(ref.context);
            this.cipherSuite = NativeCrypto.cipherSuiteToJava(NativeCrypto.SSL_SESSION_cipher(ref.context));
            this.ref = ref;
        }

        @Override
        byte[] getId() {
            return NativeCrypto.SSL_SESSION_session_id(this.ref.context);
        }

        private long getCreationTime() {
            return NativeCrypto.SSL_SESSION_get_time(this.ref.context);
        }

        @Override
        boolean isValid() {
            long creationTimeMillis = this.getCreationTime();
            long timeoutMillis = Math.max(0L, Math.min((long)this.context.getSessionTimeout(), NativeCrypto.SSL_SESSION_get_timeout(this.ref.context))) * 1000L;
            return System.currentTimeMillis() - timeoutMillis < creationTimeMillis;
        }

        @Override
        void offerToResume(SslWrapper ssl) throws SSLException {
            ssl.offerToResumeSession(this.ref.context);
        }

        @Override
        String getCipherSuite() {
            return this.cipherSuite;
        }

        @Override
        String getProtocol() {
            return this.protocol;
        }

        @Override
        String getPeerHost() {
            return this.host;
        }

        @Override
        int getPeerPort() {
            return this.port;
        }

        @Override
        byte[] getPeerOcspStapledResponse() {
            return this.peerOcspStapledResponse;
        }

        @Override
        byte[] getPeerSignedCertificateTimestamp() {
            return this.peerSignedCertificateTimestamp;
        }

        @Override
        byte[] toBytes() {
            try {
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                DataOutputStream daos = new DataOutputStream(baos);
                daos.writeInt(SSLUtils.SessionType.OPEN_SSL_WITH_TLS_SCT.value);
                byte[] data = NativeCrypto.i2d_SSL_SESSION(this.ref.context);
                daos.writeInt(data.length);
                daos.write(data);
                daos.writeInt(this.peerCertificates.length);
                for (java.security.cert.X509Certificate cert : this.peerCertificates) {
                    data = cert.getEncoded();
                    daos.writeInt(data.length);
                    daos.write(data);
                }
                if (this.peerOcspStapledResponse != null) {
                    daos.writeInt(1);
                    daos.writeInt(this.peerOcspStapledResponse.length);
                    daos.write(this.peerOcspStapledResponse);
                } else {
                    daos.writeInt(0);
                }
                if (this.peerSignedCertificateTimestamp != null) {
                    daos.writeInt(this.peerSignedCertificateTimestamp.length);
                    daos.write(this.peerSignedCertificateTimestamp);
                } else {
                    daos.writeInt(0);
                }
                return baos.toByteArray();
            }
            catch (IOException e) {
                logger.log(Level.WARNING, "Failed to convert saved SSL Session: ", e);
                return null;
            }
            catch (CertificateEncodingException e) {
                SslSessionWrapper.log(e);
                return null;
            }
        }

        @Override
        SSLSession toSSLSession() {
            return new SSLSession(){

                @Override
                public byte[] getId() {
                    return Impl.this.getId();
                }

                @Override
                public String getCipherSuite() {
                    return Impl.this.getCipherSuite();
                }

                @Override
                public String getProtocol() {
                    return Impl.this.getProtocol();
                }

                @Override
                public String getPeerHost() {
                    return Impl.this.getPeerHost();
                }

                @Override
                public int getPeerPort() {
                    return Impl.this.getPeerPort();
                }

                @Override
                public long getCreationTime() {
                    return Impl.this.getCreationTime();
                }

                @Override
                public boolean isValid() {
                    return Impl.this.isValid();
                }

                @Override
                public SSLSessionContext getSessionContext() {
                    throw new UnsupportedOperationException();
                }

                @Override
                public long getLastAccessedTime() {
                    throw new UnsupportedOperationException();
                }

                @Override
                public void invalidate() {
                    throw new UnsupportedOperationException();
                }

                @Override
                public void putValue(String s, Object o) {
                    throw new UnsupportedOperationException();
                }

                @Override
                public Object getValue(String s) {
                    throw new UnsupportedOperationException();
                }

                @Override
                public void removeValue(String s) {
                    throw new UnsupportedOperationException();
                }

                @Override
                public String[] getValueNames() {
                    throw new UnsupportedOperationException();
                }

                @Override
                public Certificate[] getPeerCertificates() throws SSLPeerUnverifiedException {
                    throw new UnsupportedOperationException();
                }

                @Override
                public Certificate[] getLocalCertificates() {
                    throw new UnsupportedOperationException();
                }

                @Override
                public X509Certificate[] getPeerCertificateChain() throws SSLPeerUnverifiedException {
                    throw new UnsupportedOperationException();
                }

                @Override
                public Principal getPeerPrincipal() throws SSLPeerUnverifiedException {
                    throw new UnsupportedOperationException();
                }

                @Override
                public Principal getLocalPrincipal() {
                    throw new UnsupportedOperationException();
                }

                @Override
                public int getPacketBufferSize() {
                    throw new UnsupportedOperationException();
                }

                @Override
                public int getApplicationBufferSize() {
                    throw new UnsupportedOperationException();
                }
            };
        }
    }
}

