/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package org.graalvm.compiler.code;

import org.checkerframework.checker.nullness.qual.EnsuresNonNullIf;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.checkerframework.dataflow.qual.Pure;
import org.checkerframework.dataflow.qual.SideEffectFree;
import static jdk.vm.ci.meta.MetaUtil.identityHashCodeString;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Objects;
import java.util.function.BiConsumer;
import org.graalvm.compiler.code.DataSection.Data;
import jdk.vm.ci.code.site.DataSectionReference;
import jdk.vm.ci.meta.SerializableConstant;
import jdk.vm.ci.meta.VMConstant;

public final class DataSection implements Iterable<Data> {

    public interface Patches {

        void registerPatch(int position, VMConstant c);
    }

    public abstract static class Data {

        protected Data(int alignment, int size) {
        }

        protected abstract void emit(ByteBuffer buffer, Patches patches);

        public void updateAlignment(int newAlignment);

        public int getAlignment();

        public int getSize();

        @Override
        public int hashCode();

        @Override
        public String toString();

        @Override
        public boolean equals(Object obj);
    }

    public static final class RawData extends Data {

        public RawData(byte[] data, int alignment) {
        }

        @Override
        protected void emit(ByteBuffer buffer, Patches patches);
    }

    public static final class SerializableData extends Data {

        public SerializableData(SerializableConstant constant) {
        }

        public SerializableData(SerializableConstant constant, int alignment) {
        }

        @Override
        protected void emit(ByteBuffer buffer, Patches patches);
    }

    public static class ZeroData extends Data {

        protected ZeroData(int alignment, int size) {
        }

        public static ZeroData create(int alignment, int size);

        @Override
        protected void emit(ByteBuffer buffer, Patches patches);
    }

    public static final class PackedData extends Data {

        public static PackedData create(Data[] nested);

        @Override
        protected void emit(ByteBuffer buffer, Patches patches);
    }

    @Override
    public int hashCode();

    @Override
    public String toString();

    @Override
    @Pure
    @EnsuresNonNullIf(expression = "#1", result = true)
    public boolean equals(@Nullable Object obj);

    public DataSectionReference insertData(Data data);

    public void addAll(DataSection other);

    public boolean closed();

    public void close();

    public int getSectionSize();

    public int getSectionAlignment();

    public void buildDataSection(ByteBuffer buffer, Patches patch);

    public void buildDataSection(ByteBuffer buffer, Patches patch, BiConsumer<DataSectionReference, Integer> onEmit);

    public Data findData(DataSectionReference ref);

    public static void emit(ByteBuffer buffer, Data data, Patches patch);

    @Override
    public Iterator<Data> iterator();

    public void clear();
}
