/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.schema;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import org.apache.cassandra.config.ParameterizedClass;
import org.apache.cassandra.db.TypeSizes;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.IVersionedSerializer;
import org.apache.cassandra.io.compress.DeflateCompressor;
import org.apache.cassandra.io.compress.ICompressor;
import org.apache.cassandra.io.compress.LZ4Compressor;
import org.apache.cassandra.io.compress.SnappyCompressor;
import org.apache.cassandra.io.util.DataInputPlus;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.cassandraunit.shaded.com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class CompressionParams {
    private static final Logger logger = LoggerFactory.getLogger(CompressionParams.class);
    private static volatile boolean hasLoggedSsTableCompressionWarning;
    private static volatile boolean hasLoggedChunkLengthWarning;
    private static volatile boolean hasLoggedCrcCheckChanceWarning;
    public static final int DEFAULT_CHUNK_LENGTH = 65536;
    public static final IVersionedSerializer<CompressionParams> serializer;
    public static final String CLASS = "class";
    public static final String CHUNK_LENGTH_IN_KB = "chunk_length_in_kb";
    public static final String ENABLED = "enabled";
    public static final CompressionParams DEFAULT;
    private static final String CRC_CHECK_CHANCE_WARNING = "The option crc_check_chance was deprecated as a compression option. You should specify it as a top-level table option instead";
    @Deprecated
    public static final String SSTABLE_COMPRESSION = "sstable_compression";
    @Deprecated
    public static final String CHUNK_LENGTH_KB = "chunk_length_kb";
    @Deprecated
    public static final String CRC_CHECK_CHANCE = "crc_check_chance";
    private final ICompressor sstableCompressor;
    private final Integer chunkLength;
    private final ImmutableMap<String, String> otherOptions;
    private volatile double crcCheckChance = 1.0;

    public static CompressionParams fromMap(Map<String, String> opts) {
        String sstableCompressionClass;
        Map<String, String> options = CompressionParams.copyOptions(opts);
        if (!opts.isEmpty() && CompressionParams.isEnabled(opts) && !CompressionParams.containsSstableCompressionClass(opts)) {
            throw new ConfigurationException(String.format("Missing sub-option '%s' for the 'compression' option.", CLASS));
        }
        if (!CompressionParams.removeEnabled(options)) {
            sstableCompressionClass = null;
            if (!options.isEmpty()) {
                throw new ConfigurationException(String.format("If the '%s' option is set to false no other options must be specified", ENABLED));
            }
        } else {
            sstableCompressionClass = CompressionParams.removeSstableCompressionClass(options);
        }
        Integer chunkLength = CompressionParams.removeChunkLength(options);
        CompressionParams cp = new CompressionParams(sstableCompressionClass, chunkLength, options);
        cp.validate();
        return cp;
    }

    public Class<? extends ICompressor> klass() {
        return this.sstableCompressor.getClass();
    }

    public static CompressionParams noCompression() {
        return new CompressionParams((ICompressor)null, (Integer)65536, Collections.emptyMap());
    }

    public static CompressionParams snappy() {
        return CompressionParams.snappy(null);
    }

    public static CompressionParams snappy(Integer chunkLength) {
        return new CompressionParams(SnappyCompressor.instance, chunkLength, Collections.emptyMap());
    }

    public static CompressionParams deflate() {
        return CompressionParams.deflate(null);
    }

    public static CompressionParams deflate(Integer chunkLength) {
        return new CompressionParams(DeflateCompressor.instance, chunkLength, Collections.emptyMap());
    }

    public static CompressionParams lz4() {
        return CompressionParams.lz4(null);
    }

    public static CompressionParams lz4(Integer chunkLength) {
        return new CompressionParams(LZ4Compressor.create(Collections.emptyMap()), chunkLength, Collections.emptyMap());
    }

    public CompressionParams(String sstableCompressorClass, Integer chunkLength, Map<String, String> otherOptions) throws ConfigurationException {
        this(CompressionParams.createCompressor(CompressionParams.parseCompressorClass(sstableCompressorClass), otherOptions), chunkLength, otherOptions);
    }

    private CompressionParams(ICompressor sstableCompressor, Integer chunkLength, Map<String, String> otherOptions) throws ConfigurationException {
        this.sstableCompressor = sstableCompressor;
        this.chunkLength = chunkLength;
        this.otherOptions = ImmutableMap.copyOf(otherOptions);
    }

    public CompressionParams copy() {
        return new CompressionParams(this.sstableCompressor, this.chunkLength, this.otherOptions);
    }

    public boolean isEnabled() {
        return this.sstableCompressor != null;
    }

    public ICompressor getSstableCompressor() {
        return this.sstableCompressor;
    }

    public ImmutableMap<String, String> getOtherOptions() {
        return this.otherOptions;
    }

    public int chunkLength() {
        return this.chunkLength == null ? 65536 : this.chunkLength;
    }

    private static Class<?> parseCompressorClass(String className) throws ConfigurationException {
        if (className == null || className.isEmpty()) {
            return null;
        }
        className = className.contains(".") ? className : "org.apache.cassandra.io.compress." + className;
        try {
            return Class.forName(className);
        }
        catch (Exception e) {
            throw new ConfigurationException("Could not create Compression for type " + className, e);
        }
    }

    private static ICompressor createCompressor(Class<?> compressorClass, Map<String, String> compressionOptions) throws ConfigurationException {
        if (compressorClass == null) {
            if (!compressionOptions.isEmpty()) {
                throw new ConfigurationException("Unknown compression options (" + compressionOptions.keySet() + ") since no compression class found");
            }
            return null;
        }
        if (compressionOptions.containsKey(CRC_CHECK_CHANCE)) {
            if (!hasLoggedCrcCheckChanceWarning) {
                logger.warn(CRC_CHECK_CHANCE_WARNING);
                hasLoggedCrcCheckChanceWarning = true;
            }
            compressionOptions.remove(CRC_CHECK_CHANCE);
        }
        try {
            Method method = compressorClass.getMethod("create", Map.class);
            ICompressor compressor = (ICompressor)method.invoke(null, compressionOptions);
            for (String provided : compressionOptions.keySet()) {
                if (compressor.supportedOptions().contains(provided)) continue;
                throw new ConfigurationException("Unknown compression options " + provided);
            }
            return compressor;
        }
        catch (NoSuchMethodException e) {
            throw new ConfigurationException("create method not found", e);
        }
        catch (SecurityException e) {
            throw new ConfigurationException("Access forbiden", e);
        }
        catch (IllegalAccessException e) {
            throw new ConfigurationException("Cannot access method create in " + compressorClass.getName(), e);
        }
        catch (InvocationTargetException e) {
            if (e.getTargetException() instanceof ConfigurationException) {
                throw (ConfigurationException)e.getTargetException();
            }
            Throwable cause = e.getCause() == null ? e : e.getCause();
            throw new ConfigurationException(String.format("%s.create() threw an error: %s %s", compressorClass.getSimpleName(), cause.getClass().getName(), cause.getMessage()), e);
        }
        catch (ExceptionInInitializerError e) {
            throw new ConfigurationException("Cannot initialize class " + compressorClass.getName());
        }
    }

    public static ICompressor createCompressor(ParameterizedClass compression) throws ConfigurationException {
        return CompressionParams.createCompressor(CompressionParams.parseCompressorClass(compression.class_name), CompressionParams.copyOptions(compression.parameters));
    }

    private static Map<String, String> copyOptions(Map<? extends CharSequence, ? extends CharSequence> co) {
        if (co == null || co.isEmpty()) {
            return Collections.emptyMap();
        }
        HashMap<String, String> compressionOptions = new HashMap<String, String>();
        for (Map.Entry<? extends CharSequence, ? extends CharSequence> entry : co.entrySet()) {
            compressionOptions.put(entry.getKey().toString(), entry.getValue().toString());
        }
        return compressionOptions;
    }

    private static Integer parseChunkLength(String chLengthKB) throws ConfigurationException {
        if (chLengthKB == null) {
            return null;
        }
        try {
            int parsed = Integer.parseInt(chLengthKB);
            if (parsed > 0x1FFFFF) {
                throw new ConfigurationException(String.format("Value of %s is too large (%s)", CHUNK_LENGTH_IN_KB, parsed));
            }
            return 1024 * parsed;
        }
        catch (NumberFormatException e) {
            throw new ConfigurationException("Invalid value for chunk_length_in_kb", e);
        }
    }

    private static Integer removeChunkLength(Map<String, String> options) {
        if (options.containsKey(CHUNK_LENGTH_IN_KB)) {
            if (options.containsKey(CHUNK_LENGTH_KB)) {
                throw new ConfigurationException(String.format("The '%s' option must not be used if the chunk length is already specified by the '%s' option", CHUNK_LENGTH_KB, CHUNK_LENGTH_IN_KB));
            }
            return CompressionParams.parseChunkLength(options.remove(CHUNK_LENGTH_IN_KB));
        }
        if (options.containsKey(CHUNK_LENGTH_KB)) {
            if (!hasLoggedChunkLengthWarning) {
                hasLoggedChunkLengthWarning = true;
                logger.warn("The {} option has been deprecated. You should use {} instead", (Object)CHUNK_LENGTH_KB, (Object)CHUNK_LENGTH_IN_KB);
            }
            return CompressionParams.parseChunkLength(options.remove(CHUNK_LENGTH_KB));
        }
        return null;
    }

    public static boolean containsSstableCompressionClass(Map<String, String> options) {
        return options.containsKey(CLASS) || options.containsKey(SSTABLE_COMPRESSION);
    }

    private static String removeSstableCompressionClass(Map<String, String> options) {
        if (options.containsKey(CLASS)) {
            if (options.containsKey(SSTABLE_COMPRESSION)) {
                throw new ConfigurationException(String.format("The '%s' option must not be used if the compression algorithm is already specified by the '%s' option", SSTABLE_COMPRESSION, CLASS));
            }
            String clazz = options.remove(CLASS);
            if (clazz.isEmpty()) {
                throw new ConfigurationException(String.format("The '%s' option must not be empty. To disable compression use 'enabled' : false", CLASS));
            }
            return clazz;
        }
        if (options.containsKey(SSTABLE_COMPRESSION) && !hasLoggedSsTableCompressionWarning) {
            hasLoggedSsTableCompressionWarning = true;
            logger.warn("The {} option has been deprecated. You should use {} instead", (Object)SSTABLE_COMPRESSION, (Object)CLASS);
        }
        return options.remove(SSTABLE_COMPRESSION);
    }

    public static boolean isEnabled(Map<String, String> options) {
        String enabled = options.get(ENABLED);
        return enabled == null || Boolean.parseBoolean(enabled);
    }

    private static boolean removeEnabled(Map<String, String> options) {
        String enabled = options.remove(ENABLED);
        return enabled == null || Boolean.parseBoolean(enabled);
    }

    public void validate() throws ConfigurationException {
        if (this.chunkLength != null) {
            if (this.chunkLength <= 0) {
                throw new ConfigurationException("Invalid negative or null chunk_length_in_kb");
            }
            boolean found = false;
            for (int c = this.chunkLength.intValue(); c != 0; c >>= 1) {
                if ((c & 1) == 0) continue;
                if (found) {
                    throw new ConfigurationException("chunk_length_in_kb must be a power of 2");
                }
                found = true;
            }
        }
    }

    public Map<String, String> asMap() {
        if (!this.isEnabled()) {
            return Collections.singletonMap(ENABLED, "false");
        }
        HashMap<String, String> options = new HashMap<String, String>(this.otherOptions);
        options.put(CLASS, this.sstableCompressor.getClass().getName());
        options.put(CHUNK_LENGTH_IN_KB, this.chunkLengthInKB());
        return options;
    }

    public String chunkLengthInKB() {
        return String.valueOf(this.chunkLength() / 1024);
    }

    public void setCrcCheckChance(double crcCheckChance) {
        this.crcCheckChance = crcCheckChance;
    }

    public double getCrcCheckChance() {
        return this.crcCheckChance;
    }

    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (obj == null || obj.getClass() != this.getClass()) {
            return false;
        }
        CompressionParams cp = (CompressionParams)obj;
        return new EqualsBuilder().append((Object)this.sstableCompressor, (Object)cp.sstableCompressor).append(this.chunkLength(), cp.chunkLength()).append(this.otherOptions, cp.otherOptions).isEquals();
    }

    public int hashCode() {
        return new HashCodeBuilder(29, 1597).append((Object)this.sstableCompressor).append(this.chunkLength()).append(this.otherOptions).toHashCode();
    }

    static {
        serializer = new Serializer();
        DEFAULT = new CompressionParams(LZ4Compressor.create(Collections.emptyMap()), (Integer)65536, Collections.emptyMap());
    }

    static class Serializer
    implements IVersionedSerializer<CompressionParams> {
        Serializer() {
        }

        @Override
        public void serialize(CompressionParams parameters, DataOutputPlus out, int version) throws IOException {
            out.writeUTF(parameters.sstableCompressor.getClass().getSimpleName());
            out.writeInt(parameters.otherOptions.size());
            for (Map.Entry entry : parameters.otherOptions.entrySet()) {
                out.writeUTF((String)entry.getKey());
                out.writeUTF((String)entry.getValue());
            }
            out.writeInt(parameters.chunkLength());
        }

        @Override
        public CompressionParams deserialize(DataInputPlus in, int version) throws IOException {
            CompressionParams parameters;
            String compressorName = in.readUTF();
            int optionCount = in.readInt();
            HashMap<String, String> options = new HashMap<String, String>();
            for (int i = 0; i < optionCount; ++i) {
                String key = in.readUTF();
                String value = in.readUTF();
                options.put(key, value);
            }
            int chunkLength = in.readInt();
            try {
                parameters = new CompressionParams(compressorName, (Integer)chunkLength, options);
            }
            catch (ConfigurationException e) {
                throw new RuntimeException("Cannot create CompressionParams for parameters", e);
            }
            return parameters;
        }

        @Override
        public long serializedSize(CompressionParams parameters, int version) {
            long size = TypeSizes.sizeof(parameters.sstableCompressor.getClass().getSimpleName());
            size += (long)TypeSizes.sizeof(parameters.otherOptions.size());
            for (Map.Entry entry : parameters.otherOptions.entrySet()) {
                size += (long)TypeSizes.sizeof((String)entry.getKey());
                size += (long)TypeSizes.sizeof((String)entry.getValue());
            }
            return size += (long)TypeSizes.sizeof(parameters.chunkLength());
        }
    }
}

