/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.db.compaction.AbstractCompactionStrategy;
import org.apache.cassandra.db.compaction.AbstractCompactionTask;
import org.apache.cassandra.db.compaction.CompactionTask;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.db.compaction.SizeTieredCompactionStrategyOptions;
import org.apache.cassandra.db.compaction.writers.CompactionAwareWriter;
import org.apache.cassandra.db.compaction.writers.SplittingSizeTieredCompactionWriter;
import org.apache.cassandra.db.lifecycle.LifecycleTransaction;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.schema.CompactionParams;
import org.apache.cassandra.utils.Pair;
import org.cassandraunit.shaded.com.google.common.annotations.VisibleForTesting;
import org.cassandraunit.shaded.com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SizeTieredCompactionStrategy
extends AbstractCompactionStrategy {
    private static final Logger logger = LoggerFactory.getLogger(SizeTieredCompactionStrategy.class);
    private static final Comparator<Pair<List<SSTableReader>, Double>> bucketsByHotnessComparator = new Comparator<Pair<List<SSTableReader>, Double>>(){

        @Override
        public int compare(Pair<List<SSTableReader>, Double> o1, Pair<List<SSTableReader>, Double> o2) {
            int comparison = Double.compare((Double)o1.right, (Double)o2.right);
            if (comparison != 0) {
                return comparison;
            }
            return Long.compare(this.avgSize((List)o1.left), this.avgSize((List)o2.left));
        }

        private long avgSize(List<SSTableReader> sstables) {
            long n = 0L;
            for (SSTableReader sstable : sstables) {
                n += sstable.bytesOnDisk();
            }
            return n / (long)sstables.size();
        }
    };
    protected SizeTieredCompactionStrategyOptions sizeTieredOptions;
    protected volatile int estimatedRemainingTasks = 0;
    @VisibleForTesting
    protected final Set<SSTableReader> sstables = new HashSet<SSTableReader>();

    public SizeTieredCompactionStrategy(ColumnFamilyStore cfs, Map<String, String> options) {
        super(cfs, options);
        this.sizeTieredOptions = new SizeTieredCompactionStrategyOptions(options);
    }

    private synchronized List<SSTableReader> getNextBackgroundSSTables(int gcBefore) {
        int minThreshold = this.cfs.getMinimumCompactionThreshold();
        int maxThreshold = this.cfs.getMaximumCompactionThreshold();
        List<SSTableReader> candidates = SizeTieredCompactionStrategy.filterSuspectSSTables(Iterables.filter(this.cfs.getUncompactingSSTables(), this.sstables::contains));
        List<List<SSTableReader>> buckets = SizeTieredCompactionStrategy.getBuckets(SizeTieredCompactionStrategy.createSSTableAndLengthPairs(candidates), this.sizeTieredOptions.bucketHigh, this.sizeTieredOptions.bucketLow, this.sizeTieredOptions.minSSTableSize);
        logger.trace("Compaction buckets are {}", buckets);
        this.estimatedRemainingTasks = SizeTieredCompactionStrategy.getEstimatedCompactionsByTasks(this.cfs, buckets);
        this.cfs.getCompactionStrategyManager().compactionLogger.pending(this, this.estimatedRemainingTasks);
        List<SSTableReader> mostInteresting = SizeTieredCompactionStrategy.mostInterestingBucket(buckets, minThreshold, maxThreshold);
        if (!mostInteresting.isEmpty()) {
            return mostInteresting;
        }
        ArrayList<SSTableReader> sstablesWithTombstones = new ArrayList<SSTableReader>();
        for (SSTableReader sstable : candidates) {
            if (!this.worthDroppingTombstones(sstable, gcBefore)) continue;
            sstablesWithTombstones.add(sstable);
        }
        if (sstablesWithTombstones.isEmpty()) {
            return Collections.emptyList();
        }
        return Collections.singletonList(Collections.max(sstablesWithTombstones, SSTableReader.sizeComparator));
    }

    public static List<SSTableReader> mostInterestingBucket(List<List<SSTableReader>> buckets, int minThreshold, int maxThreshold) {
        ArrayList<Pair<List<SSTableReader>, Double>> prunedBucketsAndHotness = new ArrayList<Pair<List<SSTableReader>, Double>>(buckets.size());
        for (List<SSTableReader> bucket : buckets) {
            Pair<List<SSTableReader>, Double> bucketAndHotness = SizeTieredCompactionStrategy.trimToThresholdWithHotness(bucket, maxThreshold);
            if (bucketAndHotness == null || ((List)bucketAndHotness.left).size() < minThreshold) continue;
            prunedBucketsAndHotness.add(bucketAndHotness);
        }
        if (prunedBucketsAndHotness.isEmpty()) {
            return Collections.emptyList();
        }
        Pair<List<SSTableReader>, Double> hottest = Collections.max(prunedBucketsAndHotness, bucketsByHotnessComparator);
        return (List)hottest.left;
    }

    @VisibleForTesting
    static Pair<List<SSTableReader>, Double> trimToThresholdWithHotness(List<SSTableReader> bucket, int maxThreshold) {
        final Map<SSTableReader, Double> hotnessSnapshot = SizeTieredCompactionStrategy.getHotnessMap(bucket);
        Collections.sort(bucket, new Comparator<SSTableReader>(){

            @Override
            public int compare(SSTableReader o1, SSTableReader o2) {
                return -1 * Double.compare((Double)hotnessSnapshot.get(o1), (Double)hotnessSnapshot.get(o2));
            }
        });
        List<SSTableReader> prunedBucket = bucket.subList(0, Math.min(bucket.size(), maxThreshold));
        double bucketHotness = 0.0;
        for (SSTableReader sstr : prunedBucket) {
            bucketHotness += SizeTieredCompactionStrategy.hotness(sstr);
        }
        return Pair.create(prunedBucket, bucketHotness);
    }

    private static Map<SSTableReader, Double> getHotnessMap(Collection<SSTableReader> sstables) {
        HashMap<SSTableReader, Double> hotness = new HashMap<SSTableReader, Double>(sstables.size());
        for (SSTableReader sstable : sstables) {
            hotness.put(sstable, SizeTieredCompactionStrategy.hotness(sstable));
        }
        return hotness;
    }

    private static double hotness(SSTableReader sstr) {
        return sstr.getReadMeter() == null ? 0.0 : sstr.getReadMeter().twoHourRate() / (double)sstr.estimatedKeys();
    }

    @Override
    public AbstractCompactionTask getNextBackgroundTask(int gcBefore) {
        List<SSTableReader> previousCandidate = null;
        List<SSTableReader> hottestBucket;
        while (!(hottestBucket = this.getNextBackgroundSSTables(gcBefore)).isEmpty()) {
            if (hottestBucket.equals(previousCandidate)) {
                logger.warn("Could not acquire references for compacting SSTables {} which is not a problem per se,unless it happens frequently, in which case it must be reported. Will retry later.", hottestBucket);
                return null;
            }
            LifecycleTransaction transaction = this.cfs.getTracker().tryModify(hottestBucket, OperationType.COMPACTION);
            if (transaction != null) {
                return new CompactionTask(this.cfs, transaction, gcBefore);
            }
            previousCandidate = hottestBucket;
        }
        return null;
    }

    @Override
    public synchronized Collection<AbstractCompactionTask> getMaximalTask(int gcBefore, boolean splitOutput) {
        List<SSTableReader> filteredSSTables = SizeTieredCompactionStrategy.filterSuspectSSTables(this.sstables);
        if (Iterables.isEmpty(filteredSSTables)) {
            return null;
        }
        LifecycleTransaction txn = this.cfs.getTracker().tryModify(filteredSSTables, OperationType.COMPACTION);
        if (txn == null) {
            return null;
        }
        if (splitOutput) {
            return Arrays.asList(new SplittingCompactionTask(this.cfs, txn, gcBefore));
        }
        return Arrays.asList(new CompactionTask(this.cfs, txn, gcBefore));
    }

    @Override
    public AbstractCompactionTask getUserDefinedTask(Collection<SSTableReader> sstables, int gcBefore) {
        assert (!sstables.isEmpty());
        LifecycleTransaction transaction = this.cfs.getTracker().tryModify(sstables, OperationType.COMPACTION);
        if (transaction == null) {
            logger.trace("Unable to mark {} for compaction; probably a background compaction got to it first.  You can disable background compactions temporarily if this is a problem", sstables);
            return null;
        }
        return new CompactionTask(this.cfs, transaction, gcBefore).setUserDefined(true);
    }

    @Override
    public int getEstimatedRemainingTasks() {
        return this.estimatedRemainingTasks;
    }

    public static List<Pair<SSTableReader, Long>> createSSTableAndLengthPairs(Iterable<SSTableReader> sstables) {
        ArrayList<Pair<SSTableReader, Long>> sstableLengthPairs = new ArrayList<Pair<SSTableReader, Long>>(Iterables.size(sstables));
        for (SSTableReader sstable : sstables) {
            sstableLengthPairs.add(Pair.create(sstable, sstable.onDiskLength()));
        }
        return sstableLengthPairs;
    }

    public static <T> List<List<T>> getBuckets(Collection<Pair<T, Long>> files, double bucketHigh, double bucketLow, long minSSTableSize) {
        ArrayList<Pair<T, Long>> sortedFiles = new ArrayList<Pair<T, Long>>(files);
        Collections.sort(sortedFiles, new Comparator<Pair<T, Long>>(){

            @Override
            public int compare(Pair<T, Long> p1, Pair<T, Long> p2) {
                return ((Long)p1.right).compareTo((Long)p2.right);
            }
        });
        HashMap<Long, List> buckets = new HashMap<Long, List>();
        block0: for (Pair pair : sortedFiles) {
            long size = (Long)pair.right;
            for (Map.Entry entry : buckets.entrySet()) {
                List bucket = (List)entry.getValue();
                long oldAverageSize = (Long)entry.getKey();
                if (!((double)size > (double)oldAverageSize * bucketLow && (double)size < (double)oldAverageSize * bucketHigh) && (size >= minSSTableSize || oldAverageSize >= minSSTableSize)) continue;
                buckets.remove(oldAverageSize);
                long totalSize = (long)bucket.size() * oldAverageSize;
                long newAverageSize = (totalSize + size) / (long)(bucket.size() + 1);
                bucket.add(pair.left);
                buckets.put(newAverageSize, bucket);
                continue block0;
            }
            ArrayList bucket = new ArrayList();
            bucket.add(pair.left);
            buckets.put(size, bucket);
        }
        return new ArrayList<List<T>>(buckets.values());
    }

    public static int getEstimatedCompactionsByTasks(ColumnFamilyStore cfs, List<List<SSTableReader>> tasks) {
        int n = 0;
        for (List<SSTableReader> bucket : tasks) {
            if (bucket.size() < cfs.getMinimumCompactionThreshold()) continue;
            n = (int)((double)n + Math.ceil((double)bucket.size() / (double)cfs.getMaximumCompactionThreshold()));
        }
        return n;
    }

    @Override
    public long getMaxSSTableBytes() {
        return Long.MAX_VALUE;
    }

    public static Map<String, String> validateOptions(Map<String, String> options) throws ConfigurationException {
        Map<String, String> uncheckedOptions = AbstractCompactionStrategy.validateOptions(options);
        uncheckedOptions = SizeTieredCompactionStrategyOptions.validateOptions(options, uncheckedOptions);
        uncheckedOptions.remove(CompactionParams.Option.MIN_THRESHOLD.toString());
        uncheckedOptions.remove(CompactionParams.Option.MAX_THRESHOLD.toString());
        return uncheckedOptions;
    }

    @Override
    public boolean shouldDefragment() {
        return true;
    }

    @Override
    public synchronized void addSSTable(SSTableReader added) {
        this.sstables.add(added);
    }

    @Override
    public synchronized void removeSSTable(SSTableReader sstable) {
        this.sstables.remove(sstable);
    }

    public String toString() {
        return String.format("SizeTieredCompactionStrategy[%s/%s]", this.cfs.getMinimumCompactionThreshold(), this.cfs.getMaximumCompactionThreshold());
    }

    private static class SplittingCompactionTask
    extends CompactionTask {
        public SplittingCompactionTask(ColumnFamilyStore cfs, LifecycleTransaction txn, int gcBefore) {
            super(cfs, txn, gcBefore);
        }

        @Override
        public CompactionAwareWriter getCompactionAwareWriter(ColumnFamilyStore cfs, Directories directories, LifecycleTransaction txn, Set<SSTableReader> nonExpiredSSTables) {
            return new SplittingSizeTieredCompactionWriter(cfs, directories, txn, nonExpiredSSTables);
        }
    }
}

