/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.hints;

import java.io.File;
import java.io.IOException;
import java.net.InetAddress;
import java.util.Deque;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Queue;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ConcurrentLinkedQueue;
import org.apache.cassandra.gms.FailureDetector;
import org.apache.cassandra.hints.HintsDescriptor;
import org.apache.cassandra.hints.HintsWriter;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.SyncUtil;
import org.cassandraunit.shaded.com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class HintsStore {
    private static final Logger logger = LoggerFactory.getLogger(HintsStore.class);
    public final UUID hostId;
    private final File hintsDirectory;
    private final ImmutableMap<String, Object> writerParams;
    private final Map<HintsDescriptor, Long> dispatchOffsets;
    private final Deque<HintsDescriptor> dispatchDequeue;
    private final Queue<HintsDescriptor> blacklistedFiles;
    private volatile long lastUsedTimestamp;
    private volatile HintsWriter hintsWriter;

    private HintsStore(UUID hostId, File hintsDirectory, ImmutableMap<String, Object> writerParams, List<HintsDescriptor> descriptors) {
        this.hostId = hostId;
        this.hintsDirectory = hintsDirectory;
        this.writerParams = writerParams;
        this.dispatchOffsets = new ConcurrentHashMap<HintsDescriptor, Long>();
        this.dispatchDequeue = new ConcurrentLinkedDeque<HintsDescriptor>(descriptors);
        this.blacklistedFiles = new ConcurrentLinkedQueue<HintsDescriptor>();
        this.lastUsedTimestamp = descriptors.stream().mapToLong(d -> d.timestamp).max().orElse(0L);
    }

    static HintsStore create(UUID hostId, File hintsDirectory, ImmutableMap<String, Object> writerParams, List<HintsDescriptor> descriptors) {
        descriptors.sort((d1, d2) -> Long.compare(d1.timestamp, d2.timestamp));
        return new HintsStore(hostId, hintsDirectory, writerParams, descriptors);
    }

    InetAddress address() {
        return StorageService.instance.getEndpointForHostId(this.hostId);
    }

    boolean isLive() {
        InetAddress address = this.address();
        return address != null && FailureDetector.instance.isAlive(address);
    }

    HintsDescriptor poll() {
        return this.dispatchDequeue.poll();
    }

    void offerFirst(HintsDescriptor descriptor) {
        this.dispatchDequeue.offerFirst(descriptor);
    }

    void offerLast(HintsDescriptor descriptor) {
        this.dispatchDequeue.offerLast(descriptor);
    }

    void deleteAllHints() {
        HintsDescriptor descriptor;
        while ((descriptor = this.poll()) != null) {
            this.cleanUp(descriptor);
            this.delete(descriptor);
        }
        while ((descriptor = this.blacklistedFiles.poll()) != null) {
            this.cleanUp(descriptor);
            this.delete(descriptor);
        }
    }

    void delete(HintsDescriptor descriptor) {
        File hintsFile = new File(this.hintsDirectory, descriptor.fileName());
        if (hintsFile.delete()) {
            logger.info("Deleted hint file {}", (Object)descriptor.fileName());
        } else {
            logger.error("Failed to delete hint file {}", (Object)descriptor.fileName());
        }
        new File(this.hintsDirectory, descriptor.checksumFileName()).delete();
    }

    boolean hasFiles() {
        return !this.dispatchDequeue.isEmpty();
    }

    Optional<Long> getDispatchOffset(HintsDescriptor descriptor) {
        return Optional.ofNullable(this.dispatchOffsets.get(descriptor));
    }

    void markDispatchOffset(HintsDescriptor descriptor, long mark) {
        this.dispatchOffsets.put(descriptor, mark);
    }

    void cleanUp(HintsDescriptor descriptor) {
        this.dispatchOffsets.remove(descriptor);
    }

    void blacklist(HintsDescriptor descriptor) {
        this.blacklistedFiles.add(descriptor);
    }

    boolean isWriting() {
        return this.hintsWriter != null;
    }

    HintsWriter getOrOpenWriter() {
        if (this.hintsWriter == null) {
            this.hintsWriter = this.openWriter();
        }
        return this.hintsWriter;
    }

    HintsWriter getWriter() {
        return this.hintsWriter;
    }

    private HintsWriter openWriter() {
        this.lastUsedTimestamp = Math.max(System.currentTimeMillis(), this.lastUsedTimestamp + 1L);
        HintsDescriptor descriptor = new HintsDescriptor(this.hostId, this.lastUsedTimestamp, this.writerParams);
        try {
            return HintsWriter.create(this.hintsDirectory, descriptor);
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, descriptor.fileName());
        }
    }

    void closeWriter() {
        if (this.hintsWriter != null) {
            this.hintsWriter.close();
            this.offerLast(this.hintsWriter.descriptor());
            this.hintsWriter = null;
            SyncUtil.trySyncDir(this.hintsDirectory);
        }
    }

    void fsyncWriter() {
        if (this.hintsWriter != null) {
            this.hintsWriter.fsync();
        }
    }
}

