/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package org.bluestemsoftware.open.eoa.system.plugin.release.util;

import java.io.File;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.maven.model.Scm;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.apache.maven.scm.ScmException;
import org.apache.maven.scm.ScmFile;
import org.apache.maven.scm.ScmFileSet;
import org.apache.maven.scm.ScmResult;
import org.apache.maven.scm.command.checkin.CheckInScmResult;
import org.apache.maven.scm.command.status.StatusScmResult;
import org.apache.maven.scm.command.tag.TagScmResult;
import org.apache.maven.scm.command.update.UpdateScmResult;
import org.apache.maven.scm.log.DefaultLog;
import org.apache.maven.scm.manager.BasicScmManager;
import org.apache.maven.scm.manager.ScmManager;
import org.apache.maven.scm.provider.ScmProvider;
import org.apache.maven.scm.provider.svn.repository.SvnScmProviderRepository;
import org.apache.maven.scm.provider.svn.svnexe.SvnExeScmProvider;
import org.apache.maven.scm.provider.svn.svnexe.command.SvnCommandLineUtils;
import org.apache.maven.scm.repository.ScmRepository;
import org.codehaus.plexus.util.cli.CommandLineUtils;
import org.codehaus.plexus.util.cli.Commandline;
import org.codehaus.plexus.util.cli.StreamConsumer;

public class SCM {

    private static final ScmManager scmManager = new BasicScmManager();

    /**
     * Checks for uncommitted changes to project scm tree and throws an exception if any found.
     * @param project
     * @param log
     * @param ignore
     *        a list of file names to exclude from check, e.g. pom.xml.
     * @throws MojoExecutionException
     */
    public static void status(MavenProject project, Log log, Set<String> ignore) throws MojoExecutionException {

        Scm metadata = project.getModel().getScm();
        scmManager.setScmProvider("svn", new SvnExeScmProvider());
        ScmRepository repo;
        try {
            repo = scmManager.makeScmRepository(metadata.getConnection());
        } catch (Exception ex) {
            throw new MojoExecutionException("Unable to create scm connection for project "
                    + project.getArtifactId()
                    + ". "
                    + ex);
        }

        log.info("checking for modifications ...");

        try {
            StatusScmResult result = scmManager.status(repo, new ScmFileSet(project.getBasedir()));
            checkResult(project, result);
            List<?> changedFiles = result.getChangedFiles();
            StringBuilder message = new StringBuilder();
            for (Iterator<?> i = changedFiles.iterator(); i.hasNext();) {
                ScmFile file = (ScmFile)i.next();
                String name = new File(file.getPath()).getName();
                if (!ignore.contains(name)) {
                    message.append(file.toString());
                    message.append(System.getProperty("line.separator"));
                }
            }
            if (message.length() > 0) {
                throw new MojoExecutionException("Project has uncommitted changes in the following files: "
                        + System.getProperty("line.separator")
                        + message);
            }
        } catch (ScmException se) {
            throw new MojoExecutionException("Unable to check scm status for project "
                    + project.getArtifactId()
                    + ". "
                    + se);
        }

    }

    /**
     * Issues an update on project source tree.
     * @param project
     * @param log
     * @throws MojoExecutionException
     */
    public static void update(MavenProject project, Log log) throws MojoExecutionException {

        // for some reason the scm logger doesn't get instantiated unless
        // we add directly. if you get duplicated log output, remove the
        // add listener code

        Scm metadata = project.getModel().getScm();
        ScmProvider scmProvider = new SvnExeScmProvider();
        scmProvider.addListener(new DefaultLog());
        scmManager.setScmProvider("svn", scmProvider);
        ScmRepository repo;
        try {
            repo = scmManager.makeScmRepository(metadata.getConnection());
        } catch (Exception ex) {
            throw new MojoExecutionException("Unable to create scm connection for project "
                    + project.getArtifactId()
                    + ". "
                    + ex);
        }

        log.info("updating project ...");

        try {
            UpdateScmResult result = scmManager.update(repo, new ScmFileSet(project.getBasedir()));
            checkResult(project, result);
            List<?> updatedFiles = result.getUpdatedFiles();
            if (updatedFiles.size() > 0) {
                StringBuilder message = new StringBuilder();
                for (Iterator<?> i = updatedFiles.iterator(); i.hasNext();) {
                    ScmFile file = (ScmFile)i.next();
                    message.append(file.toString());
                    message.append(System.getProperty("line.separator"));
                }
                log.info("updated the following files:" + System.getProperty("line.separator") + message);
            }
        } catch (ScmException se) {
            throw new MojoExecutionException("Unable to perform scm update for project "
                    + project.getArtifactId()
                    + ". "
                    + se);
        }

    }

    /**
     * Issues a commit on project source tree.
     * @param project
     * @param log
     * @throws MojoExecutionException
     */
    public static void commit(MavenProject project, Log log, String comment) throws MojoExecutionException {

        Scm metadata = project.getModel().getScm();
        ScmProvider scmProvider = new SvnExeScmProvider();
        scmProvider.addListener(new DefaultLog());
        scmManager.setScmProvider("svn", scmProvider);
        ScmRepository repo;
        try {
            repo = scmManager.makeScmRepository(metadata.getConnection());
        } catch (Exception ex) {
            throw new MojoExecutionException("Unable to create scm connection for project "
                    + project.getArtifactId()
                    + ". "
                    + ex);
        }

        log.info("committing changes ...");

        try {
            CheckInScmResult result = scmManager.checkIn(repo, new ScmFileSet(project.getBasedir()), comment);
            checkResult(project, result);
            List<?> updatedFiles = result.getCheckedInFiles();
            if (updatedFiles.size() > 0) {
                StringBuilder message = new StringBuilder();
                for (Iterator<?> i = updatedFiles.iterator(); i.hasNext();) {
                    ScmFile file = (ScmFile)i.next();
                    message.append(file.toString());
                    message.append(System.getProperty("line.separator"));
                }
                log.info("committed the following files:" + System.getProperty("line.separator") + message);
            }
        } catch (ScmException se) {
            throw new MojoExecutionException("Unable to perform scm commit for project "
                    + project.getArtifactId()
                    + ". "
                    + se);
        }

    }

    /**
     * Issues a revert on project source tree.
     * @param project
     * @param log
     * @throws MojoExecutionException
     */
    public static void revert(MavenProject project, Log log) throws MojoExecutionException {

        Scm metadata = project.getModel().getScm();
        ScmProvider scmProvider = new SvnExeScmProvider();
        scmProvider.addListener(new DefaultLog());
        scmManager.setScmProvider("svn", scmProvider);
        ScmFileSet fileset = new ScmFileSet(project.getBasedir());
        try {
            ScmRepository repo = scmManager.makeScmRepository(metadata.getConnection());
        } catch (Exception ex) {
            throw new MojoExecutionException("Unable to create scm connection for project "
                    + project.getArtifactId()
                    + ". "
                    + ex);
        }

        log.info("reverting changes ...");

        try {
            Commandline cl = new Commandline();
            cl.setExecutable("svn");
            cl.setWorkingDirectory(fileset.getBasedir());
            cl.createArg().setValue("revert");
            cl.createArg().setValue("--recursive");
            cl.createArg().setValue(".");
            SvnCommandLineUtils.addTarget(cl, fileset.getFileList());
            StreamConsumer consumer = new StreamConsumer() {
                public void consumeLine(String line) {
                    System.out.println(line);
                }
            };
            CommandLineUtils.StringStreamConsumer stderr = new CommandLineUtils.StringStreamConsumer();
            log.info("Executing: " + SvnCommandLineUtils.cryptPassword(cl));
            log.info("Working directory: " + cl.getWorkingDirectory().getAbsolutePath());
            SvnCommandLineUtils.execute(cl, consumer, stderr, new DefaultLog());
        } catch (Exception se) {
            throw new MojoExecutionException("Unable to perform scm revert for project "
                    + project.getArtifactId()
                    + ". "
                    + se);
        }

    }

    /**
     * tags the release
     * @param project
     * @param log
     * @throws MojoExecutionException
     */
    public static void tag(MavenProject project, Log log, String tag) throws MojoExecutionException {

        Scm metadata = project.getModel().getScm();
        ScmProvider scmProvider = new SvnExeScmProvider();
        scmProvider.addListener(new DefaultLog());
        scmManager.setScmProvider("svn", scmProvider);
        ScmRepository repo;
        try {
            repo = scmManager.makeScmRepository(metadata.getConnection());
            // if we're processing the 'trunks' pom, the tags location
            // is non-standard, i.e. it cannot be inferred
            if (project.getName().equals("treleis-standalone-trunks")) {
                SvnScmProviderRepository svnRepo = (SvnScmProviderRepository)repo.getProviderRepository();
                svnRepo.setTagBase("svn://treleis.org/eoa/treleis/standalone/tags");
            }
        } catch (Exception ex) {
            throw new MojoExecutionException("Unable to create scm connection for project "
                    + project.getArtifactId()
                    + ". "
                    + ex);
        }

        log.info("tagging project ...");

        try {
            TagScmResult result = scmManager.tag(repo, new ScmFileSet(project.getBasedir()), tag);
            checkResult(project, result);
            List<?> copiedFiles = result.getTaggedFiles();
            log.info("copied " + copiedFiles.size() + " files to tag " + tag);
        } catch (ScmException se) {
            throw new MojoExecutionException("Unable to perform scm tag for project "
                    + project.getArtifactId()
                    + ". "
                    + se);
        }

    }

    /**
     * Sets/Deletes externals prop.
     * 
     * @param project
     * @param log
     * @param method
     *        one of 'delete' or set. If 'set', the file .svnexternals is used to set multiline
     *        values.
     * @throws MojoExecutionException
     */
    public static void externals(MavenProject project, Log log, String file) throws MojoExecutionException {

        Scm metadata = project.getModel().getScm();
        ScmProvider scmProvider = new SvnExeScmProvider();
        scmProvider.addListener(new DefaultLog());
        scmManager.setScmProvider("svn", scmProvider);
        ScmFileSet fileset = new ScmFileSet(project.getBasedir());
        SvnScmProviderRepository svnRepo;
        try {
            ScmRepository repo = scmManager.makeScmRepository(metadata.getConnection());
            svnRepo = (SvnScmProviderRepository)repo.getProviderRepository();
        } catch (Exception ex) {
            throw new MojoExecutionException("Unable to create scm connection for project "
                    + project.getArtifactId()
                    + ". "
                    + ex);
        }

        if (file != null) {

            log.info("setting externals prop ...");

            try {
                Commandline cl = SvnCommandLineUtils.getBaseSvnCommandLine(fileset.getBasedir(), svnRepo);
                cl.createArg().setValue("propset");
                cl.createArg().setValue("svn:externals");
                cl.createArg().setValue(".");
                cl.createArg().setValue("--file");
                cl.createArg().setValue(file);
                SvnCommandLineUtils.addTarget(cl, fileset.getFileList());
                StreamConsumer consumer = new StreamConsumer() {
                    public void consumeLine(String line) {
                        System.out.println(line);
                    }
                };
                CommandLineUtils.StringStreamConsumer stderr = new CommandLineUtils.StringStreamConsumer();
                log.info("Executing: " + SvnCommandLineUtils.cryptPassword(cl));
                log.info("Working directory: " + cl.getWorkingDirectory().getAbsolutePath());
                SvnCommandLineUtils.execute(cl, consumer, stderr, new DefaultLog());
            } catch (Exception se) {
                throw new MojoExecutionException("Unable to perform scm tag for project "
                        + project.getArtifactId()
                        + ". "
                        + se);
            }

        } else {

            log.info("deleting externals prop ...");

            try {
                Commandline cl = SvnCommandLineUtils.getBaseSvnCommandLine(fileset.getBasedir(), svnRepo);
                cl.createArg().setValue("propdel");
                cl.createArg().setValue("svn:externals");
                SvnCommandLineUtils.addTarget(cl, fileset.getFileList());
                StreamConsumer consumer = new StreamConsumer() {
                    public void consumeLine(String line) {
                        System.out.println(line);
                    }
                };
                CommandLineUtils.StringStreamConsumer stderr = new CommandLineUtils.StringStreamConsumer();
                log.info("Executing: " + SvnCommandLineUtils.cryptPassword(cl));
                log.info("Working directory: " + cl.getWorkingDirectory().getAbsolutePath());
                SvnCommandLineUtils.execute(cl, consumer, stderr, new DefaultLog());
            } catch (Exception se) {
                throw new MojoExecutionException("Unable to perform scm tag for project "
                        + project.getArtifactId()
                        + ". "
                        + se);
            }

        }

    }

    private static void checkResult(MavenProject project, ScmResult result) throws MojoExecutionException {

        if (!result.isSuccess()) {
            String message = "Provider message:";
            message = message + System.getProperty("line.separator");
            message = result.getProviderMessage() == null ? message + "" : message + result.getProviderMessage();
            message = message + System.getProperty("line.separator");
            message = message + "Command output:";
            message = message + System.getProperty("line.separator");
            message = result.getCommandOutput() == null ? message + "" : message + result.getCommandOutput();
            throw new MojoExecutionException("Failed updating project " + project.getArtifactId() + ". " + message);
        }

    }

}
