/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.system.plugin.release;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.Iterator;
import java.util.List;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.bluestemsoftware.open.eoa.system.plugin.release.util.SCM;
import org.bluestemsoftware.open.eoa.system.plugin.release.util.SCMUrlUpdater;

/**
 * Copies the local working copy (which represents a prepared release) to the indicated tag.
 * 
 * @aggregator
 * @goal tag
 */
public class ReleaseTagMojo extends AbstractMojo {
    
    private static final String LINE_BREAK = System.getProperty("line.separator");

    /**
     * The maven project.
     * 
     * @parameter expression="${project}"
     * @required
     * @readonly
     * @description "the maven project to use"
     */
    private MavenProject project;

    /**
     * Tag name.
     * 
     * @parameter expression="${tag}"
     */
    private String tag;

    /**
     * @parameter expression="${reactorProjects}"
     * @required
     * @readonly
     */
    private List<?> reactorProjects;

    /*
     * (non-Javadoc)
     * @see org.apache.maven.plugin.AbstractMojo#execute()
     */
    public void execute() throws MojoExecutionException {
        for (Iterator<?> i = reactorProjects.iterator(); i.hasNext();) {
            processProject((MavenProject)i.next());
        }
    }

    /*
     * perform a tag operation within root of each trunk and within root of
     * working copy. by convention pom within root of each trunk is named
     * *-trunk and pom within root of working copy is named *-trunks. the
     * eoa-parent pom does not follow this convention and is a special case
     */
    private void processProject(MavenProject project) throws MojoExecutionException {

        boolean issueCommand = false;
        boolean isWorkingCopyRoot = false;

        if (project.getPackaging().equals("pom")) {

            if (project.getArtifactId().equals("eoa-parent")) {
                issueCommand = true;
            }

            if (project.getArtifactId().endsWith("-trunk")) {
                issueCommand = true;
            }

            if (project.getArtifactId().endsWith("-trunks")) {
                issueCommand = true;
                isWorkingCopyRoot = true;
            }

        }
        
        // if we're processing the working copy root, we must update the
        // externals property to point to the tagged version of each
        // trunk

        if (issueCommand) {
            if (isWorkingCopyRoot) {
                SCM.externals(project, getLog(), getExternalsFile(tag));
            }
            SCMUrlUpdater.updateProject(project, tag);
            SCM.tag(project, getLog(), tag);
            SCMUrlUpdater.updateProject(project, "SNAPSHOT");
            if (isWorkingCopyRoot) {
                SCM.externals(project, getLog(), getExternalsFile("SNAPSHOT"));
            }
        }

    }

    private String getExternalsFile(String toVersion) throws MojoExecutionException {

        if (toVersion.equals("SNAPSHOT")) {
            return ".svnexternals";
        }

        File externals = null;
        BufferedReader reader = null;
        BufferedWriter writer = null;
        try {
            InputStream in = new FileInputStream(new File(project.getBasedir(), ".svnexternals"));
            reader = new BufferedReader(new InputStreamReader(in));
            externals = File.createTempFile("svnexternals-", ".tmp");
            OutputStream out = new FileOutputStream(externals);
            writer = new BufferedWriter(new OutputStreamWriter(out));
            String text = reader.readLine();
            while (text != null) {
                int index = text.indexOf("trunk");
                if (index > -1) {
                    String url = text.substring(0, index) + "tags/" + toVersion;
                    writer.write(url, 0, url.length());
                    writer.write(LINE_BREAK, 0, LINE_BREAK.length());
                }
                text = reader.readLine();
            }
        } catch (Exception ex) {
            throw new MojoExecutionException("Error parsing .svnexternals.", ex);
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException ignore) {
                }
            }
            if (writer != null) {
                try {
                    writer.flush();
                    writer.close();
                } catch (IOException ignore) {
                }
            }
        }

        return externals.getAbsolutePath();

    }

}
