/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.core;

import com.google.common.base.Preconditions;
import com.google.common.io.BaseEncoding;
import java.io.IOException;
import java.io.OutputStream;
import java.math.BigInteger;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Locale;
import java.util.Objects;
import org.bitcoinj.core.ChildMessage;
import org.bitcoinj.core.Message;
import org.bitcoinj.core.MessageSerializer;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.ProtocolException;
import org.bitcoinj.core.Utils;
import org.bitcoinj.core.VarInt;
import org.bouncycastle.jcajce.provider.digest.SHA3;

public class PeerAddress
extends ChildMessage {
    private InetAddress addr;
    private String hostname;
    private int port;
    private BigInteger services;
    private long time;
    private static final BaseEncoding BASE32 = BaseEncoding.base32().lowerCase();
    private static final byte[] ONIONCAT_PREFIX = Utils.HEX.decode((CharSequence)"fd87d87eeb43");

    public PeerAddress(NetworkParameters params, byte[] payload, int offset, Message parent, MessageSerializer serializer) throws ProtocolException {
        super(params, payload, offset, parent, serializer, Integer.MIN_VALUE);
    }

    public PeerAddress(NetworkParameters params, InetAddress addr, int port, BigInteger services, MessageSerializer serializer) {
        super(params);
        this.addr = (InetAddress)Preconditions.checkNotNull((Object)addr);
        this.port = port;
        this.setSerializer(serializer);
        this.services = services;
        this.time = Utils.currentTimeSeconds();
    }

    public PeerAddress(NetworkParameters params, InetAddress addr, int port, BigInteger services) {
        this(params, addr, port, services, params.getDefaultSerializer().withProtocolVersion(0));
    }

    public PeerAddress(NetworkParameters params, InetAddress addr, int port) {
        this(params, addr, port, BigInteger.ZERO);
    }

    public PeerAddress(NetworkParameters params, InetAddress addr) {
        this(params, addr, params.getPort());
    }

    public PeerAddress(NetworkParameters params, InetSocketAddress addr) {
        this(params, addr.getAddress(), addr.getPort());
    }

    public PeerAddress(NetworkParameters params, String hostname, int port) {
        super(params);
        this.hostname = hostname;
        this.port = port;
        this.services = BigInteger.ZERO;
        this.time = Utils.currentTimeSeconds();
    }

    public static PeerAddress localhost(NetworkParameters params) {
        return new PeerAddress(params, InetAddress.getLoopbackAddress(), params.getPort());
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    protected void bitcoinSerializeToStream(OutputStream stream) throws IOException {
        int protocolVersion = this.serializer.getProtocolVersion();
        if (protocolVersion < 0 || protocolVersion > 2) {
            throw new IllegalStateException("invalid protocolVersion: " + protocolVersion);
        }
        if (protocolVersion >= 1) {
            Utils.uint32ToByteStreamLE(this.time, stream);
        }
        if (protocolVersion == 2) {
            stream.write(new VarInt(this.services.longValue()).encode());
            if (this.addr != null) {
                if (this.addr instanceof Inet4Address) {
                    stream.write(1);
                    stream.write(new VarInt(4L).encode());
                    stream.write(this.addr.getAddress());
                } else {
                    if (!(this.addr instanceof Inet6Address)) throw new IllegalStateException();
                    stream.write(2);
                    stream.write(new VarInt(16L).encode());
                    stream.write(this.addr.getAddress());
                }
            } else {
                if (this.addr != null || this.hostname == null || !this.hostname.toLowerCase(Locale.ROOT).endsWith(".onion")) throw new IllegalStateException();
                byte[] onionAddress = BASE32.decode((CharSequence)this.hostname.substring(0, this.hostname.length() - 6));
                if (onionAddress.length == 10) {
                    stream.write(3);
                    stream.write(new VarInt(10L).encode());
                    stream.write(onionAddress);
                } else {
                    if (onionAddress.length != 35) throw new IllegalStateException();
                    stream.write(4);
                    stream.write(new VarInt(32L).encode());
                    byte[] pubkey = Arrays.copyOfRange(onionAddress, 0, 32);
                    byte[] checksum = Arrays.copyOfRange(onionAddress, 32, 34);
                    byte torVersion = onionAddress[34];
                    if (torVersion != 3) {
                        throw new IllegalStateException("version");
                    }
                    if (!Arrays.equals(checksum, this.onionChecksum(pubkey, torVersion))) {
                        throw new IllegalStateException("checksum");
                    }
                    stream.write(pubkey);
                }
            }
        } else {
            Utils.uint64ToByteStreamLE(this.services, stream);
            if (this.addr != null) {
                byte[] ipBytes = this.addr.getAddress();
                if (ipBytes.length == 4) {
                    byte[] v6addr = new byte[16];
                    System.arraycopy(ipBytes, 0, v6addr, 12, 4);
                    v6addr[10] = -1;
                    v6addr[11] = -1;
                    ipBytes = v6addr;
                }
                stream.write(ipBytes);
            } else {
                if (this.hostname == null || !this.hostname.toLowerCase(Locale.ROOT).endsWith(".onion")) throw new IllegalStateException();
                byte[] onionAddress = BASE32.decode((CharSequence)this.hostname.substring(0, this.hostname.length() - 6));
                if (onionAddress.length != 10) throw new IllegalStateException();
                stream.write(ONIONCAT_PREFIX);
                stream.write(onionAddress);
            }
        }
        Utils.uint16ToByteStreamBE(this.port, stream);
    }

    @Override
    protected void parse() throws ProtocolException {
        int protocolVersion = this.serializer.getProtocolVersion();
        if (protocolVersion < 0 || protocolVersion > 2) {
            throw new IllegalStateException("invalid protocolVersion: " + protocolVersion);
        }
        this.length = 0;
        if (protocolVersion >= 1) {
            this.time = this.readUint32();
            this.length += 4;
        } else {
            this.time = -1L;
        }
        if (protocolVersion == 2) {
            VarInt servicesVarInt = this.readVarInt();
            this.length += servicesVarInt.getSizeInBytes();
            this.services = BigInteger.valueOf(servicesVarInt.longValue());
            byte networkId = this.readByte();
            ++this.length;
            byte[] addrBytes = this.readByteArray();
            int addrLen = addrBytes.length;
            this.length += VarInt.sizeOf(addrLen) + addrLen;
            if (networkId == 1) {
                if (addrLen != 4) {
                    throw new ProtocolException("invalid length of IPv4 address: " + addrLen);
                }
                this.addr = PeerAddress.getByAddress(addrBytes);
                this.hostname = null;
            } else if (networkId == 2) {
                if (addrLen != 16) {
                    throw new ProtocolException("invalid length of IPv6 address: " + addrLen);
                }
                this.addr = PeerAddress.getByAddress(addrBytes);
                this.hostname = null;
            } else if (networkId == 3) {
                if (addrLen != 10) {
                    throw new ProtocolException("invalid length of TORv2 address: " + addrLen);
                }
                this.hostname = BASE32.encode(addrBytes) + ".onion";
                this.addr = null;
            } else if (networkId == 4) {
                if (addrLen != 32) {
                    throw new ProtocolException("invalid length of TORv3 address: " + addrLen);
                }
                byte torVersion = 3;
                byte[] onionAddress = new byte[35];
                System.arraycopy(addrBytes, 0, onionAddress, 0, 32);
                System.arraycopy(this.onionChecksum(addrBytes, torVersion), 0, onionAddress, 32, 2);
                onionAddress[34] = torVersion;
                this.hostname = BASE32.encode(onionAddress) + ".onion";
                this.addr = null;
            } else {
                this.addr = null;
                this.hostname = null;
            }
        } else {
            this.services = this.readUint64();
            this.length += 8;
            byte[] addrBytes = this.readBytes(16);
            this.length += 16;
            if (Arrays.equals(ONIONCAT_PREFIX, Arrays.copyOf(addrBytes, 6))) {
                byte[] onionAddress = Arrays.copyOfRange(addrBytes, 6, 16);
                this.hostname = BASE32.encode(onionAddress) + ".onion";
            } else {
                this.addr = PeerAddress.getByAddress(addrBytes);
                this.hostname = null;
            }
        }
        this.port = Utils.readUint16BE(this.payload, this.cursor);
        this.cursor += 2;
        this.length += 2;
    }

    private static InetAddress getByAddress(byte[] addrBytes) {
        try {
            return InetAddress.getByAddress(addrBytes);
        }
        catch (UnknownHostException e) {
            throw new RuntimeException(e);
        }
    }

    private byte[] onionChecksum(byte[] pubkey, byte version) {
        if (pubkey.length != 32) {
            throw new IllegalArgumentException();
        }
        SHA3.Digest256 digest256 = new SHA3.Digest256();
        digest256.update(".onion checksum".getBytes(StandardCharsets.US_ASCII));
        digest256.update(pubkey);
        digest256.update(version);
        return Arrays.copyOf(digest256.digest(), 2);
    }

    public String getHostname() {
        return this.hostname;
    }

    public InetAddress getAddr() {
        return this.addr;
    }

    public InetSocketAddress getSocketAddress() {
        return new InetSocketAddress(this.getAddr(), this.getPort());
    }

    public int getPort() {
        return this.port;
    }

    public BigInteger getServices() {
        return this.services;
    }

    public long getTime() {
        return this.time;
    }

    public String toString() {
        if (this.hostname != null) {
            return "[" + this.hostname + "]:" + this.port;
        }
        return "[" + this.addr.getHostAddress() + "]:" + this.port;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        PeerAddress other = (PeerAddress)o;
        return other.addr.equals(this.addr) && other.port == this.port && other.services.equals(this.services);
    }

    public int hashCode() {
        return Objects.hash(this.addr, this.port, this.services);
    }

    public InetSocketAddress toSocketAddress() {
        if (this.hostname != null) {
            return InetSocketAddress.createUnresolved(this.hostname, this.port);
        }
        return new InetSocketAddress(this.addr, this.port);
    }
}

