/*
 * Decompiled with CFR 0.152.
 */
package org.apache.wss4j.dom.message;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.Collections;
import java.util.List;
import java.util.TreeMap;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.UnsupportedCallbackException;
import org.apache.wss4j.common.bsp.BSPEnforcer;
import org.apache.wss4j.common.bsp.BSPRule;
import org.apache.wss4j.common.ext.WSPasswordCallback;
import org.apache.wss4j.common.ext.WSSecurityException;
import org.apache.wss4j.common.util.DateUtil;
import org.apache.wss4j.common.util.SOAPUtil;
import org.apache.wss4j.common.util.UsernameTokenUtil;
import org.apache.wss4j.common.util.WSTimeSource;
import org.apache.wss4j.dom.common.CustomHandler;
import org.apache.wss4j.dom.common.EncodedPasswordCallbackHandler;
import org.apache.wss4j.dom.common.UsernamePasswordCallbackHandler;
import org.apache.wss4j.dom.engine.WSSecurityEngine;
import org.apache.wss4j.dom.engine.WSSecurityEngineResult;
import org.apache.wss4j.dom.handler.HandlerAction;
import org.apache.wss4j.dom.handler.RequestData;
import org.apache.wss4j.dom.handler.WSHandlerResult;
import org.apache.wss4j.dom.message.WSSecHeader;
import org.apache.wss4j.dom.message.WSSecUsernameToken;
import org.apache.wss4j.dom.message.token.UsernameToken;
import org.apache.xml.security.utils.XMLUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class UsernameTokenTest
implements CallbackHandler {
    private static final Logger LOG = LoggerFactory.getLogger(UsernameTokenTest.class);
    private static final String SOAPUTMSG = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Header><wsse:Security SOAP-ENV:mustUnderstand=\"1\" xmlns:wsse=\"http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd\"><wsse:UsernameToken wsu:Id=\"UsernameToken-29477163\" xmlns:wsu=\"http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd\"><wsse:Username>wernerd</wsse:Username><wsse:Password>verySecret</wsse:Password></wsse:UsernameToken></wsse:Security></SOAP-ENV:Header><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body>\r\n       \r\n</SOAP-ENV:Envelope>";
    private static final String SOAPUTNOUSERMSG = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Header><wsse:Security SOAP-ENV:mustUnderstand=\"1\" xmlns:wsse=\"http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd\"><wsse:UsernameToken wsu:Id=\"UsernameToken-29477163\" xmlns:wsu=\"http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd\"><wsse:Username></wsse:Username><wsse:Password Type=\"http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordText\"></wsse:Password></wsse:UsernameToken></wsse:Security></SOAP-ENV:Header><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body>\r\n       \r\n</SOAP-ENV:Envelope>";
    private static final String EMPTY_PASSWORD_MSG = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Header><wsse:Security SOAP-ENV:mustUnderstand=\"1\" xmlns:wsse=\"http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd\"><wsse:UsernameToken wsu:Id=\"UsernameToken-1\" xmlns:wsse=\"http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd\" xmlns:wsu=\"http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd\"><wsse:Username>emptyuser</wsse:Username><wsse:Password Type=\"http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordText\"/></wsse:UsernameToken></wsse:Security></SOAP-ENV:Header><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body>\r\n       \r\n</SOAP-ENV:Envelope>";
    private CallbackHandler callbackHandler = new UsernamePasswordCallbackHandler();

    @Test
    public void testUsernameTokenDigest() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setUserInfo("wernerd", "verySecret");
        LOG.info("Before adding UsernameToken PW Digest....");
        Document signedDoc = builder.build();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Message with UserNameToken PW Digest:");
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)signedDoc);
            LOG.debug(outputString);
        }
        LOG.info("After adding UsernameToken PW Digest....");
        WSHandlerResult results = this.verify(signedDoc);
        WSSecurityEngineResult actionResult = (WSSecurityEngineResult)((List)results.getActionResults().get(1)).get(0);
        UsernameToken receivedToken = (UsernameToken)actionResult.get((Object)"username-token");
        Assertions.assertNotNull((Object)receivedToken);
        UsernameToken clone = new UsernameToken(receivedToken.getElement(), false, new BSPEnforcer());
        Assertions.assertTrue((boolean)clone.equals((Object)receivedToken));
        Assertions.assertTrue((clone.hashCode() == receivedToken.hashCode() ? 1 : 0) != 0);
    }

    @Test
    public void testUsernameTokenWithEncodedPasswordBaseline() throws Exception {
        String password = "password";
        byte[] passwordHash = MessageDigest.getInstance("SHA-1").digest(password.getBytes(StandardCharsets.UTF_8));
        String nonce = "0x7bXAPZVn40AdCD0Xbt0g==";
        String created = "2010-06-28T15:16:37Z";
        String expectedPasswordDigest = "C0rena/6gKpRZ9ATj+e6ss5sAbQ=";
        byte[] decodedNonce = XMLUtils.decode((String)nonce);
        String actualPasswordDigest = UsernameTokenUtil.doPasswordDigest((byte[])decodedNonce, (String)created, (byte[])passwordHash);
        Assertions.assertEquals((Object)expectedPasswordDigest, (Object)actualPasswordDigest, (String)"the password digest is not as expected");
    }

    @Test
    public void testUsernameTokenWithEncodedPassword() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setPasswordsAreEncoded(true);
        byte[] bytes = MessageDigest.getInstance("SHA-1").digest("verySecret".getBytes(StandardCharsets.UTF_8));
        builder.setUserInfo("wernerd", XMLUtils.encodeToString((byte[])bytes));
        LOG.info("Before adding UsernameToken PW Digest....");
        Document signedDoc = builder.build();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Message with UserNameToken PW Digest:");
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)signedDoc);
            LOG.debug(outputString);
        }
        LOG.info("After adding UsernameToken PW Digest....");
        WSSecurityEngine newEngine = new WSSecurityEngine();
        RequestData requestData = new RequestData();
        requestData.setEncodePasswords(true);
        requestData.setCallbackHandler((CallbackHandler)new EncodedPasswordCallbackHandler());
        newEngine.processSecurityHeader(signedDoc, requestData);
    }

    @Test
    public void testUsernameTokenBadUsername() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setUserInfo("badusername", "verySecret");
        LOG.info("Before adding UsernameToken PW Digest....");
        Document signedDoc = builder.build();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Message with UserNameToken PW Digest:");
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)signedDoc);
            LOG.debug(outputString);
        }
        LOG.info("After adding UsernameToken PW Digest....");
        try {
            this.verify(signedDoc);
            Assertions.fail((String)"Failure expected on a bad username");
        }
        catch (WSSecurityException ex) {
            String message = ex.getMessage();
            Assertions.assertFalse((boolean)message.contains("badusername"));
            Assertions.assertTrue((ex.getErrorCode() == WSSecurityException.ErrorCode.FAILED_AUTHENTICATION ? 1 : 0) != 0);
        }
    }

    @Test
    public void testUsernameTokenBadDigest() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setUserInfo("wernerd", "verySecre");
        LOG.info("Before adding UsernameToken PW Digest....");
        Document signedDoc = builder.build();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Message with UserNameToken PW Digest:");
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)signedDoc);
            LOG.debug(outputString);
        }
        LOG.info("After adding UsernameToken PW Digest....");
        try {
            this.verify(signedDoc);
            Assertions.fail((String)"Failure expected on a bad password digest");
        }
        catch (WSSecurityException ex) {
            Assertions.assertTrue((ex.getErrorCode() == WSSecurityException.ErrorCode.FAILED_AUTHENTICATION ? 1 : 0) != 0);
        }
    }

    @Test
    public void testOldUsernameToken() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setUserInfo("wernerd", "verySecret");
        Document signedDoc = builder.build();
        if (LOG.isDebugEnabled()) {
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)signedDoc);
            LOG.debug(outputString);
        }
        RequestData requestData = new RequestData();
        requestData.setUtTTL(-1);
        requestData.setCallbackHandler(this.callbackHandler);
        try {
            WSSecurityEngine secEngine = new WSSecurityEngine();
            secEngine.processSecurityHeader(doc, requestData);
            Assertions.fail((String)"The UsernameToken validation should have failed");
        }
        catch (WSSecurityException ex) {
            Assertions.assertTrue((ex.getErrorCode() == WSSecurityException.ErrorCode.MESSAGE_EXPIRED ? 1 : 0) != 0);
        }
    }

    @Test
    public void testNearFutureCreated() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        Element usernameTokenElement = doc.createElementNS("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd", "wsse:UsernameToken");
        Element usernameElement = doc.createElementNS("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd", "wsse:Username");
        usernameElement.appendChild(doc.createTextNode("wernerd"));
        usernameTokenElement.appendChild(usernameElement);
        Element passwordElement = doc.createElementNS("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd", "wsse:Password");
        passwordElement.setAttributeNS(null, "Type", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordText");
        passwordElement.appendChild(doc.createTextNode("verySecret"));
        usernameTokenElement.appendChild(passwordElement);
        Element elementCreated = doc.createElementNS("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd", "wsu:Created");
        ZonedDateTime now = ZonedDateTime.now(ZoneOffset.UTC).plusSeconds(30L);
        elementCreated.appendChild(doc.createTextNode(DateUtil.getDateTimeFormatter((boolean)true).format(now)));
        usernameTokenElement.appendChild(elementCreated);
        secHeader.getSecurityHeaderElement().appendChild(usernameTokenElement);
        if (LOG.isDebugEnabled()) {
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)doc);
            LOG.debug(outputString);
        }
        WSSecurityEngine secEngine = new WSSecurityEngine();
        secEngine.processSecurityHeader(doc, null, this.callbackHandler, null);
        try {
            RequestData requestData = new RequestData();
            requestData.setUtFutureTTL(0);
            requestData.setCallbackHandler(this.callbackHandler);
            secEngine.processSecurityHeader(doc, requestData);
            Assertions.fail((String)"The UsernameToken validation should have failed");
        }
        catch (WSSecurityException ex) {
            Assertions.assertTrue((ex.getErrorCode() == WSSecurityException.ErrorCode.MESSAGE_EXPIRED ? 1 : 0) != 0);
        }
    }

    @Test
    public void testFutureCreated() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        Element usernameTokenElement = doc.createElementNS("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd", "wsse:UsernameToken");
        Element usernameElement = doc.createElementNS("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd", "wsse:Username");
        usernameElement.appendChild(doc.createTextNode("wernerd"));
        usernameTokenElement.appendChild(usernameElement);
        Element passwordElement = doc.createElementNS("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd", "wsse:Password");
        passwordElement.setAttributeNS(null, "Type", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordText");
        passwordElement.appendChild(doc.createTextNode("verySecret"));
        usernameTokenElement.appendChild(passwordElement);
        Element elementCreated = doc.createElementNS("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd", "wsu:Created");
        ZonedDateTime now = ZonedDateTime.now(ZoneOffset.UTC).plusSeconds(120L);
        elementCreated.appendChild(doc.createTextNode(DateUtil.getDateTimeFormatter((boolean)true).format(now)));
        usernameTokenElement.appendChild(elementCreated);
        secHeader.getSecurityHeaderElement().appendChild(usernameTokenElement);
        if (LOG.isDebugEnabled()) {
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)doc);
            LOG.debug(outputString);
        }
        try {
            WSSecurityEngine secEngine = new WSSecurityEngine();
            secEngine.processSecurityHeader(doc, null, this.callbackHandler, null);
            Assertions.fail((String)"The UsernameToken validation should have failed");
        }
        catch (WSSecurityException ex) {
            Assertions.assertTrue((ex.getErrorCode() == WSSecurityException.ErrorCode.MESSAGE_EXPIRED ? 1 : 0) != 0);
        }
    }

    @Test
    public void testUsernameTokenText() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setPasswordType("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordText");
        builder.setUserInfo("wernerd", "verySecret");
        LOG.info("Before adding UsernameToken PW Text....");
        Document signedDoc = builder.build();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Message with UserNameToken PW Text:");
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)signedDoc);
            LOG.debug(outputString);
        }
        LOG.info("After adding UsernameToken PW Text....");
        WSHandlerResult results = this.verify(signedDoc);
        WSSecurityEngineResult actionResult = (WSSecurityEngineResult)((List)results.getActionResults().get(1)).get(0);
        UsernameToken receivedToken = (UsernameToken)actionResult.get((Object)"username-token");
        Assertions.assertNotNull((Object)receivedToken);
        UsernameToken clone = new UsernameToken(receivedToken.getElement(), false, new BSPEnforcer());
        Assertions.assertTrue((boolean)clone.equals((Object)receivedToken));
        Assertions.assertTrue((clone.hashCode() == receivedToken.hashCode() ? 1 : 0) != 0);
    }

    @Test
    public void testUsernameTokenDigestText() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setPasswordType("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordText");
        byte[] password = "verySecret".getBytes();
        MessageDigest sha = MessageDigest.getInstance("MD5");
        sha.reset();
        sha.update(password);
        String passwdDigest = XMLUtils.encodeToString((byte[])sha.digest());
        builder.setUserInfo("wernerd", passwdDigest);
        LOG.info("Before adding UsernameToken PW Text....");
        Document signedDoc = builder.build();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Message with UserNameToken PW Text:");
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)signedDoc);
            LOG.debug(outputString);
        }
    }

    @Test
    public void testUsernameTokenBadText() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setPasswordType("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordText");
        builder.setUserInfo("wernerd", "verySecre");
        LOG.info("Before adding UsernameToken PW Text....");
        Document signedDoc = builder.build();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Message with UserNameToken PW Text:");
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)signedDoc);
            LOG.debug(outputString);
        }
        LOG.info("After adding UsernameToken PW Text....");
        try {
            this.verify(signedDoc);
            Assertions.fail((String)"Failure expected on a bad password text");
        }
        catch (WSSecurityException ex) {
            Assertions.assertTrue((ex.getErrorCode() == WSSecurityException.ErrorCode.FAILED_AUTHENTICATION ? 1 : 0) != 0);
        }
    }

    @Test
    public void testUsernameTokenNoPasswordType() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)SOAPUTMSG);
        if (LOG.isDebugEnabled()) {
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)doc);
            LOG.debug(outputString);
        }
        WSSecurityEngine newEngine = new WSSecurityEngine();
        try {
            newEngine.processSecurityHeader(doc, null, this.callbackHandler, null);
            Assertions.fail((String)"Expected failure as it is not BSP compliant");
        }
        catch (WSSecurityException ex) {
            Assertions.assertTrue((ex.getErrorCode() == WSSecurityException.ErrorCode.INVALID_SECURITY ? 1 : 0) != 0);
        }
        RequestData data = new RequestData();
        data.setCallbackHandler(this.callbackHandler);
        data.setIgnoredBSPRules(Collections.singletonList(BSPRule.R4201));
        newEngine.processSecurityHeader(doc, data);
    }

    @Test
    public void testUsernameTokenNoUser() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)SOAPUTNOUSERMSG);
        if (LOG.isDebugEnabled()) {
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)doc);
            LOG.debug(outputString);
        }
        try {
            this.verify(doc);
            Assertions.fail((String)"Failure expected on no password");
        }
        catch (WSSecurityException ex) {
            Assertions.assertTrue((ex.getErrorCode() == WSSecurityException.ErrorCode.FAILED_AUTHENTICATION ? 1 : 0) != 0);
        }
    }

    @Test
    public void testUsernameTokenNoPassword() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setPasswordType(null);
        builder.setUserInfo("nopassuser", null);
        LOG.info("Before adding UsernameToken with no password....");
        Document signedDoc = builder.build();
        if (LOG.isDebugEnabled()) {
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)signedDoc);
            LOG.debug(outputString);
        }
        WSHandlerResult results = this.verify(signedDoc, true);
        WSSecurityEngineResult actionResult = (WSSecurityEngineResult)((List)results.getActionResults().get(8192)).get(0);
        UsernameToken receivedToken = (UsernameToken)actionResult.get((Object)"username-token");
        Assertions.assertNotNull((Object)receivedToken);
    }

    @Test
    public void testUsernameTokenEmptyPassword() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setPasswordType("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordText");
        builder.setUserInfo("emptyuser", "");
        LOG.info("Before adding UsernameToken with an empty password....");
        Document signedDoc = builder.build();
        if (LOG.isDebugEnabled()) {
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)signedDoc);
            LOG.debug(outputString);
        }
        WSSecurityEngine secEngine = new WSSecurityEngine();
        secEngine.processSecurityHeader(doc, null, (CallbackHandler)this, null);
    }

    @Test
    public void testEmptyPasswordProcessing() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)EMPTY_PASSWORD_MSG);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Empty password message: ");
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)doc);
            LOG.debug(outputString);
        }
        WSSecurityEngine secEngine = new WSSecurityEngine();
        secEngine.processSecurityHeader(doc, null, (CallbackHandler)this, null);
    }

    @Test
    public void testUsernameTokenCustomFail() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setPasswordType("RandomType");
        builder.setUserInfo("wernerd", "verySecret");
        Document signedDoc = builder.build();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Message with UserNameToken PW Text:");
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)signedDoc);
            LOG.debug(outputString);
        }
        try {
            WSSecurityEngine secEngine = new WSSecurityEngine();
            secEngine.processSecurityHeader(signedDoc, null, (CallbackHandler)this, null);
            Assertions.fail((String)"Custom token types are not permitted");
        }
        catch (WSSecurityException ex) {
            Assertions.assertTrue((ex.getErrorCode() == WSSecurityException.ErrorCode.FAILED_AUTHENTICATION ? 1 : 0) != 0);
        }
    }

    @Test
    public void testUsernameTokenCustomPass() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setPasswordType("RandomType");
        builder.setUserInfo("wernerd", "verySecret");
        Document signedDoc = builder.build();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Message with UserNameToken PW custom type:");
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)signedDoc);
            LOG.debug(outputString);
        }
        WSSecurityEngine secEngine = new WSSecurityEngine();
        RequestData requestData = new RequestData();
        requestData.setHandleCustomPasswordTypes(true);
        requestData.setCallbackHandler(this.callbackHandler);
        secEngine.processSecurityHeader(doc, requestData);
    }

    @Test
    public void testNullNonce() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setPasswordType("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordDigest");
        builder.setUserInfo("wernerd", "BAD_PASSWORD");
        Document utDoc = builder.build();
        Element elem = builder.getUsernameTokenElement();
        NodeList list = elem.getElementsByTagNameNS("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd", "Nonce");
        Node nonceNode = list.item(0);
        Node childNode = nonceNode.getFirstChild();
        childNode.setNodeValue("");
        if (LOG.isDebugEnabled()) {
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)utDoc);
            LOG.debug(outputString);
        }
        try {
            this.verify(utDoc);
            Assertions.fail((String)"Expected failure due to a bad password");
        }
        catch (WSSecurityException ex) {
            Assertions.assertTrue((ex.getErrorCode() == WSSecurityException.ErrorCode.FAILED_AUTHENTICATION ? 1 : 0) != 0);
        }
    }

    @Test
    public void testNullCreated() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setPasswordType("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordDigest");
        builder.setUserInfo("wernerd", "BAD_PASSWORD");
        Document utDoc = builder.build();
        Element elem = builder.getUsernameTokenElement();
        NodeList list = elem.getElementsByTagNameNS("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd", "Created");
        Node nonceNode = list.item(0);
        Node childNode = nonceNode.getFirstChild();
        childNode.setNodeValue("");
        if (LOG.isDebugEnabled()) {
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)utDoc);
            LOG.debug(outputString);
        }
        try {
            this.verify(utDoc);
            Assertions.fail((String)"Expected failure due to a bad password");
        }
        catch (WSSecurityException ex) {
            Assertions.assertTrue((ex.getErrorCode() == WSSecurityException.ErrorCode.FAILED_AUTHENTICATION ? 1 : 0) != 0);
        }
    }

    @Test
    public void testUsernameTokenNonceEncodingType() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setUserInfo("wernerd", "verySecret");
        LOG.info("Before adding UsernameToken PW Digest....");
        Document signedDoc = builder.build();
        String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)signedDoc);
        Assertions.assertTrue((boolean)outputString.contains("EncodingType"));
    }

    @Test
    public void testUsernameTokenWSHandler() throws Exception {
        CustomHandler handler = new CustomHandler();
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        RequestData reqData = new RequestData();
        TreeMap<String, String> config = new TreeMap<String, String>();
        config.put("password", "verySecret");
        config.put("passwordType", "PasswordText");
        reqData.setUsername("wernerd");
        reqData.setMsgContext(config);
        HandlerAction action = new HandlerAction(Integer.valueOf(1));
        handler.send(doc, reqData, Collections.singletonList(action), true);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Username Token via WSHandler");
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)doc);
            LOG.debug(outputString);
        }
    }

    @Test
    public void testUsernameTokenWSHandlerNoPassword() throws Exception {
        CustomHandler handler = new CustomHandler();
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        RequestData reqData = new RequestData();
        TreeMap<String, String> config = new TreeMap<String, String>();
        config.put("passwordType", "PasswordNone");
        reqData.setUsername("wernerd");
        reqData.setMsgContext(config);
        HandlerAction action = new HandlerAction(Integer.valueOf(1));
        handler.send(doc, reqData, Collections.singletonList(action), true);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Username Token via WSHandler");
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)doc);
            LOG.debug(outputString);
        }
    }

    @Test
    public void testUsernameTokenWSHandlerNoPassword2() throws Exception {
        CustomHandler handler = new CustomHandler();
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        RequestData reqData = new RequestData();
        TreeMap config = new TreeMap();
        reqData.setUsername("wernerd");
        reqData.setMsgContext(config);
        HandlerAction action = new HandlerAction(Integer.valueOf(8192));
        handler.send(doc, reqData, Collections.singletonList(action), true);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Username Token via WSHandler");
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)doc);
            LOG.debug(outputString);
        }
    }

    @Test
    public void testUsernameTokenWSHandlerEmptyPassword() throws Exception {
        CustomHandler handler = new CustomHandler();
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        RequestData reqData = new RequestData();
        TreeMap<String, Object> config = new TreeMap<String, Object>();
        config.put("passwordType", "PasswordText");
        config.put("passwordCallbackRef", this);
        reqData.setUsername("emptyuser");
        reqData.setMsgContext(config);
        HandlerAction action = new HandlerAction(Integer.valueOf(1));
        handler.send(doc, reqData, Collections.singletonList(action), true);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Username Token with an empty password via WSHandler");
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)doc);
            LOG.debug(outputString);
        }
    }

    @Test
    public void testMultipleNonce() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setPasswordType("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordDigest");
        builder.setUserInfo("wernerd", "verySecret");
        Document utDoc = builder.build();
        Element elem = builder.getUsernameTokenElement();
        NodeList list = elem.getElementsByTagNameNS("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd", "Nonce");
        Node nonceNode = list.item(0);
        Node nonceCopy = nonceNode.cloneNode(true);
        nonceNode.getParentNode().insertBefore(nonceCopy, nonceNode);
        if (LOG.isDebugEnabled()) {
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)utDoc);
            LOG.debug(outputString);
        }
        WSSecurityEngine newEngine = new WSSecurityEngine();
        try {
            newEngine.processSecurityHeader(doc, null, this.callbackHandler, null);
            Assertions.fail((String)"Expected failure as it is not BSP compliant");
        }
        catch (WSSecurityException ex) {
            Assertions.assertTrue((ex.getErrorCode() == WSSecurityException.ErrorCode.INVALID_SECURITY ? 1 : 0) != 0);
        }
        RequestData data = new RequestData();
        data.setCallbackHandler(this.callbackHandler);
        data.setIgnoredBSPRules(Collections.singletonList(BSPRule.R4225));
        newEngine.processSecurityHeader(doc, data);
    }

    @Test
    public void testMultipleCreated() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setPasswordType("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordDigest");
        builder.setUserInfo("wernerd", "verySecret");
        Document utDoc = builder.build();
        Element elem = builder.getUsernameTokenElement();
        NodeList list = elem.getElementsByTagNameNS("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd", "Created");
        Node createdNode = list.item(0);
        Node createdCopy = createdNode.cloneNode(true);
        createdNode.getParentNode().insertBefore(createdCopy, createdNode);
        if (LOG.isDebugEnabled()) {
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)utDoc);
            LOG.debug(outputString);
        }
        WSSecurityEngine newEngine = new WSSecurityEngine();
        try {
            newEngine.processSecurityHeader(doc, null, this.callbackHandler, null);
            Assertions.fail((String)"Expected failure as it is not BSP compliant");
        }
        catch (WSSecurityException ex) {
            Assertions.assertTrue((ex.getErrorCode() == WSSecurityException.ErrorCode.INVALID_SECURITY ? 1 : 0) != 0);
        }
        RequestData data = new RequestData();
        data.setCallbackHandler(this.callbackHandler);
        data.setIgnoredBSPRules(Collections.singletonList(BSPRule.R4223));
        newEngine.processSecurityHeader(doc, data);
    }

    @Test
    public void testMultiplePassword() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setPasswordType("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordDigest");
        builder.setUserInfo("wernerd", "verySecret");
        Document utDoc = builder.build();
        Element elem = builder.getUsernameTokenElement();
        NodeList list = elem.getElementsByTagNameNS("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd", "Password");
        Node passwordNode = list.item(0);
        Node passwordCopy = passwordNode.cloneNode(true);
        passwordNode.getParentNode().insertBefore(passwordCopy, passwordNode);
        if (LOG.isDebugEnabled()) {
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)utDoc);
            LOG.debug(outputString);
        }
        WSSecurityEngine newEngine = new WSSecurityEngine();
        try {
            newEngine.processSecurityHeader(doc, null, this.callbackHandler, null);
            Assertions.fail((String)"Expected failure as it is not BSP compliant");
        }
        catch (WSSecurityException ex) {
            Assertions.assertTrue((ex.getErrorCode() == WSSecurityException.ErrorCode.INVALID_SECURITY ? 1 : 0) != 0);
        }
        RequestData data = new RequestData();
        data.setCallbackHandler(this.callbackHandler);
        data.setIgnoredBSPRules(Collections.singletonList(BSPRule.R4222));
        newEngine.processSecurityHeader(doc, data);
    }

    @Test
    public void testNonceBadEncodingType() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setPasswordType("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordDigest");
        builder.setUserInfo("wernerd", "verySecret");
        Document utDoc = builder.build();
        Element elem = builder.getUsernameTokenElement();
        NodeList list = elem.getElementsByTagNameNS("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd", "Nonce");
        Node nonceNode = list.item(0);
        ((Element)nonceNode).setAttributeNS(null, "EncodingType", "http://bad_encoding_type");
        if (LOG.isDebugEnabled()) {
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)utDoc);
            LOG.debug(outputString);
        }
        WSSecurityEngine newEngine = new WSSecurityEngine();
        try {
            newEngine.processSecurityHeader(doc, null, this.callbackHandler, null);
            Assertions.fail((String)"Expected failure as it is not BSP compliant");
        }
        catch (WSSecurityException ex) {
            Assertions.assertTrue((ex.getErrorCode() == WSSecurityException.ErrorCode.INVALID_SECURITY ? 1 : 0) != 0);
        }
        RequestData data = new RequestData();
        data.setCallbackHandler(this.callbackHandler);
        data.setIgnoredBSPRules(Collections.singletonList(BSPRule.R4221));
        newEngine.processSecurityHeader(doc, data);
    }

    @Test
    public void testUsernameTokenWSHandlerNonceCreated() throws Exception {
        CustomHandler handler = new CustomHandler();
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        RequestData reqData = new RequestData();
        TreeMap<String, String> config = new TreeMap<String, String>();
        config.put("password", "verySecret");
        config.put("passwordType", "PasswordText");
        config.put("addUsernameTokenNonce", "true");
        config.put("addUsernameTokenCreated", "true");
        reqData.setUsername("wernerd");
        reqData.setMsgContext(config);
        HandlerAction action = new HandlerAction(Integer.valueOf(1));
        handler.send(doc, reqData, Collections.singletonList(action), true);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Username Token via WSHandler");
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)doc);
            LOG.debug(outputString);
        }
    }

    @Test
    public void testSpoofedUsernameToken() throws Exception {
        Document doc = SOAPUtil.toSOAPPart((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><SOAP-ENV:Body><add xmlns=\"http://ws.apache.org/counter/counter_port_type\"><value xmlns=\"\">15</value></add></SOAP-ENV:Body></SOAP-ENV:Envelope>");
        WSSecHeader secHeader = new WSSecHeader(doc);
        secHeader.insertSecurityHeader();
        WSSecUsernameToken builder = new WSSecUsernameToken(secHeader);
        builder.setUserInfo("wernerd", "verySecret");
        WSTimeSource spoofedTimeSource = new WSTimeSource(){

            public Instant now() {
                return Instant.now().minusSeconds(500L);
            }
        };
        builder.setWsTimeSource(spoofedTimeSource);
        Document signedDoc = builder.build();
        if (LOG.isDebugEnabled()) {
            String outputString = org.apache.wss4j.common.util.XMLUtils.prettyDocumentToString((Document)signedDoc);
            LOG.debug(outputString);
        }
        try {
            WSSecurityEngine secEngine = new WSSecurityEngine();
            secEngine.processSecurityHeader(doc, null, this.callbackHandler, null);
            Assertions.fail((String)"The UsernameToken validation should have failed");
        }
        catch (WSSecurityException ex) {
            Assertions.assertTrue((ex.getErrorCode() == WSSecurityException.ErrorCode.MESSAGE_EXPIRED ? 1 : 0) != 0);
        }
    }

    private WSHandlerResult verify(Document doc) throws Exception {
        return this.verify(doc, false);
    }

    private WSHandlerResult verify(Document doc, boolean allowUsernameTokenDerivedKeys) throws Exception {
        WSSecurityEngine secEngine = new WSSecurityEngine();
        RequestData requestData = new RequestData();
        requestData.setAllowUsernameTokenNoPassword(allowUsernameTokenDerivedKeys);
        requestData.setCallbackHandler(this.callbackHandler);
        return secEngine.processSecurityHeader(doc, requestData);
    }

    @Override
    public void handle(Callback[] callbacks) throws IOException, UnsupportedCallbackException {
        for (Callback callback : callbacks) {
            if (callback instanceof WSPasswordCallback) {
                WSPasswordCallback pc = (WSPasswordCallback)callback;
                if (pc.getUsage() != 2) continue;
                if ("emptyuser".equals(pc.getIdentifier())) {
                    pc.setPassword("");
                    continue;
                }
                if ("customUser".equals(pc.getIdentifier())) {
                    return;
                }
                if (null != pc.getIdentifier()) continue;
                return;
            }
            throw new UnsupportedCallbackException(callback, "Unrecognized Callback");
        }
    }
}

