/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package javax.enterprise.util;

import java.io.Serializable;
import java.lang.reflect.GenericArrayType;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;

/**
 * Type literal implementation.
 * @version $Rev: 1799996 $ $Date: 2017-06-26 17:20:09 -0400 (Mon, 26 Jun 2017) $
 *
 * @param <T> wrapped type
 */
@SuppressWarnings("unchecked")
public abstract class TypeLiteral<T> implements Serializable
{
    private static final long serialVersionUID = 6993258591899719600L;
    
    private Type definedType;

    protected TypeLiteral()
    {
        this.definedType = getDefinedType(this.getClass());
    }

    public final Type getType()
    {
        return definedType;
    }

    public final Class<T> getRawType()
    {
        Class<T> rawType;

        if (this.definedType instanceof Class)
        {
            rawType = (Class<T>) this.definedType;
        }
        else if (this.definedType instanceof ParameterizedType)
        {
            ParameterizedType pt = (ParameterizedType) this.definedType;
            rawType = (Class<T>) pt.getRawType();

        }
        else if (this.definedType instanceof GenericArrayType)
        {
            rawType = (Class<T>) Object[].class;
        }
        else
        {
            throw new RuntimeException("Illegal type for the Type Literal Class");
        }

        return rawType;
    }

    private Type getDefinedType(Class<?> clazz)
    {
        Type type;

        if (clazz == null)
        {
            throw new RuntimeException("Class parameter can not be null!");
        }

        Type superClazz = clazz.getGenericSuperclass();

        if (superClazz.equals(Object.class))
        {
            throw new RuntimeException("Super class must be parametrized type!");
        }
        else if (superClazz instanceof ParameterizedType)
        {
            ParameterizedType pt = (ParameterizedType) superClazz;            
            Type[] actualArgs = pt.getActualTypeArguments();

            if (actualArgs.length == 1)
            {
                type = actualArgs[0];
            }
            else
            {
                throw new RuntimeException("More than one parametric type!");
            }
        }
        else
        {
            type = getDefinedType((Class<?>) superClazz);
        }

        return type;
    }

    @Override
    public int hashCode()
    {
        int prime = 31;
        int result = 1;
        
        result = prime * result + ((definedType == null) ? 0 : definedType.hashCode());
        
        return result;
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;   
        }
        
        if (obj == null)
        {
            return false;   
        }
        
        if (getClass() != obj.getClass())
        {
            return false;   
        }
        
        TypeLiteral other = (TypeLiteral) obj;
        if (definedType == null)
        {
            if (other.definedType != null)
            {
                return false;   
            }
        }
        else if (!definedType.equals(other.definedType))
        {
            return false;   
        }
        
        return true;
    }

}