/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tinkerpop.gremlin.driver;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import org.apache.tinkerpop.gremlin.driver.Cluster;
import org.apache.tinkerpop.gremlin.driver.Connection;
import org.apache.tinkerpop.gremlin.driver.ConnectionPool;
import org.apache.tinkerpop.gremlin.driver.Host;
import org.apache.tinkerpop.gremlin.driver.ResultSet;
import org.apache.tinkerpop.gremlin.driver.exception.ConnectionException;
import org.apache.tinkerpop.gremlin.driver.message.RequestMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class Client {
    private static final Logger logger = LoggerFactory.getLogger(Client.class);
    protected final Cluster cluster;
    protected volatile boolean initialized;

    Client(Cluster cluster) {
        this.cluster = cluster;
    }

    public RequestMessage buildMessage(RequestMessage.Builder builder) {
        return builder.create();
    }

    protected abstract void initializeImplementation();

    protected abstract Connection chooseConnection(RequestMessage var1) throws TimeoutException, ConnectionException;

    public abstract CompletableFuture<Void> closeAsync();

    public abstract Client rebind(String var1);

    public synchronized Client init() {
        if (this.initialized) {
            return this;
        }
        logger.debug("Initializing client on cluster [{}]", (Object)this.cluster);
        this.cluster.init();
        this.initializeImplementation();
        this.initialized = true;
        return this;
    }

    public ResultSet submit(String gremlin) {
        return this.submit(gremlin, null);
    }

    public ResultSet submit(String gremlin, Map<String, Object> parameters) {
        try {
            return this.submitAsync(gremlin, parameters).get();
        }
        catch (Exception ex) {
            throw new RuntimeException(ex);
        }
    }

    public CompletableFuture<ResultSet> submitAsync(String gremlin) {
        return this.submitAsync(gremlin, null);
    }

    public CompletableFuture<ResultSet> submitAsync(String gremlin, Map<String, Object> parameters) {
        RequestMessage.Builder request = RequestMessage.build("eval").add("gremlin", gremlin).add("batchSize", this.cluster.connectionPoolSettings().resultIterationBatchSize);
        Optional.ofNullable(parameters).ifPresent(params -> request.addArg("bindings", parameters));
        return this.submitAsync(this.buildMessage(request));
    }

    /*
     * Loose catch block
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public CompletableFuture<ResultSet> submitAsync(RequestMessage msg) {
        CompletableFuture<ResultSet> completableFuture;
        if (!this.initialized) {
            this.init();
        }
        CompletableFuture<ResultSet> future = new CompletableFuture<ResultSet>();
        Connection connection = null;
        try {
            connection = this.chooseConnection(msg);
            connection.write(msg, future);
            completableFuture = future;
        }
        catch (TimeoutException toe) {
            try {
                throw new RuntimeException(toe);
                catch (ConnectionException ce) {
                    throw new RuntimeException(ce);
                }
                catch (Exception ex) {
                    throw new RuntimeException(ex);
                }
            }
            catch (Throwable throwable) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Submitted {} to - {}", (Object)msg, (Object)(null == connection ? "connection not initialized" : connection.toString()));
                }
                throw throwable;
            }
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Submitted {} to - {}", (Object)msg, (Object)(null == connection ? "connection not initialized" : connection.toString()));
        }
        return completableFuture;
    }

    public void close() {
        this.closeAsync().join();
    }

    public static final class SessionedClient
    extends Client {
        private final String sessionId;
        private ConnectionPool connectionPool;

        SessionedClient(Cluster cluster, String sessionId) {
            super(cluster);
            this.sessionId = sessionId;
        }

        String getSessionId() {
            return this.sessionId;
        }

        @Override
        public Client rebind(String graphOrTraversalSourceName) {
            throw new UnsupportedOperationException("Sessioned client do no support rebinding");
        }

        @Override
        public RequestMessage buildMessage(RequestMessage.Builder builder) {
            builder.processor("session");
            builder.addArg("session", this.sessionId);
            return builder.create();
        }

        @Override
        protected Connection chooseConnection(RequestMessage msg) throws TimeoutException, ConnectionException {
            return this.connectionPool.borrowConnection(this.cluster.connectionPoolSettings().maxWaitForConnection, TimeUnit.MILLISECONDS);
        }

        @Override
        protected void initializeImplementation() {
            List hosts = this.cluster.allHosts().stream().filter(Host::isAvailable).collect(Collectors.toList());
            Collections.shuffle(hosts);
            Host host = (Host)hosts.get(0);
            this.connectionPool = new ConnectionPool(host, this, Optional.of(1), Optional.of(1));
        }

        @Override
        public CompletableFuture<Void> closeAsync() {
            return this.connectionPool.closeAsync();
        }
    }

    public static final class ReboundClusteredClient
    extends Client {
        private final ClusteredClient clusteredClient;
        private final Map<String, String> rebindings = new HashMap<String, String>();
        final CompletableFuture<Void> close = new CompletableFuture();

        ReboundClusteredClient(ClusteredClient clusteredClient, String graphOrTraversalSource) {
            super(clusteredClient.cluster);
            this.clusteredClient = clusteredClient;
            this.rebindings.put("g", graphOrTraversalSource);
        }

        ReboundClusteredClient(ClusteredClient clusteredClient, Map<String, String> rebindings) {
            super(clusteredClient.cluster);
            this.clusteredClient = clusteredClient;
            this.rebindings.putAll(rebindings);
        }

        @Override
        public synchronized Client init() {
            if (this.close.isDone()) {
                throw new IllegalStateException("Client is closed");
            }
            this.clusteredClient.init();
            return this;
        }

        @Override
        public RequestMessage buildMessage(RequestMessage.Builder builder) {
            if (this.close.isDone()) {
                throw new IllegalStateException("Client is closed");
            }
            if (!this.rebindings.isEmpty()) {
                builder.addArg("rebindings", this.rebindings);
            }
            return builder.create();
        }

        @Override
        protected void initializeImplementation() {
            if (this.close.isDone()) {
                throw new IllegalStateException("Client is closed");
            }
        }

        @Override
        protected Connection chooseConnection(RequestMessage msg) throws TimeoutException, ConnectionException {
            if (this.close.isDone()) {
                throw new IllegalStateException("Client is closed");
            }
            return this.clusteredClient.chooseConnection(msg);
        }

        @Override
        public CompletableFuture<Void> closeAsync() {
            this.close.complete(null);
            return this.close;
        }

        @Override
        public Client rebind(String graphOrTraversalSource) {
            if (this.close.isDone()) {
                throw new IllegalStateException("Client is closed");
            }
            return new ReboundClusteredClient(this.clusteredClient, graphOrTraversalSource);
        }
    }

    public static final class ClusteredClient
    extends Client {
        private ConcurrentMap<Host, ConnectionPool> hostConnectionPools = new ConcurrentHashMap<Host, ConnectionPool>();

        ClusteredClient(Cluster cluster) {
            super(cluster);
        }

        public ResultSet submit(String gremlin, String graphOrTraversalSource) {
            return this.submit(gremlin, graphOrTraversalSource, null);
        }

        public ResultSet submit(String gremlin, String graphOrTraversalSource, Map<String, Object> parameters) {
            try {
                return this.submitAsync(gremlin, graphOrTraversalSource, parameters).get();
            }
            catch (Exception ex) {
                throw new RuntimeException(ex);
            }
        }

        public CompletableFuture<ResultSet> submitAsync(String gremlin, String graphOrTraversalSource, Map<String, Object> parameters) {
            RequestMessage.Builder request = RequestMessage.build("eval").add("gremlin", gremlin).add("batchSize", this.cluster.connectionPoolSettings().resultIterationBatchSize);
            Optional.ofNullable(parameters).ifPresent(params -> request.addArg("bindings", parameters));
            if (graphOrTraversalSource != null && !graphOrTraversalSource.isEmpty()) {
                request.addArg("rebindings", this.makeRebindings(graphOrTraversalSource));
            }
            return this.submitAsync(this.buildMessage(request));
        }

        @Override
        public Client rebind(String graphOrTraversalSource) {
            return new ReboundClusteredClient(this, graphOrTraversalSource);
        }

        public Client rebind(Map<String, String> rebindings) {
            return new ReboundClusteredClient(this, rebindings);
        }

        @Override
        protected Connection chooseConnection(RequestMessage msg) throws TimeoutException, ConnectionException {
            Iterator<Host> possibleHosts = this.cluster.loadBalancingStrategy().select(msg);
            if (!possibleHosts.hasNext()) {
                throw new TimeoutException("Timed out waiting for an available host.");
            }
            Host bestHost = this.cluster.loadBalancingStrategy().select(msg).next();
            ConnectionPool pool = (ConnectionPool)this.hostConnectionPools.get(bestHost);
            return pool.borrowConnection(this.cluster.connectionPoolSettings().maxWaitForConnection, TimeUnit.MILLISECONDS);
        }

        @Override
        protected void initializeImplementation() {
            this.cluster.allHosts().forEach(host -> {
                try {
                    this.hostConnectionPools.put((Host)host, new ConnectionPool((Host)host, this));
                    this.cluster.loadBalancingStrategy().onNew((Host)host);
                }
                catch (Exception ex) {
                    logger.warn("Could not initialize connection pool for {} - will try later", host);
                }
            });
        }

        @Override
        public CompletableFuture<Void> closeAsync() {
            CompletableFuture[] poolCloseFutures = new CompletableFuture[this.hostConnectionPools.size()];
            this.hostConnectionPools.values().stream().map(ConnectionPool::closeAsync).collect(Collectors.toList()).toArray(poolCloseFutures);
            return CompletableFuture.allOf(poolCloseFutures);
        }

        private Map<String, String> makeRebindings(String graphOrTraversalSource) {
            HashMap<String, String> rebindings = new HashMap<String, String>();
            rebindings.put("g", graphOrTraversalSource);
            return rebindings;
        }
    }
}

