/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tinkerpop.gremlin.driver;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.tinkerpop.gremlin.driver.Cluster;
import org.apache.tinkerpop.gremlin.driver.Connection;
import org.apache.tinkerpop.gremlin.driver.Host;
import org.apache.tinkerpop.gremlin.driver.Settings;
import org.apache.tinkerpop.gremlin.driver.exception.ConnectionException;
import org.apache.tinkerpop.gremlin.util.TimeUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ConnectionPool {
    private static final Logger logger = LoggerFactory.getLogger(ConnectionPool.class);
    public static final int MIN_POOL_SIZE = 2;
    public static final int MAX_POOL_SIZE = 8;
    public static final int MIN_SIMULTANEOUS_REQUESTS_PER_CONNECTION = 8;
    public static final int MAX_SIMULTANEOUS_REQUESTS_PER_CONNECTION = 16;
    public final Host host;
    private final Cluster cluster;
    private final List<Connection> connections;
    private final AtomicInteger open;
    private final Set<Connection> bin = new CopyOnWriteArraySet<Connection>();
    private final int minPoolSize;
    private final int maxPoolSize;
    private final int minSimultaneousRequestsPerConnection;
    private final int maxSimultaneousRequestsPerConnection;
    private final int minInProcess;
    private final AtomicInteger scheduledForCreation = new AtomicInteger();
    private final AtomicReference<CompletableFuture<Void>> closeFuture = new AtomicReference();
    private volatile int waiter = 0;
    private final Lock waitLock = new ReentrantLock(true);
    private final Condition hasAvailableConnection = this.waitLock.newCondition();

    public ConnectionPool(Host host, Cluster cluster) {
        this.host = host;
        this.cluster = cluster;
        Settings.ConnectionPoolSettings settings = this.settings();
        this.minPoolSize = settings.minSize;
        this.maxPoolSize = settings.maxSize;
        this.minSimultaneousRequestsPerConnection = settings.minSimultaneousRequestsPerConnection;
        this.maxSimultaneousRequestsPerConnection = settings.maxSimultaneousRequestsPerConnection;
        this.minInProcess = settings.minInProcessPerConnection;
        ArrayList<Connection> l = new ArrayList<Connection>(this.minPoolSize);
        try {
            for (int i = 0; i < this.minPoolSize; ++i) {
                l.add(new Connection(host.getHostUri(), this, cluster, settings.maxInProcessPerConnection));
            }
        }
        catch (ConnectionException ce) {
            logger.debug("Could not initialize connections in pool for {} - pool size at {}", (Object)host, (Object)l.size());
            this.considerUnavailable();
        }
        this.connections = new CopyOnWriteArrayList<Connection>(l);
        this.open = new AtomicInteger(this.connections.size());
        logger.info("Opening connection pool on {} with core size of {}", (Object)host, (Object)this.minPoolSize);
    }

    public Settings.ConnectionPoolSettings settings() {
        return this.cluster.connectionPoolSettings();
    }

    public Connection borrowConnection(long timeout, TimeUnit unit) throws TimeoutException, ConnectionException {
        int inFlight;
        logger.debug("Borrowing connection from pool on {} - timeout in {} {}", new Object[]{this.host, timeout, unit});
        if (this.isClosed()) {
            throw new ConnectionException(this.host.getHostUri(), this.host.getAddress(), "Pool is shutdown");
        }
        Connection leastUsedConn = this.selectLeastUsed();
        if (this.connections.isEmpty()) {
            logger.debug("Tried to borrow connection but the pool was empty for {} - scheduling pool creation and waiting for connection", (Object)this.host);
            for (int i = 0; i < this.minPoolSize; ++i) {
                this.scheduledForCreation.incrementAndGet();
                this.newConnection();
            }
            return this.waitForConnection(timeout, unit);
        }
        if (null == leastUsedConn) {
            if (this.isClosed()) {
                throw new ConnectionException(this.host.getHostUri(), this.host.getAddress(), "Pool is shutdown");
            }
            logger.debug("Pool was initialized but a connection could not be selected earlier - waiting for connection on {}", (Object)this.host);
            return this.waitForConnection(timeout, unit);
        }
        int currentPoolSize = this.connections.size();
        if (leastUsedConn.inFlight.get() >= this.maxSimultaneousRequestsPerConnection && currentPoolSize < this.maxPoolSize) {
            logger.debug("Least used {} on {} exceeds maxSimultaneousRequestsPerConnection but pool size {} < maxPoolSize - consider new connection", new Object[]{leastUsedConn, this.host, currentPoolSize});
            this.considerNewConnection();
        }
        do {
            inFlight = leastUsedConn.inFlight.get();
            int availableInProcess = leastUsedConn.availableInProcess();
            if (inFlight < leastUsedConn.availableInProcess()) continue;
            logger.debug("Least used connection selected from pool for {} but inFlight [{}] >= availableInProcess [{}] - wait", new Object[]{this.host, inFlight, availableInProcess});
            return this.waitForConnection(timeout, unit);
        } while (!leastUsedConn.inFlight.compareAndSet(inFlight, inFlight + 1));
        logger.debug("Return least used {} on {}", (Object)leastUsedConn, (Object)this.host);
        return leastUsedConn;
    }

    public void returnConnection(Connection connection) throws ConnectionException {
        logger.debug("Attempting to return {} on {}", (Object)connection, (Object)this.host);
        if (this.isClosed()) {
            throw new ConnectionException(this.host.getHostUri(), this.host.getAddress(), "Pool is shutdown");
        }
        int inFlight = connection.inFlight.decrementAndGet();
        if (connection.isDead()) {
            logger.debug("Marking {} as dead", (Object)this.host);
            this.considerUnavailable();
        } else {
            if (this.bin.contains(connection) && inFlight == 0) {
                logger.debug("{} is already in the bin and it has no inflight requests so it is safe to close", (Object)connection);
                if (this.bin.remove(connection)) {
                    connection.closeAsync();
                }
                return;
            }
            int poolSize = this.connections.size();
            int availableInProcess = connection.availableInProcess();
            if (poolSize > this.minPoolSize && inFlight <= this.minSimultaneousRequestsPerConnection) {
                logger.debug("On {} pool size of {} > minPoolSize {} and inFlight of {} <= minSimultaneousRequestsPerConnection {} so destroy {}", new Object[]{this.host, poolSize, this.minPoolSize, inFlight, this.minSimultaneousRequestsPerConnection, connection});
                this.destroyConnection(connection);
            } else if (connection.availableInProcess() < this.minInProcess) {
                logger.debug("On {} availableInProcess {} < minInProcess {} so replace {}", new Object[]{this.host, availableInProcess, this.minInProcess, connection});
                this.replaceConnection(connection);
            } else {
                this.announceAvailableConnection();
            }
        }
    }

    public boolean isClosed() {
        return this.closeFuture.get() != null;
    }

    public CompletableFuture<Void> closeAsync() {
        logger.info("Signalled closing of connection pool on {} with core size of {}", (Object)this.host, (Object)this.minPoolSize);
        CompletableFuture<Void> future = this.closeFuture.get();
        if (future != null) {
            return future;
        }
        this.announceAllAvailableConnection();
        future = CompletableFuture.allOf(this.killAvailableConnections());
        return this.closeFuture.compareAndSet(null, future) ? future : this.closeFuture.get();
    }

    public int opened() {
        return this.open.get();
    }

    private CompletableFuture[] killAvailableConnections() {
        ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>(this.connections.size());
        for (Connection connection : this.connections) {
            CompletableFuture<Void> future = connection.closeAsync();
            future.thenRunAsync(this.open::decrementAndGet);
            futures.add(future);
        }
        return futures.toArray(new CompletableFuture[futures.size()]);
    }

    private void replaceConnection(Connection connection) {
        logger.debug("Replace {}", (Object)connection);
        this.open.decrementAndGet();
        this.considerNewConnection();
        this.definitelyDestroyConnection(connection);
    }

    private void considerNewConnection() {
        int inCreation;
        logger.debug("Considering new connection on {} where pool size is {}", (Object)this.host, (Object)this.connections.size());
        do {
            inCreation = this.scheduledForCreation.get();
            logger.debug("There are {} connections scheduled for creation on {}", (Object)inCreation, (Object)this.host);
            if (inCreation < 1) continue;
            return;
        } while (!this.scheduledForCreation.compareAndSet(inCreation, inCreation + 1));
        this.newConnection();
    }

    private void newConnection() {
        this.cluster.executor().submit(() -> {
            this.addConnectionIfUnderMaximum();
            this.scheduledForCreation.decrementAndGet();
            return null;
        });
    }

    private boolean addConnectionIfUnderMaximum() {
        int opened;
        do {
            if ((opened = this.open.get()) < this.maxPoolSize) continue;
            return false;
        } while (!this.open.compareAndSet(opened, opened + 1));
        if (this.isClosed()) {
            this.open.decrementAndGet();
            return false;
        }
        try {
            this.connections.add(new Connection(this.host.getHostUri(), this, this.cluster, this.settings().maxInProcessPerConnection));
        }
        catch (ConnectionException ce) {
            logger.debug("Connections were under max, but there was an error creating the connection.", (Throwable)ce);
            this.considerUnavailable();
            return false;
        }
        this.announceAvailableConnection();
        return true;
    }

    private boolean destroyConnection(Connection connection) {
        int opened;
        do {
            if ((opened = this.open.get()) > this.minPoolSize) continue;
            return false;
        } while (!this.open.compareAndSet(opened, opened - 1));
        this.definitelyDestroyConnection(connection);
        return true;
    }

    private void definitelyDestroyConnection(Connection connection) {
        this.bin.add(connection);
        this.connections.remove(connection);
        if (connection.inFlight.get() == 0 && this.bin.remove(connection)) {
            connection.closeAsync();
        }
        logger.debug("{} destroyed", (Object)connection);
    }

    private Connection waitForConnection(long timeout, TimeUnit unit) throws TimeoutException, ConnectionException {
        long start = System.nanoTime();
        long remaining = timeout;
        long to = timeout;
        do {
            block6: {
                try {
                    this.awaitAvailableConnection(remaining, unit);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    to = 0L;
                }
                if (this.isClosed()) {
                    throw new ConnectionException(this.host.getHostUri(), this.host.getAddress(), "Pool is shutdown");
                }
                Connection leastUsed = this.selectLeastUsed();
                if (leastUsed != null) {
                    int inFlight;
                    do {
                        int availableInProcess;
                        if ((inFlight = leastUsed.inFlight.get()) < (availableInProcess = leastUsed.availableInProcess())) continue;
                        logger.debug("Least used {} on {} has requests inFlight [{}] >= availableInProcess [{}] - may timeout waiting for connection", new Object[]{leastUsed, this.host, inFlight, availableInProcess});
                        break block6;
                    } while (!leastUsed.inFlight.compareAndSet(inFlight, inFlight + 1));
                    logger.debug("Return least used {} on {} after waiting", (Object)leastUsed, (Object)this.host);
                    return leastUsed;
                }
            }
            remaining = to - TimeUtil.timeSince((long)start, (TimeUnit)unit);
            logger.debug("Continue to wait for connection on {} if {} > 0", (Object)remaining);
        } while (remaining > 0L);
        logger.debug("Timed-out waiting for connection on {} - possibly unavailable", (Object)this.host);
        this.considerUnavailable();
        throw new TimeoutException();
    }

    private void considerUnavailable() {
        this.host.makeUnavailable(this::tryReconnect);
        this.cluster.loadBalancingStrategy().onUnavailable(this.host);
    }

    private boolean tryReconnect(Host h) {
        logger.debug("Trying to re-establish connection on {}", (Object)this.host);
        try {
            this.connections.add(new Connection(this.host.getHostUri(), this, this.cluster, this.settings().maxInProcessPerConnection));
            this.open.set(this.connections.size());
            this.cluster.loadBalancingStrategy().onAvailable(this.host);
            return true;
        }
        catch (Exception ex) {
            return false;
        }
    }

    private void announceAvailableConnection() {
        logger.debug("Announce connection available on {}", (Object)this.host);
        if (this.waiter == 0) {
            return;
        }
        this.waitLock.lock();
        try {
            this.hasAvailableConnection.signal();
        }
        finally {
            this.waitLock.unlock();
        }
    }

    private Connection selectLeastUsed() {
        int minInFlight = Integer.MAX_VALUE;
        Connection leastBusy = null;
        for (Connection connection : this.connections) {
            int inFlight = connection.inFlight.get();
            if (connection.isDead() || inFlight >= minInFlight) continue;
            minInFlight = inFlight;
            leastBusy = connection;
        }
        return leastBusy;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void awaitAvailableConnection(long timeout, TimeUnit unit) throws InterruptedException {
        logger.debug("Wait {} {} for an available connection on {} with {}", new Object[]{timeout, unit, this.host, Thread.currentThread()});
        this.waitLock.lock();
        ++this.waiter;
        try {
            this.hasAvailableConnection.await(timeout, unit);
        }
        finally {
            --this.waiter;
            this.waitLock.unlock();
        }
    }

    private void announceAllAvailableConnection() {
        if (this.waiter == 0) {
            return;
        }
        this.waitLock.lock();
        try {
            this.hasAvailableConnection.signalAll();
        }
        finally {
            this.waitLock.unlock();
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("ConnectionPool (");
        sb.append(this.host);
        sb.append(") - ");
        this.connections.forEach(c -> {
            sb.append(c);
            sb.append(",");
        });
        return sb.toString().trim();
    }
}

