/*
 * Decompiled with CFR 0.152.
 */
package org.easysearch.common;

import java.time.Instant;
import java.time.ZoneId;
import java.time.temporal.ChronoField;
import java.time.zone.ZoneOffsetTransition;
import java.time.zone.ZoneOffsetTransitionRule;
import java.time.zone.ZoneRules;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;

public abstract class LocalTimeOffset {
    private final long millis;
    private static final int MAX_TRANSITIONS = 5000;

    public static Lookup lookup(ZoneId zone, long minUtcMillis, long maxUtcMillis) {
        if (minUtcMillis > maxUtcMillis) {
            throw new IllegalArgumentException("[" + minUtcMillis + "] must be <= [" + maxUtcMillis + "]");
        }
        ZoneRules rules = zone.getRules();
        LocalTimeOffset fixed = LocalTimeOffset.checkForFixedZone(zone, rules);
        if (fixed != null) {
            return new FixedLookup(zone, fixed);
        }
        List<ZoneOffsetTransition> transitions = LocalTimeOffset.collectTransitions(zone, rules, minUtcMillis, maxUtcMillis);
        if (transitions == null) {
            return null;
        }
        if (transitions.size() < 3) {
            return new LinkedListLookup(zone, minUtcMillis, maxUtcMillis, transitions);
        }
        return new TransitionArrayLookup(zone, minUtcMillis, maxUtcMillis, transitions);
    }

    public static LocalTimeOffset fixedOffset(ZoneId zone) {
        return LocalTimeOffset.checkForFixedZone(zone, zone.getRules());
    }

    private LocalTimeOffset(long millis) {
        this.millis = millis;
    }

    public final long utcToLocalTime(long utcMillis) {
        return utcMillis + this.millis;
    }

    public final long localToUtcInThisOffset(long localMillis) {
        return localMillis - this.millis;
    }

    public abstract long localToUtc(long var1, Strategy var3);

    protected abstract boolean containsUtcMillis(long var1);

    protected abstract LocalTimeOffset offsetContaining(long var1);

    protected abstract boolean anyMoveBackToPreviousDay();

    public String toString() {
        return this.toString(this.millis);
    }

    protected abstract String toString(long var1);

    private static LocalTimeOffset checkForFixedZone(ZoneId zone, ZoneRules rules) {
        if (!rules.isFixedOffset()) {
            return null;
        }
        NoPrevious fixedTransition = new NoPrevious(rules.getOffset(Instant.EPOCH).getTotalSeconds() * 1000);
        return fixedTransition;
    }

    private static List<ZoneOffsetTransition> collectTransitions(ZoneId zone, ZoneRules rules, long minUtcMillis, long maxUtcMillis) {
        long minSecond = minUtcMillis / 1000L;
        long maxSecond = maxUtcMillis / 1000L;
        List<ZoneOffsetTransition> transitions = new ArrayList<ZoneOffsetTransition>();
        ZoneOffsetTransition t = null;
        Iterator<ZoneOffsetTransition> itr = rules.getTransitions().iterator();
        while (itr.hasNext() && (t = itr.next()).toEpochSecond() < minSecond) {
        }
        if (!itr.hasNext()) {
            if (minSecond < t.toEpochSecond() && t.toEpochSecond() < maxSecond) {
                transitions.add(t);
                minSecond = t.toEpochSecond() + 1L;
            }
            if ((transitions = LocalTimeOffset.buildTransitionsFromRules(transitions, zone, rules, minSecond, maxSecond)) != null && transitions.isEmpty()) {
                transitions.add(t);
            }
            return transitions;
        }
        transitions.add(t);
        while (itr.hasNext()) {
            t = itr.next();
            if (t.toEpochSecond() > maxSecond) {
                return transitions;
            }
            transitions.add(t);
            if (transitions.size() <= 5000) continue;
            return null;
        }
        return LocalTimeOffset.buildTransitionsFromRules(transitions, zone, rules, t.toEpochSecond() + 1L, maxSecond);
    }

    private static List<ZoneOffsetTransition> buildTransitionsFromRules(List<ZoneOffsetTransition> transitions, ZoneId zone, ZoneRules rules, long minSecond, long maxSecond) {
        List<ZoneOffsetTransitionRule> transitionRules = rules.getTransitionRules();
        if (transitionRules.isEmpty()) {
            return transitions;
        }
        int minYear = Instant.ofEpochSecond(minSecond).atZone(zone).toLocalDate().getYear();
        int maxYear = Instant.ofEpochSecond(maxSecond).atZone(zone).toLocalDate().getYear();
        ZoneOffsetTransition lastTransitionFromMinYear = null;
        for (ZoneOffsetTransitionRule rule : transitionRules) {
            lastTransitionFromMinYear = rule.createTransition(minYear);
            if (lastTransitionFromMinYear.toEpochSecond() < minSecond) continue;
            transitions.add(lastTransitionFromMinYear);
            if (transitions.size() <= 5000) continue;
            return null;
        }
        if (minYear == maxYear) {
            if (transitions.isEmpty()) {
                transitions.add(lastTransitionFromMinYear);
            }
            return transitions;
        }
        if (transitions.size() + (maxYear - ++minYear) * transitionRules.size() > 5000) {
            return null;
        }
        for (int year = minYear; year <= maxYear; ++year) {
            for (ZoneOffsetTransitionRule rule : transitionRules) {
                transitions.add(rule.createTransition(year));
            }
        }
        return transitions;
    }

    private static class FixedLookup
    extends Lookup {
        private final ZoneId zone;
        private final LocalTimeOffset fixed;

        private FixedLookup(ZoneId zone, LocalTimeOffset fixed) {
            this.zone = zone;
            this.fixed = fixed;
        }

        @Override
        public LocalTimeOffset lookup(long utcMillis) {
            return this.fixed;
        }

        @Override
        public LocalTimeOffset fixedInRange(long minUtcMillis, long maxUtcMillis) {
            return this.fixed;
        }

        @Override
        int size() {
            return 1;
        }

        public String toString() {
            return String.format(Locale.ROOT, "FixedLookup[for %s at %s]", this.zone, this.fixed);
        }

        @Override
        public boolean anyMoveBackToPreviousDay() {
            return false;
        }
    }

    private static class LinkedListLookup
    extends AbstractManyTransitionsLookup {
        private final LocalTimeOffset lastOffset;
        private final int size;

        LinkedListLookup(ZoneId zone, long minUtcMillis, long maxUtcMillis, List<ZoneOffsetTransition> transitions) {
            super(zone, minUtcMillis, maxUtcMillis);
            int size = 1;
            LocalTimeOffset last = LinkedListLookup.buildNoPrevious(transitions.get(0));
            for (ZoneOffsetTransition t : transitions) {
                last = LinkedListLookup.buildTransition(t, last);
                ++size;
            }
            this.lastOffset = last;
            this.size = size;
        }

        @Override
        public LocalTimeOffset innerLookup(long utcMillis) {
            return this.lastOffset.offsetContaining(utcMillis);
        }

        @Override
        int size() {
            return this.size;
        }

        @Override
        public boolean anyMoveBackToPreviousDay() {
            return this.lastOffset.anyMoveBackToPreviousDay();
        }
    }

    private static class TransitionArrayLookup
    extends AbstractManyTransitionsLookup {
        private final LocalTimeOffset[] offsets;
        private final long[] transitionOutUtcMillis;

        private TransitionArrayLookup(ZoneId zone, long minUtcMillis, long maxUtcMillis, List<ZoneOffsetTransition> transitions) {
            super(zone, minUtcMillis, maxUtcMillis);
            this.offsets = new LocalTimeOffset[transitions.size() + 1];
            this.transitionOutUtcMillis = new long[transitions.size()];
            this.offsets[0] = TransitionArrayLookup.buildNoPrevious(transitions.get(0));
            int i = 0;
            for (ZoneOffsetTransition t : transitions) {
                Transition transition = TransitionArrayLookup.buildTransition(t, this.offsets[i]);
                this.transitionOutUtcMillis[i] = transition.startUtcMillis();
                this.offsets[++i] = transition;
            }
        }

        @Override
        protected LocalTimeOffset innerLookup(long utcMillis) {
            int index = Arrays.binarySearch(this.transitionOutUtcMillis, utcMillis);
            index = index < 0 ? -index - 1 : ++index;
            assert (index < this.offsets.length) : "binarySearch did something weird";
            return this.offsets[index];
        }

        @Override
        int size() {
            return this.offsets.length;
        }

        @Override
        public boolean anyMoveBackToPreviousDay() {
            return this.offsets[this.offsets.length - 1].anyMoveBackToPreviousDay();
        }

        public String toString() {
            return String.format(Locale.ROOT, "TransitionArrayLookup[for %s between %s and %s]", this.zone, Instant.ofEpochMilli(this.minUtcMillis), Instant.ofEpochMilli(this.maxUtcMillis));
        }
    }

    private static class NoPrevious
    extends LocalTimeOffset {
        NoPrevious(long millis) {
            super(millis);
        }

        @Override
        public long localToUtc(long localMillis, Strategy strat) {
            return this.localToUtcInThisOffset(localMillis);
        }

        @Override
        protected boolean containsUtcMillis(long utcMillis) {
            return true;
        }

        @Override
        protected LocalTimeOffset offsetContaining(long utcMillis) {
            return this;
        }

        @Override
        protected boolean anyMoveBackToPreviousDay() {
            return false;
        }

        @Override
        protected String toString(long millis) {
            return Long.toString(millis);
        }
    }

    private static abstract class AbstractManyTransitionsLookup
    extends Lookup {
        protected final ZoneId zone;
        protected final long minUtcMillis;
        protected final long maxUtcMillis;

        AbstractManyTransitionsLookup(ZoneId zone, long minUtcMillis, long maxUtcMillis) {
            this.zone = zone;
            this.minUtcMillis = minUtcMillis;
            this.maxUtcMillis = maxUtcMillis;
        }

        @Override
        public final LocalTimeOffset lookup(long utcMillis) {
            assert (utcMillis >= this.minUtcMillis);
            assert (utcMillis <= this.maxUtcMillis);
            return this.innerLookup(utcMillis);
        }

        protected abstract LocalTimeOffset innerLookup(long var1);

        @Override
        public final LocalTimeOffset fixedInRange(long minUtcMillis, long maxUtcMillis) {
            LocalTimeOffset offset = this.lookup(maxUtcMillis);
            return offset.containsUtcMillis(minUtcMillis) ? offset : null;
        }

        protected static NoPrevious buildNoPrevious(ZoneOffsetTransition transition) {
            return new NoPrevious(transition.getOffsetBefore().getTotalSeconds() * 1000);
        }

        protected static Transition buildTransition(ZoneOffsetTransition transition, LocalTimeOffset previous) {
            long utcStart = transition.toEpochSecond() * 1000L;
            long offsetBeforeMillis = transition.getOffsetBefore().getTotalSeconds() * 1000;
            long offsetAfterMillis = transition.getOffsetAfter().getTotalSeconds() * 1000;
            assert (!(previous instanceof Transition) || ((Transition)previous).startUtcMillis < utcStart) : "transition list out of order at [" + previous + "] and [" + transition + "]";
            assert (previous.millis != offsetAfterMillis) : "transition list is has a duplicate at [" + previous + "] and [" + transition + "]";
            if (transition.isGap()) {
                long firstMissingLocalTime = utcStart + offsetBeforeMillis;
                long firstLocalTimeAfterGap = utcStart + offsetAfterMillis;
                return new Gap(offsetAfterMillis, previous, utcStart, firstMissingLocalTime, firstLocalTimeAfterGap);
            }
            long firstOverlappingLocalTime = utcStart + offsetAfterMillis;
            long firstNonOverlappingLocalTime = utcStart + offsetBeforeMillis;
            return new Overlap(offsetAfterMillis, previous, utcStart, firstOverlappingLocalTime, firstNonOverlappingLocalTime, AbstractManyTransitionsLookup.movesBackToPreviousDay(transition));
        }

        private static boolean movesBackToPreviousDay(ZoneOffsetTransition transition) {
            if (transition.getDateTimeBefore().getDayOfMonth() == transition.getDateTimeAfter().getDayOfMonth()) {
                return false;
            }
            return transition.getDateTimeBefore().getLong(ChronoField.NANO_OF_DAY) != 0L;
        }
    }

    public static class Overlap
    extends Transition {
        private final long firstOverlappingLocalTime;
        private final long firstNonOverlappingLocalTime;
        private final boolean movesBackToPreviousDay;

        private Overlap(long millis, LocalTimeOffset previous, long startUtcMillis, long firstOverlappingLocalTime, long firstNonOverlappingLocalTime, boolean movesBackToPreviousDay) {
            super(millis, previous, startUtcMillis);
            this.firstOverlappingLocalTime = firstOverlappingLocalTime;
            this.firstNonOverlappingLocalTime = firstNonOverlappingLocalTime;
            assert (firstOverlappingLocalTime < firstNonOverlappingLocalTime);
            this.movesBackToPreviousDay = movesBackToPreviousDay;
        }

        @Override
        public long localToUtc(long localMillis, Strategy strat) {
            if (localMillis >= this.firstNonOverlappingLocalTime) {
                return this.localToUtcInThisOffset(localMillis);
            }
            if (localMillis >= this.firstOverlappingLocalTime) {
                return strat.inOverlap(localMillis, this);
            }
            return strat.beforeOverlap(localMillis, this);
        }

        public long firstNonOverlappingLocalTime() {
            return this.firstNonOverlappingLocalTime;
        }

        public long firstOverlappingLocalTime() {
            return this.firstOverlappingLocalTime;
        }

        @Override
        protected boolean anyMoveBackToPreviousDay() {
            return this.movesBackToPreviousDay || this.previous().anyMoveBackToPreviousDay();
        }

        @Override
        protected String toString(long millis) {
            return "Overlap of " + millis + "@" + Instant.ofEpochMilli(this.startUtcMillis());
        }
    }

    public static class Gap
    extends Transition {
        private final long firstMissingLocalTime;
        private final long firstLocalTimeAfterGap;

        private Gap(long millis, LocalTimeOffset previous, long startUtcMillis, long firstMissingLocalTime, long firstLocalTimeAfterGap) {
            super(millis, previous, startUtcMillis);
            this.firstMissingLocalTime = firstMissingLocalTime;
            this.firstLocalTimeAfterGap = firstLocalTimeAfterGap;
            assert (firstMissingLocalTime < firstLocalTimeAfterGap);
        }

        @Override
        public long localToUtc(long localMillis, Strategy strat) {
            if (localMillis >= this.firstLocalTimeAfterGap) {
                return this.localToUtcInThisOffset(localMillis);
            }
            if (localMillis >= this.firstMissingLocalTime) {
                return strat.inGap(localMillis, this);
            }
            return strat.beforeGap(localMillis, this);
        }

        public long firstMissingLocalTime() {
            return this.firstMissingLocalTime;
        }

        @Override
        protected boolean anyMoveBackToPreviousDay() {
            return this.previous().anyMoveBackToPreviousDay();
        }

        @Override
        protected String toString(long millis) {
            return "Gap of " + millis + "@" + Instant.ofEpochMilli(this.startUtcMillis());
        }
    }

    public static abstract class Transition
    extends LocalTimeOffset {
        private final LocalTimeOffset previous;
        private final long startUtcMillis;

        private Transition(long millis, LocalTimeOffset previous, long startUtcMillis) {
            super(millis);
            this.previous = previous;
            this.startUtcMillis = startUtcMillis;
        }

        public LocalTimeOffset previous() {
            return this.previous;
        }

        @Override
        protected final boolean containsUtcMillis(long utcMillis) {
            return utcMillis >= this.startUtcMillis;
        }

        @Override
        protected final LocalTimeOffset offsetContaining(long utcMillis) {
            if (this.containsUtcMillis(utcMillis)) {
                return this;
            }
            return this.previous.offsetContaining(utcMillis);
        }

        public long startUtcMillis() {
            return this.startUtcMillis;
        }
    }

    public static abstract class Lookup {
        public abstract LocalTimeOffset lookup(long var1);

        public abstract LocalTimeOffset fixedInRange(long var1, long var3);

        public abstract boolean anyMoveBackToPreviousDay();

        abstract int size();
    }

    public static interface Strategy {
        public long inGap(long var1, Gap var3);

        public long beforeGap(long var1, Gap var3);

        public long inOverlap(long var1, Overlap var3);

        public long beforeOverlap(long var1, Overlap var3);
    }
}

