/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.io.file;

import java.io.File;
import java.io.FileFilter;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.apache.pulsar.io.file.FileSourceConfig;

public class FileListingThread
extends Thread {
    private final AtomicLong queueLastUpdated = new AtomicLong(0L);
    private final Lock listingLock = new ReentrantLock();
    private final AtomicReference<FileFilter> fileFilterRef = new AtomicReference();
    private final BlockingQueue<File> workQueue;
    private final BlockingQueue<File> inProcess;
    private final BlockingQueue<File> recentlyProcessed;
    private final String inputDir;
    private final boolean recurseDirs;
    private final boolean keepOriginal;
    private final long pollingInterval;

    public FileListingThread(FileSourceConfig fileConfig, BlockingQueue<File> workQueue, BlockingQueue<File> inProcess, BlockingQueue<File> recentlyProcessed) {
        this.workQueue = workQueue;
        this.inProcess = inProcess;
        this.recentlyProcessed = recentlyProcessed;
        this.inputDir = fileConfig.getInputDirectory();
        this.recurseDirs = Optional.ofNullable(fileConfig.getRecurse()).orElse(true);
        this.keepOriginal = Optional.ofNullable(fileConfig.getKeepFile()).orElse(false);
        this.pollingInterval = Optional.ofNullable(fileConfig.getPollingInterval()).orElse(10000L);
        this.fileFilterRef.set(this.createFileFilter(fileConfig));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        while (true) {
            if (this.queueLastUpdated.get() < System.currentTimeMillis() - this.pollingInterval && this.listingLock.tryLock()) {
                try {
                    File directory = new File(this.inputDir);
                    Set<File> listing = this.performListing(directory, this.fileFilterRef.get(), this.recurseDirs);
                    if (listing != null && !listing.isEmpty()) {
                        listing.removeAll(this.inProcess);
                        if (!this.keepOriginal) {
                            listing.removeAll(this.recentlyProcessed);
                        }
                        for (File f : listing) {
                            if (this.workQueue.contains(f)) continue;
                            this.workQueue.offer(f);
                        }
                        this.queueLastUpdated.set(System.currentTimeMillis());
                    }
                }
                finally {
                    this.listingLock.unlock();
                }
            }
            try {
                FileListingThread.sleep(this.pollingInterval - 1L);
            }
            catch (InterruptedException interruptedException) {
            }
        }
    }

    private Set<File> performListing(File directory, FileFilter filter, boolean recurseSubdirectories) {
        Path p = directory.toPath();
        if (!Files.isReadable(p)) {
            throw new IllegalStateException("Cannot read directory: '" + String.valueOf(directory));
        }
        HashSet<File> queue = new HashSet<File>();
        if (!directory.exists()) {
            return queue;
        }
        File[] children = directory.listFiles();
        if (children == null) {
            return queue;
        }
        for (File child : children) {
            if (child.isDirectory()) {
                if (!recurseSubdirectories) continue;
                queue.addAll(this.performListing(child, filter, recurseSubdirectories));
                continue;
            }
            if (!filter.accept(child)) continue;
            queue.add(child);
        }
        return queue;
    }

    private FileFilter createFileFilter(FileSourceConfig fileConfig) {
        final long minSize = Optional.ofNullable(fileConfig.getMinimumSize()).orElse(1).intValue();
        final Double maxSize = Optional.ofNullable(fileConfig.getMaximumSize()).orElse((Double)Double.MAX_VALUE);
        final long minAge = Optional.ofNullable(fileConfig.getMinimumFileAge()).orElse(0).intValue();
        final Long maxAge = Optional.ofNullable(fileConfig.getMaximumFileAge()).orElse(Long.MAX_VALUE);
        final boolean ignoreHidden = Optional.ofNullable(fileConfig.getIgnoreHiddenFiles()).orElse(true);
        final Pattern filePattern = Pattern.compile(Optional.ofNullable(fileConfig.getFileFilter()).orElse("[^\\.].*"));
        final String indir = fileConfig.getInputDirectory();
        String pathPatternStr = fileConfig.getPathFilter();
        final Pattern pathPattern = !this.recurseDirs || pathPatternStr == null ? null : Pattern.compile(pathPatternStr);
        final String processedFileSuffix = fileConfig.getProcessedFileSuffix();
        return new FileFilter(){

            @Override
            public boolean accept(File file) {
                Path reldir;
                if (minSize > file.length()) {
                    return false;
                }
                if (maxSize != null && maxSize < (double)file.length()) {
                    return false;
                }
                long fileAge = System.currentTimeMillis() - file.lastModified();
                if (minAge > fileAge) {
                    return false;
                }
                if (maxAge != null && maxAge < fileAge) {
                    return false;
                }
                if (ignoreHidden && file.isHidden()) {
                    return false;
                }
                if (pathPattern != null && (reldir = Paths.get(indir, new String[0]).relativize(file.toPath()).getParent()) != null && !reldir.toString().isEmpty() && !pathPattern.matcher(reldir.toString()).matches()) {
                    return false;
                }
                if (!Files.isReadable(file.toPath())) {
                    return false;
                }
                if (!FileListingThread.this.keepOriginal && !Files.isWritable(file.toPath().getParent())) {
                    return false;
                }
                if (!FileListingThread.this.keepOriginal && !StringUtils.isBlank((CharSequence)processedFileSuffix) && file.getName().endsWith(processedFileSuffix)) {
                    return false;
                }
                return filePattern.matcher(file.getName()).matches();
            }
        };
    }
}

