/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pdfbox.examples.signature.cert;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.security.GeneralSecurityException;
import java.security.cert.CRLException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509CRLEntry;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Hashtable;
import java.util.List;
import java.util.Set;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.InitialDirContext;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.pdfbox.examples.signature.SigUtils;
import org.apache.pdfbox.examples.signature.cert.CertificateVerificationException;
import org.apache.pdfbox.examples.signature.cert.CertificateVerifier;
import org.apache.pdfbox.examples.signature.cert.RevokedCertificateException;
import org.apache.pdfbox.pdmodel.encryption.SecurityProvider;
import org.bouncycastle.asn1.ASN1IA5String;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.x509.CRLDistPoint;
import org.bouncycastle.asn1.x509.DistributionPoint;
import org.bouncycastle.asn1.x509.DistributionPointName;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;

public final class CRLVerifier {
    private static final Log LOG = LogFactory.getLog(CRLVerifier.class);

    private CRLVerifier() {
    }

    public static void verifyCertificateCRLs(X509Certificate cert, Date signDate, Set<X509Certificate> additionalCerts) throws CertificateVerificationException, RevokedCertificateException {
        try {
            Date now = Calendar.getInstance().getTime();
            Exception firstException = null;
            List<String> crlDistributionPointsURLs = CRLVerifier.getCrlDistributionPoints(cert);
            for (String crlDistributionPointsURL : crlDistributionPointsURLs) {
                X509CRL crl;
                LOG.info((Object)("Checking distribution point URL: " + crlDistributionPointsURL));
                try {
                    crl = CRLVerifier.downloadCRL(crlDistributionPointsURL);
                }
                catch (IOException | GeneralSecurityException | NamingException | CertificateVerificationException ex) {
                    LOG.warn((Object)("Caught " + ex.getClass().getSimpleName() + " downloading CRL, will try next distribution point if available"));
                    if (firstException != null) continue;
                    firstException = ex;
                    continue;
                }
                Set<X509Certificate> mergedCertSet = CertificateVerifier.downloadExtraCertificates(crl);
                mergedCertSet.addAll(additionalCerts);
                Certificate crlIssuerCert = null;
                for (X509Certificate possibleCert : mergedCertSet) {
                    try {
                        cert.verify(possibleCert.getPublicKey(), SecurityProvider.getProvider());
                        crlIssuerCert = possibleCert;
                        break;
                    }
                    catch (GeneralSecurityException generalSecurityException) {
                    }
                }
                if (crlIssuerCert == null) {
                    throw new CertificateVerificationException("Certificate for " + crl.getIssuerX500Principal() + "not found in certificate chain, so the CRL at " + crlDistributionPointsURL + " could not be verified");
                }
                crl.verify(crlIssuerCert.getPublicKey(), SecurityProvider.getProvider());
                if (crl.getThisUpdate().after(now)) {
                    LOG.error((Object)("CRL not yet valid, thisUpdate is " + crl.getThisUpdate()));
                }
                if (crl.getNextUpdate().before(now)) {
                    LOG.error((Object)("CRL no longer valid, nextUpdate is " + crl.getNextUpdate()));
                }
                if (!crl.getIssuerX500Principal().equals(cert.getIssuerX500Principal())) {
                    LOG.info((Object)"CRL issuer certificate is not identical to cert issuer, check needed");
                    CertificateVerifier.verifyCertificate((X509Certificate)crlIssuerCert, mergedCertSet, true, now);
                    LOG.info((Object)"CRL issuer certificate checked successfully");
                } else {
                    LOG.info((Object)"CRL issuer certificate is identical to cert issuer, no extra check needed");
                }
                CRLVerifier.checkRevocation(crl, cert, signDate, crlDistributionPointsURL);
                return;
            }
            if (firstException != null) {
                throw firstException;
            }
        }
        catch (CertificateVerificationException | RevokedCertificateException ex) {
            throw ex;
        }
        catch (Exception ex) {
            throw new CertificateVerificationException("Cannot verify CRL for certificate: " + cert.getSubjectX500Principal(), ex);
        }
    }

    public static void checkRevocation(X509CRL crl, X509Certificate cert, Date signDate, String crlDistributionPointsURL) throws RevokedCertificateException {
        X509CRLEntry revokedCRLEntry = crl.getRevokedCertificate(cert);
        if (revokedCRLEntry != null && revokedCRLEntry.getRevocationDate().compareTo(signDate) <= 0) {
            throw new RevokedCertificateException("The certificate was revoked by CRL " + crlDistributionPointsURL + " on " + revokedCRLEntry.getRevocationDate(), revokedCRLEntry.getRevocationDate());
        }
        if (revokedCRLEntry != null) {
            LOG.info((Object)("The certificate was revoked after signing by CRL " + crlDistributionPointsURL + " on " + revokedCRLEntry.getRevocationDate()));
        } else {
            LOG.info((Object)("The certificate was not revoked by CRL " + crlDistributionPointsURL));
        }
    }

    private static X509CRL downloadCRL(String crlURL) throws IOException, CertificateException, CRLException, CertificateVerificationException, NamingException, URISyntaxException {
        if (crlURL.startsWith("http://") || crlURL.startsWith("https://") || crlURL.startsWith("ftp://")) {
            return CRLVerifier.downloadCRLFromWeb(crlURL);
        }
        if (crlURL.startsWith("ldap://")) {
            return CRLVerifier.downloadCRLFromLDAP(crlURL);
        }
        throw new CertificateVerificationException("Can not download CRL from certificate distribution point: " + crlURL);
    }

    private static X509CRL downloadCRLFromLDAP(String ldapURL) throws CertificateException, NamingException, CRLException, CertificateVerificationException {
        Hashtable<String, String> env = new Hashtable<String, String>();
        env.put("java.naming.factory.initial", "com.sun.jndi.ldap.LdapCtxFactory");
        env.put("java.naming.provider.url", ldapURL);
        env.put("com.sun.jndi.ldap.connect.timeout", "1000");
        InitialDirContext ctx = new InitialDirContext(env);
        Attributes avals = ctx.getAttributes("");
        Attribute aval = avals.get("certificateRevocationList;binary");
        byte[] val = (byte[])aval.get();
        if (val == null || val.length == 0) {
            throw new CertificateVerificationException("Can not download CRL from: " + ldapURL);
        }
        ByteArrayInputStream inStream = new ByteArrayInputStream(val);
        CertificateFactory cf = CertificateFactory.getInstance("X.509");
        return (X509CRL)cf.generateCRL(inStream);
    }

    public static X509CRL downloadCRLFromWeb(String crlURL) throws IOException, CertificateException, CRLException, URISyntaxException {
        try (InputStream crlStream = SigUtils.openURL(crlURL);){
            X509CRL x509CRL = (X509CRL)CertificateFactory.getInstance("X.509").generateCRL(crlStream);
            return x509CRL;
        }
    }

    public static List<String> getCrlDistributionPoints(X509Certificate cert) throws IOException {
        ASN1Primitive derObj2;
        ASN1Primitive derObjCrlDP;
        byte[] crldpExt = cert.getExtensionValue(Extension.cRLDistributionPoints.getId());
        if (crldpExt == null) {
            return new ArrayList<String>();
        }
        try (ASN1InputStream oAsnInStream = new ASN1InputStream(crldpExt);){
            derObjCrlDP = oAsnInStream.readObject();
        }
        if (!(derObjCrlDP instanceof ASN1OctetString)) {
            LOG.warn((Object)("CRL distribution points for certificate subject " + cert.getSubjectX500Principal().getName() + " should be an octet string, but is " + derObjCrlDP));
            return new ArrayList<String>();
        }
        ASN1OctetString dosCrlDP = (ASN1OctetString)derObjCrlDP;
        byte[] crldpExtOctets = dosCrlDP.getOctets();
        try (ASN1InputStream oAsnInStream2 = new ASN1InputStream(crldpExtOctets);){
            derObj2 = oAsnInStream2.readObject();
        }
        CRLDistPoint distPoint = CRLDistPoint.getInstance((Object)derObj2);
        ArrayList<String> crlUrls = new ArrayList<String>();
        for (DistributionPoint dp : distPoint.getDistributionPoints()) {
            DistributionPointName dpn = dp.getDistributionPoint();
            if (dpn == null || dpn.getType() != 0) continue;
            for (GeneralName genName : GeneralNames.getInstance((Object)dpn.getName()).getNames()) {
                if (genName.getTagNo() != 6) continue;
                String url = ASN1IA5String.getInstance((Object)genName.getName()).getString();
                crlUrls.add(url);
            }
        }
        return crlUrls;
    }
}

