/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.debug.logs.container.utils;

import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.container.ReplicationManagerReport;
import org.apache.hadoop.ozone.debug.logs.container.utils.DatanodeContainerInfo;
import org.apache.hadoop.ozone.debug.logs.container.utils.SQLDBConstants;
import org.apache.hadoop.ozone.om.OMConfigKeys;
import org.sqlite.SQLiteConfig;

public class ContainerDatanodeDatabase {
    private final String databasePath;
    private static final int DEFAULT_REPLICATION_FACTOR;
    private final PrintWriter out;
    private final PrintWriter err;

    public ContainerDatanodeDatabase(String dbPath) {
        this.databasePath = dbPath;
        this.out = new PrintWriter((Writer)new OutputStreamWriter((OutputStream)System.out, StandardCharsets.UTF_8), true);
        this.err = new PrintWriter((Writer)new OutputStreamWriter((OutputStream)System.err, StandardCharsets.UTF_8), true);
    }

    public ContainerDatanodeDatabase(String dbPath, PrintWriter out, PrintWriter err) {
        this.databasePath = dbPath;
        this.out = out;
        this.err = err;
    }

    private Connection getConnection() throws Exception {
        if (this.databasePath == null) {
            throw new IllegalStateException("Database path not set");
        }
        Class.forName("org.sqlite.JDBC");
        SQLiteConfig config = new SQLiteConfig();
        config.setJournalMode(SQLiteConfig.JournalMode.OFF);
        config.setCacheSize(1000000);
        config.setLockingMode(SQLiteConfig.LockingMode.EXCLUSIVE);
        config.setSynchronous(SQLiteConfig.SynchronousMode.OFF);
        config.setTempStore(SQLiteConfig.TempStore.MEMORY);
        return DriverManager.getConnection("jdbc:sqlite:" + this.databasePath, config.toProperties());
    }

    public void createDatanodeContainerLogTable() throws SQLException {
        String createTableSQL = "CREATE TABLE IF NOT EXISTS DatanodeContainerLogTable (datanode_id TEXT NOT NULL, container_id INTEGER NOT NULL, timestamp TEXT NOT NULL, container_state TEXT, bcsid INTEGER, error_message TEXT, log_level TEXT NOT NULL, index_value INTEGER);";
        try (Connection connection = this.getConnection();
             Statement dropStmt = connection.createStatement();
             Statement createStmt = connection.createStatement();){
            this.dropTable("DatanodeContainerLogTable", dropStmt);
            createStmt.execute(createTableSQL);
            this.createDatanodeContainerIndex(createStmt);
        }
        catch (SQLException e) {
            throw new SQLException("Error while creating the table: " + e.getMessage());
        }
        catch (Exception e) {
            throw new RuntimeException("Unexpected error: " + e);
        }
    }

    private void createContainerLogTable() throws SQLException {
        String createTableSQL = "CREATE TABLE IF NOT EXISTS ContainerLogTable (datanode_id TEXT NOT NULL, container_id INTEGER NOT NULL, latest_state TEXT, latest_bcsid INTEGER, PRIMARY KEY (datanode_id, container_id));";
        try (Connection connection = this.getConnection();
             Statement dropStmt = connection.createStatement();
             Statement createStmt = connection.createStatement();){
            this.dropTable("ContainerLogTable", dropStmt);
            createStmt.execute(createTableSQL);
        }
        catch (SQLException e) {
            throw new SQLException("Error while creating the table: " + e.getMessage());
        }
        catch (Exception e) {
            throw new RuntimeException("Unexpected error: " + e);
        }
    }

    public void createIndexes() throws SQLException {
        try (Connection connection = this.getConnection();
             Statement stmt = connection.createStatement();){
            this.createIdxDclContainerStateTime(stmt);
            this.createContainerLogIndex(stmt);
            this.createIdxContainerlogContainerId(stmt);
            this.createIndexForQuasiClosedQuery(stmt);
        }
        catch (SQLException e) {
            throw new SQLException("Error while creating index: " + e.getMessage());
        }
        catch (Exception e) {
            throw new RuntimeException("Unexpected error: " + e);
        }
    }

    private void createIdxDclContainerStateTime(Statement stmt) throws SQLException {
        String createIndexSQL = "CREATE INDEX IF NOT EXISTS idx_dcl_container_state_time ON DatanodeContainerLogTable(container_id, container_state, timestamp);";
        stmt.execute(createIndexSQL);
    }

    private void createContainerLogIndex(Statement stmt) throws SQLException {
        String createIndexSQL = "CREATE INDEX IF NOT EXISTS idx_container_log_state ON ContainerLogTable(latest_state);";
        stmt.execute(createIndexSQL);
    }

    private void createIdxContainerlogContainerId(Statement stmt) throws SQLException {
        String createIndexSQL = "CREATE INDEX IF NOT EXISTS idx_containerlog_container_id ON ContainerLogTable(container_id);";
        stmt.execute(createIndexSQL);
    }

    private void createIndexForQuasiClosedQuery(Statement stmt) throws SQLException {
        String createIndexSQL = "CREATE INDEX IF NOT EXISTS idx_dcl_state_container_datanode_time ON DatanodeContainerLogTable(container_state, container_id, datanode_id, timestamp DESC);";
        stmt.execute(createIndexSQL);
    }

    public synchronized void insertContainerDatanodeData(List<DatanodeContainerInfo> transitionList) throws SQLException {
        String insertSQL = "INSERT INTO DatanodeContainerLogTable (datanode_id, container_id, timestamp, container_state, bcsid, error_message, log_level, index_value) VALUES (?, ?, ?, ?, ?, ?, ?, ?);";
        long containerId = 0L;
        String datanodeId = null;
        try (Connection connection = this.getConnection();
             PreparedStatement preparedStatement = connection.prepareStatement(insertSQL);){
            int count = 0;
            for (DatanodeContainerInfo info : transitionList) {
                datanodeId = info.getDatanodeId();
                containerId = info.getContainerId();
                preparedStatement.setString(1, datanodeId);
                preparedStatement.setLong(2, containerId);
                preparedStatement.setString(3, info.getTimestamp());
                preparedStatement.setString(4, info.getState());
                preparedStatement.setLong(5, info.getBcsid());
                preparedStatement.setString(6, info.getErrorMessage());
                preparedStatement.setString(7, info.getLogLevel());
                preparedStatement.setInt(8, info.getIndexValue());
                preparedStatement.addBatch();
                if (++count % 2500 != 0) continue;
                preparedStatement.executeBatch();
                count = 0;
            }
            if (count != 0) {
                preparedStatement.executeBatch();
            }
        }
        catch (SQLException e) {
            throw new SQLException("Failed to insert container log for container " + containerId + " on datanode " + datanodeId);
        }
        catch (Exception e) {
            throw new RuntimeException("Unexpected error: " + e);
        }
    }

    private void createDatanodeContainerIndex(Statement stmt) throws SQLException {
        String createIndexSQL = "CREATE INDEX IF NOT EXISTS idx_datanode_container ON DatanodeContainerLogTable (datanode_id, container_id, timestamp);";
        stmt.execute(createIndexSQL);
    }

    public void insertLatestContainerLogData() throws SQLException {
        this.createContainerLogTable();
        String selectSQL = "SELECT a.datanode_id, a.container_id, a.container_state, a.bcsid, a.timestamp FROM DatanodeContainerLogTable AS a JOIN  (SELECT datanode_id, container_id, MAX(timestamp) as timestamp FROM DatanodeContainerLogTable GROUP BY datanode_id, container_id) as b ON a.datanode_id = b.datanode_id AND a.container_id = b.container_id AND a.timestamp=b.timestamp;";
        String insertSQL = "INSERT OR REPLACE INTO ContainerLogTable (datanode_id, container_id, latest_state, latest_bcsid) VALUES (?, ?, ?, ?);";
        try (Connection connection = this.getConnection();
             PreparedStatement selectStmt = connection.prepareStatement(selectSQL);
             ResultSet resultSet = selectStmt.executeQuery();
             PreparedStatement insertStmt = connection.prepareStatement(insertSQL);){
            int count = 0;
            while (resultSet.next()) {
                String datanodeId = resultSet.getString("datanode_id");
                long containerId = resultSet.getLong("container_id");
                String containerState = resultSet.getString("container_state");
                long bcsid = resultSet.getLong("bcsid");
                try {
                    insertStmt.setString(1, datanodeId);
                    insertStmt.setLong(2, containerId);
                    insertStmt.setString(3, containerState);
                    insertStmt.setLong(4, bcsid);
                    insertStmt.addBatch();
                    if (++count % 2500 != 0) continue;
                    insertStmt.executeBatch();
                    count = 0;
                }
                catch (SQLException e) {
                    throw new SQLException("Failed to insert container log entry for container " + containerId + " on datanode " + datanodeId);
                }
            }
            if (count != 0) {
                insertStmt.executeBatch();
            }
        }
        catch (SQLException e) {
            throw new SQLException("Failed to insert container log entry: " + e.getMessage());
        }
        catch (Exception e) {
            throw new RuntimeException("Unexpected error: " + e);
        }
    }

    private void dropTable(String tableName, Statement stmt) throws SQLException {
        String dropTableSQL = "DROP TABLE IF EXISTS {table_name};".replace("{table_name}", tableName);
        stmt.executeUpdate(dropTableSQL);
    }

    public void listContainersByState(String state, Integer limit) throws SQLException {
        int count = 0;
        boolean limitProvided = limit != Integer.MAX_VALUE;
        String baseQuery = "SELECT cl.datanode_id, cl.container_id, cl.latest_state, cl.latest_bcsid, dcl.error_message, dcl.index_value, dcl.timestamp FROM ContainerLogTable cl LEFT JOIN DatanodeContainerLogTable dcl ON cl.datanode_id = dcl.datanode_id AND cl.container_id = dcl.container_id AND cl.latest_bcsid = dcl.bcsid AND cl.latest_state = dcl.container_state WHERE cl.latest_state = ? AND dcl.timestamp = (SELECT MAX(timestamp) FROM DatanodeContainerLogTable sub_dcl WHERE sub_dcl.datanode_id = cl.datanode_id AND sub_dcl.container_id = cl.container_id AND sub_dcl.bcsid = cl.latest_bcsid AND sub_dcl.container_state = cl.latest_state)";
        String finalQuery = limitProvided ? baseQuery + " LIMIT ?" : baseQuery;
        try (Connection connection = this.getConnection();
             PreparedStatement pstmt = connection.prepareStatement(finalQuery);){
            pstmt.setString(1, state);
            if (limitProvided) {
                pstmt.setInt(2, limit + 1);
            }
            try (ResultSet rs = pstmt.executeQuery();){
                this.out.printf("%-25s | %-35s | %-15s | %-15s | %-40s | %-12s%n", "Timestamp", "Datanode ID", "Container ID", "BCSID", "Message", "Index Value");
                this.out.println("----------------------------------------------------------------------------------------------------------------------------------------------------------------------------");
                while (rs.next()) {
                    if (limitProvided && count >= limit) {
                        this.out.println("Note: There might be more containers. Use -all option to list all entries");
                        break;
                    }
                    String timestamp = rs.getString("timestamp");
                    String datanodeId = rs.getString("datanode_id");
                    long containerId = rs.getLong("container_id");
                    long latestBcsid = rs.getLong("latest_bcsid");
                    String errorMessage = rs.getString("error_message");
                    int indexValue = rs.getInt("index_value");
                    ++count;
                    this.out.printf("%-25s | %-35s | %-15d | %-15d | %-40s | %-12d%n", timestamp, datanodeId, containerId, latestBcsid, errorMessage, indexValue);
                }
                if (count == 0) {
                    this.out.printf("No containers found for state: %s%n", state);
                } else {
                    this.out.printf("Number of containers listed: %d%n", count);
                }
            }
        }
        catch (SQLException e) {
            throw new SQLException("Error while retrieving containers with state " + state);
        }
        catch (Exception e) {
            throw new RuntimeException("Unexpected error: " + e);
        }
    }

    public void showContainerDetails(Long containerID) throws SQLException {
        try (Connection connection = this.getConnection();){
            List<DatanodeContainerInfo> logEntries = this.getContainerLogData(containerID, connection);
            if (logEntries.isEmpty()) {
                this.out.println("Missing container with ID: " + containerID);
                return;
            }
            this.out.printf("%-25s | %-15s | %-35s | %-20s | %-10s | %-30s | %-12s%n", "Timestamp", "Container ID", "Datanode ID", "Container State", "BCSID", "Message", "Index Value");
            this.out.println("------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------");
            for (DatanodeContainerInfo entry : logEntries) {
                this.out.printf("%-25s | %-15d | %-35s | %-20s | %-10d | %-30s | %-12d%n", entry.getTimestamp(), entry.getContainerId(), entry.getDatanodeId(), entry.getState(), entry.getBcsid(), entry.getErrorMessage(), entry.getIndexValue());
            }
            logEntries.sort(Comparator.comparing(DatanodeContainerInfo::getTimestamp));
            if (this.checkForMultipleOpenStates(logEntries)) {
                this.out.println("Container " + containerID + " might have duplicate OPEN state.");
                return;
            }
            HashMap<String, DatanodeContainerInfo> latestPerDatanode = new HashMap<String, DatanodeContainerInfo>();
            for (DatanodeContainerInfo entry : logEntries) {
                String datanodeId = entry.getDatanodeId();
                DatanodeContainerInfo existing = (DatanodeContainerInfo)latestPerDatanode.get(datanodeId);
                if (existing != null && entry.getTimestamp().compareTo(existing.getTimestamp()) <= 0) continue;
                latestPerDatanode.put(datanodeId, entry);
            }
            this.analyzeContainerHealth(containerID, latestPerDatanode);
        }
        catch (SQLException e) {
            throw new SQLException("Error while retrieving container with ID " + containerID);
        }
        catch (Exception e) {
            throw new RuntimeException("Unexpected error: " + e);
        }
    }

    private void analyzeContainerHealth(Long containerID, Map<String, DatanodeContainerInfo> latestPerDatanode) {
        boolean allClosedNewer;
        HashSet<String> lifeCycleStates = new HashSet<String>();
        for (HddsProtos.LifeCycleState state : HddsProtos.LifeCycleState.values()) {
            lifeCycleStates.add(state.name());
        }
        HashSet<String> healthStates = new HashSet<String>();
        for (ReplicationManagerReport.HealthState state : ReplicationManagerReport.HealthState.values()) {
            healthStates.add(state.name());
        }
        HashSet<String> unhealthyReplicas = new HashSet<String>();
        HashSet<String> closedReplicas = new HashSet<String>();
        HashSet<String> openReplicas = new HashSet<String>();
        HashSet<String> quasiclosedReplicas = new HashSet<String>();
        HashSet<String> deletedReplicas = new HashSet<String>();
        HashSet<Long> bcsids = new HashSet<Long>();
        HashSet<String> datanodeIds = new HashSet<String>();
        ArrayList<String> closedTimestamps = new ArrayList<String>();
        ArrayList<String> otherTimestamps = new ArrayList<String>();
        for (DatanodeContainerInfo entry : latestPerDatanode.values()) {
            String datanodeId = entry.getDatanodeId();
            String state = entry.getState();
            long bcsid = entry.getBcsid();
            String stateTimestamp = entry.getTimestamp();
            datanodeIds.add(datanodeId);
            if (healthStates.contains(state.toUpperCase())) {
                ReplicationManagerReport.HealthState healthState = ReplicationManagerReport.HealthState.valueOf((String)state.toUpperCase());
                if (healthState != ReplicationManagerReport.HealthState.UNHEALTHY) continue;
                unhealthyReplicas.add(datanodeId);
                continue;
            }
            if (!lifeCycleStates.contains(state.toUpperCase())) continue;
            HddsProtos.LifeCycleState lifeCycleState = HddsProtos.LifeCycleState.valueOf((String)state.toUpperCase());
            switch (lifeCycleState) {
                case OPEN: {
                    openReplicas.add(datanodeId);
                    otherTimestamps.add(stateTimestamp);
                    break;
                }
                case CLOSING: {
                    otherTimestamps.add(stateTimestamp);
                    break;
                }
                case CLOSED: {
                    closedReplicas.add(datanodeId);
                    bcsids.add(bcsid);
                    closedTimestamps.add(stateTimestamp);
                    break;
                }
                case QUASI_CLOSED: {
                    quasiclosedReplicas.add(datanodeId);
                    otherTimestamps.add(stateTimestamp);
                    break;
                }
                case DELETED: {
                    deletedReplicas.add(datanodeId);
                    break;
                }
            }
        }
        int closedCount = closedReplicas.size();
        boolean bl = allClosedNewer = closedCount > 0 && closedTimestamps.stream().allMatch(ct -> otherTimestamps.stream().allMatch(ot -> ct.compareTo((String)ot) > 0));
        if (bcsids.size() > 1) {
            this.out.println("Container " + containerID + " has MISMATCHED REPLICATION as there are multiple CLOSED containers with varying BCSIDs.");
        } else if (closedCount == DEFAULT_REPLICATION_FACTOR && allClosedNewer) {
            this.out.println("Container " + containerID + " has enough replicas.");
        } else if (closedCount > DEFAULT_REPLICATION_FACTOR && allClosedNewer) {
            this.out.println("Container " + containerID + " is OVER-REPLICATED.");
        } else if (closedCount < DEFAULT_REPLICATION_FACTOR && closedCount != 0 && allClosedNewer) {
            this.out.println("Container " + containerID + " is UNDER-REPLICATED.");
        } else {
            int replicaCount = datanodeIds.size();
            if (!quasiclosedReplicas.isEmpty() && closedReplicas.isEmpty() && quasiclosedReplicas.size() >= DEFAULT_REPLICATION_FACTOR) {
                this.out.println("Container " + containerID + " might be QUASI_CLOSED_STUCK.");
            } else if (!unhealthyReplicas.isEmpty()) {
                this.out.println("Container " + containerID + " has UNHEALTHY replicas.");
            } else if (!openReplicas.isEmpty() && replicaCount - openReplicas.size() > 0) {
                this.out.println("Container " + containerID + " might be OPEN_UNHEALTHY.");
            } else if (replicaCount - deletedReplicas.size() < DEFAULT_REPLICATION_FACTOR) {
                this.out.println("Container " + containerID + " is UNDER-REPLICATED.");
            } else if (replicaCount - deletedReplicas.size() > DEFAULT_REPLICATION_FACTOR) {
                this.out.println("Container " + containerID + " is OVER-REPLICATED.");
            } else {
                this.out.println("Container " + containerID + " has enough replicas.");
            }
        }
    }

    private boolean checkForMultipleOpenStates(List<DatanodeContainerInfo> entries) {
        ArrayList<String> firstOpenTimestamps = new ArrayList<String>();
        HashSet<String> firstOpenDatanodes = new HashSet<String>();
        boolean issueFound = false;
        for (DatanodeContainerInfo entry : entries) {
            if (!"OPEN".equalsIgnoreCase(entry.getState())) continue;
            if (firstOpenTimestamps.size() < DEFAULT_REPLICATION_FACTOR && !firstOpenDatanodes.contains(entry.getDatanodeId())) {
                firstOpenTimestamps.add(entry.getTimestamp());
                firstOpenDatanodes.add(entry.getDatanodeId());
                continue;
            }
            if (!this.isTimestampAfterFirstOpens(entry.getTimestamp(), firstOpenTimestamps)) continue;
            issueFound = true;
        }
        return issueFound;
    }

    private boolean isTimestampAfterFirstOpens(String timestamp, List<String> firstOpenTimestamps) {
        for (String firstTimestamp : firstOpenTimestamps) {
            if (timestamp.compareTo(firstTimestamp) <= 0) continue;
            return true;
        }
        return false;
    }

    private List<DatanodeContainerInfo> getContainerLogData(Long containerID, Connection connection) throws SQLException {
        String query = "SELECT d.timestamp, d.container_id, d.datanode_id, d.container_state, d.bcsid, d.error_message, d.index_value FROM DatanodeContainerLogTable d WHERE d.container_id = ? ORDER BY d.datanode_id ASC, d.timestamp ASC;";
        ArrayList<DatanodeContainerInfo> logEntries = new ArrayList<DatanodeContainerInfo>();
        try (PreparedStatement preparedStatement = connection.prepareStatement(query);){
            preparedStatement.setLong(1, containerID);
            try (ResultSet rs = preparedStatement.executeQuery();){
                while (rs.next()) {
                    DatanodeContainerInfo entry = new DatanodeContainerInfo.Builder().setTimestamp(rs.getString("timestamp")).setContainerId(rs.getLong("container_id")).setDatanodeId(rs.getString("datanode_id")).setState(rs.getString("container_state")).setBcsid(rs.getLong("bcsid")).setErrorMessage(rs.getString("error_message")).setIndexValue(rs.getInt("index_value")).build();
                    logEntries.add(entry);
                }
            }
        }
        return logEntries;
    }

    public void findDuplicateOpenContainer() throws SQLException {
        String sql = "SELECT DISTINCT container_id FROM ContainerLogTable";
        try (Connection connection = this.getConnection();
             PreparedStatement statement = connection.prepareStatement(sql);
             ResultSet resultSet = statement.executeQuery();){
            int count = 0;
            while (resultSet.next()) {
                Long containerID = resultSet.getLong("container_id");
                List<DatanodeContainerInfo> logEntries = this.getContainerLogDataForOpenContainers(containerID, connection);
                boolean hasIssue = this.checkForMultipleOpenStates(logEntries);
                if (!hasIssue) continue;
                int openStateCount = (int)logEntries.stream().filter(entry -> "OPEN".equalsIgnoreCase(entry.getState())).count();
                ++count;
                this.out.println("Container ID: " + containerID + " - OPEN state count: " + openStateCount);
            }
            this.out.println("Total containers that might have duplicate OPEN state : " + count);
        }
        catch (SQLException e) {
            throw new SQLException("Error while retrieving containers." + e.getMessage(), e);
        }
        catch (Exception e) {
            throw new RuntimeException("Unexpected error: " + e);
        }
    }

    private List<DatanodeContainerInfo> getContainerLogDataForOpenContainers(Long containerID, Connection connection) throws SQLException {
        String query = "SELECT d.timestamp, d.container_id, d.datanode_id, d.container_state FROM DatanodeContainerLogTable d WHERE d.container_id = ? AND d.container_state = 'OPEN' ORDER BY d.timestamp ASC;";
        ArrayList<DatanodeContainerInfo> logEntries = new ArrayList<DatanodeContainerInfo>();
        try (PreparedStatement preparedStatement = connection.prepareStatement(query);){
            preparedStatement.setLong(1, containerID);
            try (ResultSet rs = preparedStatement.executeQuery();){
                while (rs.next()) {
                    DatanodeContainerInfo entry = new DatanodeContainerInfo.Builder().setTimestamp(rs.getString("timestamp")).setContainerId(rs.getLong("container_id")).setDatanodeId(rs.getString("datanode_id")).setState(rs.getString("container_state")).build();
                    logEntries.add(entry);
                }
            }
        }
        return logEntries;
    }

    public void listReplicatedContainers(String overOrUnder, Integer limit) throws SQLException {
        boolean limitProvided;
        String operator;
        if ("OVER_REPLICATED".equalsIgnoreCase(overOrUnder)) {
            operator = ">";
        } else if ("UNDER_REPLICATED".equalsIgnoreCase(overOrUnder)) {
            operator = "<";
        } else {
            this.err.println("Invalid type. Use OVER_REPLICATED or UNDER_REPLICATED.");
            return;
        }
        String rawQuery = SQLDBConstants.SELECT_REPLICATED_CONTAINERS;
        if (!rawQuery.contains("{operator}")) {
            this.err.println("Query not defined correctly.");
            return;
        }
        String finalQuery = rawQuery.replace("{operator}", operator);
        boolean bl = limitProvided = limit != Integer.MAX_VALUE;
        if (limitProvided) {
            finalQuery = finalQuery + " LIMIT ?";
        }
        try (Connection connection = this.getConnection();
             PreparedStatement pstmt = connection.prepareStatement(finalQuery);){
            pstmt.setInt(1, DEFAULT_REPLICATION_FACTOR);
            if (limitProvided) {
                pstmt.setInt(2, limit + 1);
            }
            try (ResultSet rs = pstmt.executeQuery();){
                int count = 0;
                while (rs.next()) {
                    if (limitProvided && count >= limit) {
                        this.err.println("Note: There might be more containers. Use --all option to list all entries.");
                        break;
                    }
                    this.out.printf("Container ID = %s - Count = %d%n", rs.getLong("container_id"), rs.getInt("replica_count"));
                    ++count;
                }
                this.out.println("Number of containers listed: " + count);
            }
        }
        catch (SQLException e) {
            throw new SQLException("Error while retrieving containers." + e.getMessage(), e);
        }
        catch (Exception e) {
            throw new RuntimeException("Unexpected error: " + e);
        }
    }

    public void listUnhealthyContainers(Integer limit) throws SQLException {
        boolean limitProvided;
        String query = SQLDBConstants.SELECT_UNHEALTHY_CONTAINERS;
        boolean bl = limitProvided = limit != Integer.MAX_VALUE;
        if (limitProvided) {
            query = query + " LIMIT ?";
        }
        try (Connection connection = this.getConnection();
             PreparedStatement stmt = connection.prepareStatement(query);){
            if (limitProvided) {
                stmt.setInt(1, limit + 1);
            }
            try (ResultSet rs = stmt.executeQuery();){
                int count = 0;
                while (rs.next()) {
                    if (limitProvided && count >= limit) {
                        this.err.println("Note: There might be more containers. Use --all option to list all entries.");
                        break;
                    }
                    this.out.printf("Container ID = %s - Count = %d%n", rs.getString("container_id"), rs.getInt("unhealthy_replica_count"));
                    ++count;
                }
                this.out.println("Number of containers listed: " + count);
            }
        }
        catch (SQLException e) {
            throw new SQLException("Error while retrieving containers." + e.getMessage(), e);
        }
        catch (Exception e) {
            throw new RuntimeException("Unexpected error: " + e);
        }
    }

    public void listQuasiClosedStuckContainers(Integer limit) throws SQLException {
        boolean limitProvided;
        String query = SQLDBConstants.SELECT_QUASI_CLOSED_STUCK_CONTAINERS;
        boolean bl = limitProvided = limit != Integer.MAX_VALUE;
        if (limitProvided) {
            query = query + " LIMIT ?";
        }
        try (Connection connection = this.getConnection();
             PreparedStatement statement = connection.prepareStatement(query);){
            if (limitProvided) {
                statement.setInt(1, limit + 1);
            }
            try (ResultSet resultSet = statement.executeQuery();){
                int count = 0;
                while (resultSet.next()) {
                    if (limitProvided && count >= limit) {
                        this.err.println("Note: There might be more containers. Use --all option to list all entries.");
                        break;
                    }
                    this.out.printf("Container ID = %s - Count = %d%n", resultSet.getString("container_id"), resultSet.getInt("quasi_closed_replica_count"));
                    ++count;
                }
                this.out.println("Number of containers listed: " + count);
            }
        }
        catch (SQLException e) {
            throw new SQLException("Error while retrieving containers." + e.getMessage(), e);
        }
        catch (Exception e) {
            throw new RuntimeException("Unexpected error: " + e);
        }
    }

    static {
        OzoneConfiguration configuration = new OzoneConfiguration();
        String replication = configuration.getTrimmed("ozone.server.default.replication", OMConfigKeys.OZONE_SERVER_DEFAULT_REPLICATION_DEFAULT);
        DEFAULT_REPLICATION_FACTOR = Integer.parseInt(replication.toUpperCase());
    }
}

