/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.recon.fsck;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.PlacementPolicy;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.ContainerManager;
import org.apache.hadoop.hdds.scm.container.ContainerNotFoundException;
import org.apache.hadoop.hdds.scm.container.common.helpers.ContainerWithPipeline;
import org.apache.hadoop.ozone.common.statemachine.InvalidStateTransitionException;
import org.apache.hadoop.ozone.recon.fsck.ContainerHealthStatus;
import org.apache.hadoop.ozone.recon.metrics.ContainerHealthMetrics;
import org.apache.hadoop.ozone.recon.persistence.ContainerHealthSchemaManager;
import org.apache.hadoop.ozone.recon.scm.ReconScmTask;
import org.apache.hadoop.ozone.recon.spi.ReconContainerMetadataManager;
import org.apache.hadoop.ozone.recon.spi.StorageContainerServiceProvider;
import org.apache.hadoop.ozone.recon.tasks.ReconTaskConfig;
import org.apache.hadoop.ozone.recon.tasks.updater.ReconTaskStatusUpdater;
import org.apache.hadoop.ozone.recon.tasks.updater.ReconTaskStatusUpdaterManager;
import org.apache.hadoop.util.Time;
import org.apache.ozone.recon.schema.ContainerSchemaDefinition;
import org.apache.ozone.recon.schema.generated.tables.pojos.UnhealthyContainers;
import org.apache.ozone.recon.schema.generated.tables.records.UnhealthyContainersRecord;
import org.jooq.Cursor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerHealthTask
extends ReconScmTask {
    private static final Logger LOG = LoggerFactory.getLogger(ContainerHealthTask.class);
    public static final int FETCH_COUNT = Integer.parseInt("1000");
    private final ReadWriteLock lock = new ReentrantReadWriteLock(true);
    private final StorageContainerServiceProvider scmClient;
    private final ContainerManager containerManager;
    private final ContainerHealthSchemaManager containerHealthSchemaManager;
    private final ReconContainerMetadataManager reconContainerMetadataManager;
    private final PlacementPolicy placementPolicy;
    private final long interval;
    private Map<ContainerSchemaDefinition.UnHealthyContainerStates, Map<String, Long>> unhealthyContainerStateStatsMapForTesting;
    private final Set<ContainerInfo> processedContainers = new HashSet<ContainerInfo>();
    private final OzoneConfiguration conf;
    private final ReconTaskStatusUpdater taskStatusUpdater;
    private final ContainerHealthMetrics containerHealthMetrics;

    public ContainerHealthTask(ContainerManager containerManager, StorageContainerServiceProvider scmClient, ContainerHealthSchemaManager containerHealthSchemaManager, PlacementPolicy placementPolicy, ReconTaskConfig reconTaskConfig, ReconContainerMetadataManager reconContainerMetadataManager, OzoneConfiguration conf, ReconTaskStatusUpdaterManager taskStatusUpdaterManager) {
        super(taskStatusUpdaterManager);
        this.scmClient = scmClient;
        this.containerHealthSchemaManager = containerHealthSchemaManager;
        this.reconContainerMetadataManager = reconContainerMetadataManager;
        this.placementPolicy = placementPolicy;
        this.containerManager = containerManager;
        this.conf = conf;
        this.interval = reconTaskConfig.getMissingContainerTaskInterval().toMillis();
        this.taskStatusUpdater = this.getTaskStatusUpdater();
        this.containerHealthMetrics = ContainerHealthMetrics.create();
    }

    @Override
    public void run() {
        try {
            while (this.canRun()) {
                this.initializeAndRunTask();
                Thread.sleep(this.interval);
            }
        }
        catch (Throwable t) {
            LOG.error("Exception in Container Health task thread.", t);
            if (t instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            this.taskStatusUpdater.setLastTaskRunStatus(-1);
            this.taskStatusUpdater.recordRunCompletion();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected void runTask() throws Exception {
        this.lock.writeLock().lock();
        try {
            HashMap<ContainerSchemaDefinition.UnHealthyContainerStates, Map<String, Long>> unhealthyContainerStateStatsMap = new HashMap<ContainerSchemaDefinition.UnHealthyContainerStates, Map<String, Long>>(Collections.emptyMap());
            this.initializeUnhealthyContainerStateStatsMap(unhealthyContainerStateStatsMap);
            long start = Time.monotonicNow();
            long currentTime = System.currentTimeMillis();
            long existingCount = this.processExistingDBRecords(currentTime, unhealthyContainerStateStatsMap);
            LOG.debug("Container Health task thread took {} milliseconds to process {} existing database records.", (Object)(Time.monotonicNow() - start), (Object)existingCount);
            start = Time.monotonicNow();
            this.checkAndProcessContainers(unhealthyContainerStateStatsMap, currentTime);
            LOG.debug("Container Health Task thread took {} milliseconds to process containers", (Object)(Time.monotonicNow() - start));
            this.taskStatusUpdater.setLastTaskRunStatus(0);
            this.processedContainers.clear();
            this.logUnhealthyContainerStats(unhealthyContainerStateStatsMap);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    private void checkAndProcessContainers(Map<ContainerSchemaDefinition.UnHealthyContainerStates, Map<String, Long>> unhealthyContainerStateStatsMap, long currentTime) {
        ContainerID startID = ContainerID.valueOf((long)1L);
        List containers = this.containerManager.getContainers(startID, FETCH_COUNT);
        long iterationCount = 0L;
        while (!containers.isEmpty()) {
            long start = Time.monotonicNow();
            containers.stream().filter(c -> !this.processedContainers.contains(c)).forEach(c -> this.processContainer((ContainerInfo)c, currentTime, unhealthyContainerStateStatsMap));
            LOG.debug("Container Health task thread took {} milliseconds for processing {} containers.", (Object)(Time.monotonicNow() - start), (Object)containers.size());
            if (containers.size() >= FETCH_COUNT) {
                startID = ContainerID.valueOf((long)(((ContainerInfo)containers.get(containers.size() - 1)).getContainerID() + 1L));
                containers = this.containerManager.getContainers(startID, FETCH_COUNT);
            } else {
                containers.clear();
            }
            ++iterationCount;
        }
        LOG.info("Container Health task thread took {} iterations to fetch all containers using batched approach with batch size of {}", (Object)iterationCount, (Object)FETCH_COUNT);
    }

    private void logUnhealthyContainerStats(Map<ContainerSchemaDefinition.UnHealthyContainerStates, Map<String, Long>> unhealthyContainerStateStatsMap) {
        this.unhealthyContainerStateStatsMapForTesting = new HashMap<ContainerSchemaDefinition.UnHealthyContainerStates, Map<String, Long>>(unhealthyContainerStateStatsMap);
        unhealthyContainerStateStatsMap.forEach((unhealthyContainerState, containerStateStatsMap) -> {
            Optional.of(containerStateStatsMap).filter(Map::isEmpty).ifPresent(emptyMap -> this.resetContainerHealthMetrics((ContainerSchemaDefinition.UnHealthyContainerStates)unhealthyContainerState));
            String logMessage = containerStateStatsMap.entrySet().stream().peek(entry -> this.updateContainerHealthMetrics((ContainerSchemaDefinition.UnHealthyContainerStates)unhealthyContainerState, (Map.Entry<String, Long>)entry)).map(entry -> (String)entry.getKey() + " -> " + entry.getValue()).collect(Collectors.joining(" , ", unhealthyContainerState + " **Container State Stats:** \n\t", ""));
            if (!containerStateStatsMap.isEmpty()) {
                LOG.info(logMessage);
            }
        });
    }

    private void updateContainerHealthMetrics(ContainerSchemaDefinition.UnHealthyContainerStates state, Map.Entry<String, Long> entry) {
        HashMap<ContainerSchemaDefinition.UnHealthyContainerStates, Consumer<Long>> metricUpdaters = new HashMap<ContainerSchemaDefinition.UnHealthyContainerStates, Consumer<Long>>();
        metricUpdaters.put(ContainerSchemaDefinition.UnHealthyContainerStates.MISSING, this.containerHealthMetrics::setMissingContainerCount);
        metricUpdaters.put(ContainerSchemaDefinition.UnHealthyContainerStates.UNDER_REPLICATED, this.containerHealthMetrics::setUnderReplicatedContainerCount);
        Optional.ofNullable((Consumer)metricUpdaters.get(state)).filter(updater -> "CONTAINER_COUNT".equals(entry.getKey())).ifPresent(updater -> updater.accept((Long)entry.getValue()));
    }

    private void resetContainerHealthMetrics(ContainerSchemaDefinition.UnHealthyContainerStates state) {
        HashMap<ContainerSchemaDefinition.UnHealthyContainerStates, Consumer<Long>> resetActions = new HashMap<ContainerSchemaDefinition.UnHealthyContainerStates, Consumer<Long>>();
        resetActions.put(ContainerSchemaDefinition.UnHealthyContainerStates.MISSING, this.containerHealthMetrics::setMissingContainerCount);
        resetActions.put(ContainerSchemaDefinition.UnHealthyContainerStates.UNDER_REPLICATED, this.containerHealthMetrics::setUnderReplicatedContainerCount);
        Optional.ofNullable((Consumer)resetActions.get(state)).ifPresent(action -> action.accept(0L));
    }

    private void initializeUnhealthyContainerStateStatsMap(Map<ContainerSchemaDefinition.UnHealthyContainerStates, Map<String, Long>> unhealthyContainerStateStatsMap) {
        unhealthyContainerStateStatsMap.put(ContainerSchemaDefinition.UnHealthyContainerStates.MISSING, new HashMap());
        unhealthyContainerStateStatsMap.put(ContainerSchemaDefinition.UnHealthyContainerStates.EMPTY_MISSING, new HashMap());
        unhealthyContainerStateStatsMap.put(ContainerSchemaDefinition.UnHealthyContainerStates.UNDER_REPLICATED, new HashMap());
        unhealthyContainerStateStatsMap.put(ContainerSchemaDefinition.UnHealthyContainerStates.OVER_REPLICATED, new HashMap());
        unhealthyContainerStateStatsMap.put(ContainerSchemaDefinition.UnHealthyContainerStates.MIS_REPLICATED, new HashMap());
        unhealthyContainerStateStatsMap.put(ContainerSchemaDefinition.UnHealthyContainerStates.NEGATIVE_SIZE, new HashMap());
        unhealthyContainerStateStatsMap.put(ContainerSchemaDefinition.UnHealthyContainerStates.REPLICA_MISMATCH, new HashMap());
    }

    private ContainerHealthStatus setCurrentContainer(long recordId) throws ContainerNotFoundException {
        ContainerInfo container = this.containerManager.getContainer(ContainerID.valueOf((long)recordId));
        Set replicas = this.containerManager.getContainerReplicas(container.containerID());
        return new ContainerHealthStatus(container, replicas, this.placementPolicy, this.reconContainerMetadataManager, this.conf);
    }

    private void completeProcessingContainer(ContainerHealthStatus container, Set<String> existingRecords, long currentTime, Map<ContainerSchemaDefinition.UnHealthyContainerStates, Map<String, Long>> unhealthyContainerStateCountMap) {
        this.containerHealthSchemaManager.insertUnhealthyContainerRecords(ContainerHealthRecords.generateUnhealthyRecords(container, existingRecords, currentTime, unhealthyContainerStateCountMap));
        this.processedContainers.add(container.getContainer());
    }

    private long processExistingDBRecords(long currentTime, Map<ContainerSchemaDefinition.UnHealthyContainerStates, Map<String, Long>> unhealthyContainerStateCountMap) {
        long recordCount = 0L;
        try (Cursor<UnhealthyContainersRecord> cursor = this.containerHealthSchemaManager.getAllUnhealthyRecordsCursor();){
            ContainerHealthStatus currentContainer = null;
            HashSet<String> existingRecords = new HashSet<String>();
            while (cursor.hasNext()) {
                ++recordCount;
                UnhealthyContainersRecord rec = (UnhealthyContainersRecord)cursor.fetchNext();
                try {
                    if (currentContainer == null) {
                        currentContainer = this.setCurrentContainer(rec.getContainerId());
                    }
                    if (currentContainer.getContainerID() != rec.getContainerId().longValue()) {
                        this.completeProcessingContainer(currentContainer, existingRecords, currentTime, unhealthyContainerStateCountMap);
                        existingRecords.clear();
                        currentContainer = this.setCurrentContainer(rec.getContainerId());
                    }
                    if (!ContainerHealthRecords.retainOrUpdateRecord(currentContainer, rec)) {
                        rec.delete();
                        LOG.info("DELETED existing unhealthy container record...for Container: {}", (Object)currentContainer.getContainerID());
                    }
                    if (currentContainer.isMissing() && this.containerDeletedInSCM(currentContainer.getContainer())) {
                        rec.delete();
                        LOG.info("DELETED existing MISSING unhealthy container record...as container deleted in SCM as well: {}", (Object)currentContainer.getContainerID());
                    }
                    existingRecords.add(rec.getContainerState());
                    if (!rec.changed()) continue;
                    rec.update();
                }
                catch (ContainerNotFoundException cnf) {
                    rec.delete();
                    currentContainer = null;
                }
            }
            if (currentContainer != null) {
                this.completeProcessingContainer(currentContainer, existingRecords, currentTime, unhealthyContainerStateCountMap);
            }
        }
        return recordCount;
    }

    private void processContainer(ContainerInfo container, long currentTime, Map<ContainerSchemaDefinition.UnHealthyContainerStates, Map<String, Long>> unhealthyContainerStateStatsMap) {
        try {
            Set containerReplicas = this.containerManager.getContainerReplicas(container.containerID());
            ContainerHealthStatus h = new ContainerHealthStatus(container, containerReplicas, this.placementPolicy, this.reconContainerMetadataManager, this.conf);
            if (h.isHealthilyReplicated() && !h.isDataChecksumMismatched() || h.isDeleted()) {
                return;
            }
            if (h.isMissing() && this.containerDeletedInSCM(container)) {
                return;
            }
            this.containerHealthSchemaManager.insertUnhealthyContainerRecords(ContainerHealthRecords.generateUnhealthyRecords(h, currentTime, unhealthyContainerStateStatsMap));
        }
        catch (ContainerNotFoundException e) {
            LOG.error("Container not found while processing container in Container Health task", (Throwable)e);
        }
    }

    private boolean containerDeletedInSCM(ContainerInfo containerInfo) {
        try {
            ContainerWithPipeline containerWithPipeline = this.scmClient.getContainerWithPipeline(containerInfo.getContainerID());
            if (containerWithPipeline.getContainerInfo().getState() == HddsProtos.LifeCycleState.DELETED) {
                if (containerInfo.getState() == HddsProtos.LifeCycleState.CLOSED) {
                    this.containerManager.updateContainerState(containerInfo.containerID(), HddsProtos.LifeCycleEvent.DELETE);
                    LOG.debug("Successfully changed container {} state from CLOSED to DELETING.", (Object)containerInfo.containerID());
                }
                if (containerInfo.getState() == HddsProtos.LifeCycleState.DELETING && this.containerManager.getContainerReplicas(containerInfo.containerID()).isEmpty()) {
                    this.containerManager.updateContainerState(containerInfo.containerID(), HddsProtos.LifeCycleEvent.CLEANUP);
                    LOG.info("Successfully Deleted container {} from Recon.", (Object)containerInfo.containerID());
                }
                return true;
            }
        }
        catch (InvalidStateTransitionException e) {
            LOG.error("Failed to transition Container state while processing container in Container Health task", (Throwable)e);
        }
        catch (IOException e) {
            LOG.error("Got exception while processing container in Container Health task", (Throwable)e);
        }
        return false;
    }

    private static void handleNegativeSizedContainers(ContainerHealthStatus containerHealthStatus, long currentTime, Map<ContainerSchemaDefinition.UnHealthyContainerStates, Map<String, Long>> unhealthyContainerStateStatsMap) {
        ContainerInfo container = containerHealthStatus.getContainer();
        LOG.error("Container {} has negative size.", (Object)container.getContainerID());
        ContainerHealthTask.populateContainerStats(containerHealthStatus, ContainerSchemaDefinition.UnHealthyContainerStates.NEGATIVE_SIZE, unhealthyContainerStateStatsMap);
    }

    private static void handleEmptyMissingContainers(ContainerHealthStatus containerHealthStatus, long currentTime, Map<ContainerSchemaDefinition.UnHealthyContainerStates, Map<String, Long>> unhealthyContainerStateStatsMap) {
        ContainerInfo container = containerHealthStatus.getContainer();
        LOG.debug("Empty container {} is missing. It will be logged in the unhealthy container statistics, but no record will be created in the UNHEALTHY_CONTAINERS table.", (Object)container.getContainerID());
        ContainerHealthTask.populateContainerStats(containerHealthStatus, ContainerSchemaDefinition.UnHealthyContainerStates.EMPTY_MISSING, unhealthyContainerStateStatsMap);
    }

    private static void populateContainerStats(ContainerHealthStatus container, ContainerSchemaDefinition.UnHealthyContainerStates unhealthyState, Map<ContainerSchemaDefinition.UnHealthyContainerStates, Map<String, Long>> unhealthyContainerStateStatsMap) {
        if (unhealthyContainerStateStatsMap.containsKey(unhealthyState)) {
            Map<String, Long> containerStatsMap = unhealthyContainerStateStatsMap.get(unhealthyState);
            containerStatsMap.compute("CONTAINER_COUNT", (containerCount, value) -> value == null ? 1L : value + 1L);
            containerStatsMap.compute("TOTAL_KEYS", (totalKeyCount, value) -> value == null ? container.getNumKeys() : value + container.getNumKeys());
            containerStatsMap.compute("TOTAL_USED_BYTES", (totalUsedBytes, value) -> value == null ? container.getContainer().getUsedBytes() : value + container.getContainer().getUsedBytes());
        }
    }

    @Override
    public synchronized void stop() {
        super.stop();
        this.containerHealthMetrics.unRegister();
    }

    @VisibleForTesting
    public Map<ContainerSchemaDefinition.UnHealthyContainerStates, Map<String, Long>> getUnhealthyContainerStateStatsMap() {
        return this.unhealthyContainerStateStatsMapForTesting;
    }

    public static class ContainerHealthRecords {
        public static boolean retainOrUpdateRecord(ContainerHealthStatus container, UnhealthyContainersRecord rec) {
            boolean returnValue;
            switch (ContainerSchemaDefinition.UnHealthyContainerStates.valueOf((String)rec.getContainerState())) {
                case MISSING: {
                    returnValue = container.isMissing() && !container.isEmpty();
                    break;
                }
                case MIS_REPLICATED: {
                    returnValue = ContainerHealthRecords.keepMisReplicatedRecord(container, rec);
                    break;
                }
                case UNDER_REPLICATED: {
                    returnValue = ContainerHealthRecords.keepUnderReplicatedRecord(container, rec);
                    break;
                }
                case OVER_REPLICATED: {
                    returnValue = ContainerHealthRecords.keepOverReplicatedRecord(container, rec);
                    break;
                }
                case REPLICA_MISMATCH: {
                    returnValue = ContainerHealthRecords.keepReplicaMismatchRecord(container, rec);
                    break;
                }
                default: {
                    returnValue = false;
                }
            }
            return returnValue;
        }

        public static List<UnhealthyContainers> generateUnhealthyRecords(ContainerHealthStatus container, long time, Map<ContainerSchemaDefinition.UnHealthyContainerStates, Map<String, Long>> unhealthyContainerStateStatsMap) {
            return ContainerHealthRecords.generateUnhealthyRecords(container, new HashSet<String>(), time, unhealthyContainerStateStatsMap);
        }

        public static List<UnhealthyContainers> generateUnhealthyRecords(ContainerHealthStatus container, Set<String> recordForStateExists, long time, Map<ContainerSchemaDefinition.UnHealthyContainerStates, Map<String, Long>> unhealthyContainerStateStatsMap) {
            boolean shouldAddRecord;
            ArrayList<UnhealthyContainers> records = new ArrayList<UnhealthyContainers>();
            if (container.isHealthilyReplicated() && !container.isDataChecksumMismatched() || container.isDeleted()) {
                return records;
            }
            if (container.isMissing()) {
                boolean shouldAddRecord2;
                boolean bl = shouldAddRecord2 = !recordForStateExists.contains(ContainerSchemaDefinition.UnHealthyContainerStates.MISSING.toString());
                if (!container.isEmpty()) {
                    LOG.info("Non-empty container {} is missing. It has {} keys and {} bytes used according to SCM metadata. Please visit Recon's missing container page for a list of keys (and their metadata) mapped to this container.", new Object[]{container.getContainerID(), container.getNumKeys(), container.getContainer().getUsedBytes()});
                    if (shouldAddRecord2) {
                        records.add(ContainerHealthRecords.recordForState(container, ContainerSchemaDefinition.UnHealthyContainerStates.MISSING, time));
                    }
                    ContainerHealthTask.populateContainerStats(container, ContainerSchemaDefinition.UnHealthyContainerStates.MISSING, unhealthyContainerStateStatsMap);
                } else {
                    ContainerHealthTask.handleEmptyMissingContainers(container, time, unhealthyContainerStateStatsMap);
                }
                return records;
            }
            if (container.getContainer().getUsedBytes() < 0L) {
                ContainerHealthTask.handleNegativeSizedContainers(container, time, unhealthyContainerStateStatsMap);
            }
            if (container.isUnderReplicated()) {
                boolean bl = shouldAddRecord = !recordForStateExists.contains(ContainerSchemaDefinition.UnHealthyContainerStates.UNDER_REPLICATED.toString());
                if (shouldAddRecord) {
                    records.add(ContainerHealthRecords.recordForState(container, ContainerSchemaDefinition.UnHealthyContainerStates.UNDER_REPLICATED, time));
                }
                ContainerHealthTask.populateContainerStats(container, ContainerSchemaDefinition.UnHealthyContainerStates.UNDER_REPLICATED, unhealthyContainerStateStatsMap);
            }
            if (container.isOverReplicated()) {
                boolean bl = shouldAddRecord = !recordForStateExists.contains(ContainerSchemaDefinition.UnHealthyContainerStates.OVER_REPLICATED.toString());
                if (shouldAddRecord) {
                    records.add(ContainerHealthRecords.recordForState(container, ContainerSchemaDefinition.UnHealthyContainerStates.OVER_REPLICATED, time));
                }
                ContainerHealthTask.populateContainerStats(container, ContainerSchemaDefinition.UnHealthyContainerStates.OVER_REPLICATED, unhealthyContainerStateStatsMap);
            }
            if (container.isDataChecksumMismatched() && !recordForStateExists.contains(ContainerSchemaDefinition.UnHealthyContainerStates.REPLICA_MISMATCH.toString())) {
                records.add(ContainerHealthRecords.recordForState(container, ContainerSchemaDefinition.UnHealthyContainerStates.REPLICA_MISMATCH, time));
                ContainerHealthTask.populateContainerStats(container, ContainerSchemaDefinition.UnHealthyContainerStates.REPLICA_MISMATCH, unhealthyContainerStateStatsMap);
            }
            if (container.isMisReplicated()) {
                boolean bl = shouldAddRecord = !recordForStateExists.contains(ContainerSchemaDefinition.UnHealthyContainerStates.MIS_REPLICATED.toString());
                if (shouldAddRecord) {
                    records.add(ContainerHealthRecords.recordForState(container, ContainerSchemaDefinition.UnHealthyContainerStates.MIS_REPLICATED, time));
                }
                ContainerHealthTask.populateContainerStats(container, ContainerSchemaDefinition.UnHealthyContainerStates.MIS_REPLICATED, unhealthyContainerStateStatsMap);
            }
            return records;
        }

        private static UnhealthyContainers recordForState(ContainerHealthStatus container, ContainerSchemaDefinition.UnHealthyContainerStates state, long time) {
            UnhealthyContainers rec = new UnhealthyContainers();
            rec.setContainerId(container.getContainerID());
            if (state == ContainerSchemaDefinition.UnHealthyContainerStates.MIS_REPLICATED) {
                rec.setExpectedReplicaCount(container.expectedPlacementCount());
                rec.setActualReplicaCount(container.actualPlacementCount());
                rec.setReplicaDelta(container.misReplicatedDelta());
                rec.setReason(container.misReplicatedReason());
            } else {
                rec.setExpectedReplicaCount(container.getReplicationFactor());
                rec.setActualReplicaCount(container.getReplicaCount());
                rec.setReplicaDelta(container.replicaDelta());
            }
            rec.setContainerState(state.toString());
            rec.setInStateSince(time);
            return rec;
        }

        private static boolean keepOverReplicatedRecord(ContainerHealthStatus container, UnhealthyContainersRecord rec) {
            if (container.isOverReplicated()) {
                ContainerHealthRecords.updateExpectedReplicaCount(rec, container.getReplicationFactor());
                ContainerHealthRecords.updateActualReplicaCount(rec, container.getReplicaCount());
                ContainerHealthRecords.updateReplicaDelta(rec, container.replicaDelta());
                return true;
            }
            return false;
        }

        private static boolean keepUnderReplicatedRecord(ContainerHealthStatus container, UnhealthyContainersRecord rec) {
            if (container.isUnderReplicated()) {
                ContainerHealthRecords.updateExpectedReplicaCount(rec, container.getReplicationFactor());
                ContainerHealthRecords.updateActualReplicaCount(rec, container.getReplicaCount());
                ContainerHealthRecords.updateReplicaDelta(rec, container.replicaDelta());
                return true;
            }
            return false;
        }

        private static boolean keepMisReplicatedRecord(ContainerHealthStatus container, UnhealthyContainersRecord rec) {
            if (container.isMisReplicated()) {
                ContainerHealthRecords.updateExpectedReplicaCount(rec, container.expectedPlacementCount());
                ContainerHealthRecords.updateActualReplicaCount(rec, container.actualPlacementCount());
                ContainerHealthRecords.updateReplicaDelta(rec, container.misReplicatedDelta());
                ContainerHealthRecords.updateReason(rec, container.misReplicatedReason());
                return true;
            }
            return false;
        }

        private static boolean keepReplicaMismatchRecord(ContainerHealthStatus container, UnhealthyContainersRecord rec) {
            if (container.isDataChecksumMismatched()) {
                ContainerHealthRecords.updateExpectedReplicaCount(rec, container.getReplicationFactor());
                ContainerHealthRecords.updateActualReplicaCount(rec, container.getReplicaCount());
                ContainerHealthRecords.updateReplicaDelta(rec, container.replicaDelta());
                return true;
            }
            return false;
        }

        private static void updateExpectedReplicaCount(UnhealthyContainersRecord rec, int expectedCount) {
            if (rec.getExpectedReplicaCount() != expectedCount) {
                rec.setExpectedReplicaCount(expectedCount);
            }
        }

        private static void updateActualReplicaCount(UnhealthyContainersRecord rec, int actualCount) {
            if (rec.getActualReplicaCount() != actualCount) {
                rec.setActualReplicaCount(actualCount);
            }
        }

        private static void updateReplicaDelta(UnhealthyContainersRecord rec, int delta) {
            if (rec.getReplicaDelta() != delta) {
                rec.setReplicaDelta(delta);
            }
        }

        private static void updateReason(UnhealthyContainersRecord rec, String reason) {
            if (!rec.getReason().equals(reason)) {
                rec.setReason(reason);
            }
        }
    }
}

