/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.common;

import java.io.IOException;
import java.nio.Buffer;
import java.nio.BufferOverflowException;
import java.nio.ByteBuffer;
import java.nio.channels.GatheringByteChannel;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import org.apache.hadoop.hdds.utils.db.CodecBuffer;
import org.apache.hadoop.ozone.common.ChunkBuffer;
import org.apache.hadoop.ozone.common.ChunkBufferToByteString;
import org.apache.hadoop.ozone.common.utils.BufferUtils;
import org.apache.hadoop.ozone.shaded.com.google.common.base.Preconditions;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;

final class IncrementalChunkBuffer
implements ChunkBuffer {
    private final int limit;
    private final int increment;
    private final int limitIndex;
    private final List<ByteBuffer> buffers;
    private final List<CodecBuffer> underlying;
    private final boolean isDuplicated;
    private int firstNonFullIndex = 0;

    IncrementalChunkBuffer(int limit, int increment, boolean isDuplicated) {
        Preconditions.checkArgument(limit >= 0);
        Preconditions.checkArgument(increment > 0);
        this.limit = limit;
        this.increment = increment;
        this.limitIndex = limit / increment;
        int size = this.limitIndex + (limit % increment == 0 ? 0 : 1);
        this.buffers = new ArrayList<ByteBuffer>(size);
        this.underlying = isDuplicated ? Collections.emptyList() : new ArrayList(size);
        this.isDuplicated = isDuplicated;
    }

    @Override
    public void close() {
        this.underlying.forEach(CodecBuffer::release);
        this.underlying.clear();
    }

    private int getBufferCapacityAtIndex(int i) {
        Preconditions.checkArgument(i >= 0);
        Preconditions.checkArgument(i <= this.limitIndex);
        return i < this.limitIndex ? this.increment : this.limit % this.increment;
    }

    private void assertInt(int expected, int computed, String name, int i) {
        ChunkBufferToByteString.assertInt(expected, computed, () -> this + ": Unexpected " + name + " at index " + i);
    }

    private ByteBuffer getAtIndex(int i) {
        ByteBuffer ith;
        Preconditions.checkArgument(i >= 0);
        Preconditions.checkArgument(i <= this.limitIndex);
        ByteBuffer byteBuffer = ith = i < this.buffers.size() ? this.buffers.get(i) : null;
        if (ith != null) {
            if (!this.isDuplicated) {
                this.assertInt(this.getBufferCapacityAtIndex(i), ith.capacity(), "capacity", i);
            } else if (i < this.limitIndex) {
                this.assertInt(this.increment, ith.capacity(), "capacity", i);
            } else if (i == this.limitIndex) {
                this.assertInt(this.getBufferCapacityAtIndex(i), ith.limit(), "capacity", i);
            } else {
                this.assertInt(0, ith.limit(), "capacity", i);
            }
        }
        return ith;
    }

    private ByteBuffer getAndAllocateAtIndex(int index) {
        int i;
        Preconditions.checkState(!this.isDuplicated, "Duplicated buffer is readonly.");
        Preconditions.checkArgument(index >= 0);
        if (this.limit % this.increment == 0) {
            Preconditions.checkArgument(index < this.limitIndex);
        } else {
            Preconditions.checkArgument(index <= this.limitIndex);
        }
        if (index < i) {
            return this.getAtIndex(index);
        }
        ByteBuffer b = null;
        for (i = this.buffers.size(); i <= index; ++i) {
            CodecBuffer c = CodecBuffer.allocateDirect(this.getBufferCapacityAtIndex(i));
            this.underlying.add(c);
            b = c.asWritableByteBuffer();
            this.buffers.add(b);
        }
        return b;
    }

    private ByteBuffer getAndAllocateAtPosition(int position) {
        Preconditions.checkArgument(position >= 0);
        Preconditions.checkArgument(position < this.limit);
        int i = position / this.increment;
        ByteBuffer ith = this.getAndAllocateAtIndex(i);
        this.assertInt(position % this.increment, ith.position(), "position", i);
        return ith;
    }

    private int firstNonFullIndex() {
        for (int i = this.firstNonFullIndex; i < this.buffers.size(); ++i) {
            if (this.getAtIndex(i).position() == this.increment) continue;
            this.firstNonFullIndex = i;
            return this.firstNonFullIndex;
        }
        this.firstNonFullIndex = this.buffers.size();
        return this.firstNonFullIndex;
    }

    @Override
    public int position() {
        int i = this.firstNonFullIndex();
        ByteBuffer ith = this.getAtIndex(i);
        int position = i * this.increment + Optional.ofNullable(ith).map(Buffer::position).orElse(0);
        assert (this.assertRemainingList(ith, i));
        return position;
    }

    private boolean assertRemainingList(ByteBuffer ith, int i) {
        if (ith != null) {
            ++i;
            while (i < this.buffers.size() && (ith = this.getAtIndex(i)) != null) {
                this.assertInt(0, ith.position(), "position", i);
                ++i;
            }
        }
        int j = i;
        ChunkBufferToByteString.assertInt(this.buffers.size(), i, () -> "i = " + j + " != buffers.size() = " + this.buffers.size());
        return true;
    }

    @Override
    public int remaining() {
        return this.limit - this.position();
    }

    @Override
    public int limit() {
        return this.limit;
    }

    @Override
    public ChunkBuffer rewind() {
        this.buffers.forEach(Buffer::rewind);
        this.firstNonFullIndex = 0;
        return this;
    }

    @Override
    public ChunkBuffer clear() {
        this.buffers.forEach(Buffer::clear);
        this.firstNonFullIndex = 0;
        return this;
    }

    @Override
    public ChunkBuffer put(ByteBuffer that) {
        if (that.remaining() > this.remaining()) {
            BufferOverflowException boe = new BufferOverflowException();
            boe.initCause(new IllegalArgumentException("Failed to put since that.remaining() = " + that.remaining() + " > this.remaining() = " + this.remaining()));
            throw boe;
        }
        int thatLimit = that.limit();
        int p = this.position();
        while (that.position() < thatLimit) {
            ByteBuffer b = this.getAndAllocateAtPosition(p);
            int min2 = Math.min(b.remaining(), thatLimit - that.position());
            that.limit(that.position() + min2);
            b.put(that);
            p += min2;
        }
        return this;
    }

    @Override
    public ChunkBuffer duplicate(int newPosition, int newLimit) {
        Preconditions.checkArgument(newPosition >= 0);
        Preconditions.checkArgument(newPosition <= newLimit);
        Preconditions.checkArgument(newLimit <= this.limit);
        IncrementalChunkBuffer duplicated = new IncrementalChunkBuffer(newLimit, this.increment, true);
        int pi = newPosition / this.increment;
        int pr = newPosition % this.increment;
        int li = newLimit / this.increment;
        int lr = newLimit % this.increment;
        int newSize = lr == 0 ? li : li + 1;
        for (int i = 0; i < newSize; ++i) {
            int pos;
            int n = i < pi ? this.increment : (pos = i == pi ? pr : 0);
            int lim = i < li ? this.increment : (i == li ? lr : 0);
            duplicated.buffers.add(this.duplicate(i, pos, lim));
        }
        return duplicated;
    }

    private ByteBuffer duplicate(int i, int pos, int lim) {
        ByteBuffer ith = this.getAtIndex(i);
        Objects.requireNonNull(ith, () -> "buffers[" + i + "] == null");
        ByteBuffer b = ith.duplicate();
        b.position(pos).limit(lim);
        return b;
    }

    @Override
    public Iterable<ByteBuffer> iterate(int bufferSize) {
        if (bufferSize != this.increment) {
            throw new UnsupportedOperationException("Buffer size and increment mismatched: bufferSize = " + bufferSize + " but increment = " + this.increment);
        }
        return this.asByteBufferList();
    }

    @Override
    public List<ByteBuffer> asByteBufferList() {
        return Collections.unmodifiableList(this.buffers);
    }

    @Override
    public long writeTo(GatheringByteChannel channel) throws IOException {
        return BufferUtils.writeFully(channel, this.buffers);
    }

    @Override
    public ByteString toByteStringImpl(Function<ByteBuffer, ByteString> f) {
        ByteString result = ByteString.EMPTY;
        for (ByteBuffer buffer : this.buffers) {
            result = result.concat(f.apply(buffer));
        }
        return result;
    }

    @Override
    public List<ByteString> toByteStringListImpl(Function<ByteBuffer, ByteString> f) {
        ArrayList<ByteString> byteStringList = new ArrayList<ByteString>();
        for (ByteBuffer buffer : this.buffers) {
            byteStringList.add(f.apply(buffer));
        }
        return byteStringList;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof IncrementalChunkBuffer)) {
            return false;
        }
        IncrementalChunkBuffer that = (IncrementalChunkBuffer)obj;
        return this.limit == that.limit && this.buffers.equals(that.buffers);
    }

    public int hashCode() {
        return this.buffers.hashCode();
    }

    public String toString() {
        return this.getClass().getSimpleName() + ":limit=" + this.limit + ",increment=" + this.increment;
    }
}

