/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.volume;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import jakarta.annotation.Nullable;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.LinkedList;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.Queue;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Stream;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.fs.SpaceUsageCheckFactory;
import org.apache.hadoop.hdds.fs.SpaceUsageCheckParams;
import org.apache.hadoop.hdds.fs.SpaceUsageSource;
import org.apache.hadoop.hdfs.server.datanode.StorageLocation;
import org.apache.hadoop.hdfs.server.datanode.checker.Checkable;
import org.apache.hadoop.hdfs.server.datanode.checker.VolumeCheckResult;
import org.apache.hadoop.ozone.common.InconsistentStorageStateException;
import org.apache.hadoop.ozone.container.common.HDDSVolumeLayoutVersion;
import org.apache.hadoop.ozone.container.common.helpers.DatanodeVersionFile;
import org.apache.hadoop.ozone.container.common.impl.StorageLocationReport;
import org.apache.hadoop.ozone.container.common.statemachine.DatanodeConfiguration;
import org.apache.hadoop.ozone.container.common.utils.DiskCheckUtil;
import org.apache.hadoop.ozone.container.common.utils.StorageVolumeUtil;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.common.volume.MutableVolumeSet;
import org.apache.hadoop.ozone.container.common.volume.VolumeSet;
import org.apache.hadoop.ozone.container.common.volume.VolumeUsage;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class StorageVolume
implements Checkable<Boolean, VolumeCheckResult> {
    private static final Logger LOG = LoggerFactory.getLogger(StorageVolume.class);
    public static final String TMP_DIR_NAME = "tmp";
    public static final String TMP_DISK_CHECK_DIR_NAME = "disk-check";
    private volatile VolumeState state;
    private String storageID;
    private String clusterID;
    private String datanodeUuid;
    private long cTime;
    private int layoutVersion;
    private final ConfigurationSource conf;
    private final DatanodeConfiguration dnConf;
    private final StorageType storageType;
    private final String volumeRoot;
    private final File storageDir;
    private final String storageDirStr;
    private String workingDirName;
    private File tmpDir;
    private File diskCheckDir;
    private final Optional<VolumeUsage> volumeUsage;
    private final VolumeSet volumeSet;
    private final int ioTestCount;
    private final int ioFailureTolerance;
    private AtomicInteger currentIOFailureCount;
    private Queue<Boolean> ioTestSlidingWindow;
    private int healthCheckFileSize;

    protected StorageVolume(Builder<?> b) throws IOException {
        this.storageType = ((Builder)b).storageType;
        this.volumeRoot = ((Builder)b).volumeRootStr;
        if (!((Builder)b).failedVolume) {
            StorageLocation location = StorageLocation.parse((String)this.volumeRoot);
            this.storageDir = new File(location.getUri().getPath(), ((Builder)b).storageDirStr);
            SpaceUsageCheckParams checkParams = StorageVolume.getSpaceUsageCheckParams(b, this::getContainerDirsPath);
            checkParams.setContainerUsedSpace(this::containerUsedSpace);
            this.volumeUsage = Optional.of(new VolumeUsage(checkParams, ((Builder)b).conf));
            this.volumeSet = ((Builder)b).volumeSet;
            this.state = VolumeState.NOT_INITIALIZED;
            this.clusterID = ((Builder)b).clusterID;
            this.datanodeUuid = ((Builder)b).datanodeUuid;
            this.conf = ((Builder)b).conf;
            this.dnConf = (DatanodeConfiguration)((Object)this.conf.getObject(DatanodeConfiguration.class));
            this.ioTestCount = this.dnConf.getVolumeIOTestCount();
            this.ioFailureTolerance = this.dnConf.getVolumeIOFailureTolerance();
            this.ioTestSlidingWindow = new LinkedList<Boolean>();
            this.currentIOFailureCount = new AtomicInteger(0);
            this.healthCheckFileSize = this.dnConf.getVolumeHealthCheckFileSize();
        } else {
            this.storageDir = new File(((Builder)b).volumeRootStr);
            this.volumeUsage = Optional.empty();
            this.volumeSet = null;
            this.storageID = UUID.randomUUID().toString();
            this.state = VolumeState.FAILED;
            this.ioTestCount = 0;
            this.ioFailureTolerance = 0;
            this.conf = null;
            this.dnConf = null;
        }
        this.storageDirStr = this.storageDir.getAbsolutePath();
    }

    protected long containerUsedSpace() {
        return 0L;
    }

    public File getContainerDirsPath() {
        return null;
    }

    public void setGatherContainerUsages(Function<HddsVolume, Long> gatherContainerUsages) {
    }

    public void format(String cid) throws IOException {
        Preconditions.checkNotNull((Object)cid, (Object)"clusterID cannot be null while formatting Volume");
        this.clusterID = cid;
        this.initialize();
    }

    public void start() throws IOException {
        this.volumeUsage.ifPresent(VolumeUsage::start);
    }

    protected final void initialize() throws IOException {
        try {
            this.initializeImpl();
        }
        catch (Exception e) {
            this.shutdown();
            throw e;
        }
    }

    protected void initializeImpl() throws IOException {
        VolumeState intialVolumeState = this.analyzeVolumeState();
        switch (intialVolumeState.ordinal()) {
            case 2: {
                if (!this.getStorageDir().mkdirs()) {
                    throw new IOException("Cannot create directory " + this.getStorageDir());
                }
                this.setState(VolumeState.NOT_FORMATTED);
                this.createVersionFile();
                break;
            }
            case 4: {
                this.createVersionFile();
                break;
            }
            case 5: {
                this.readVersionFile();
                this.setState(VolumeState.NORMAL);
                break;
            }
            case 3: {
                throw new IOException("Volume is in an " + (Object)((Object)VolumeState.INCONSISTENT) + " state. Skipped loading volume: " + this.getStorageDir().getPath());
            }
            default: {
                throw new IOException("Unrecognized initial state : " + (Object)((Object)intialVolumeState) + "of volume : " + this.getStorageDir());
            }
        }
    }

    public void createWorkingDir(String dirName, MutableVolumeSet dbVolumeSet) throws IOException {
        File idDir = new File(this.getStorageDir(), dirName);
        if (!idDir.exists() && !idDir.mkdir()) {
            throw new IOException("Unable to create ID directory " + idDir + " for datanode.");
        }
        this.workingDirName = dirName;
    }

    public void createTmpDirs(String workDirName) throws IOException {
        this.tmpDir = new File(new File(this.getStorageDir(), workDirName), TMP_DIR_NAME);
        Files.createDirectories(this.tmpDir.toPath(), new FileAttribute[0]);
        this.diskCheckDir = this.createTmpSubdirIfNeeded(TMP_DISK_CHECK_DIR_NAME);
        this.cleanTmpDiskCheckDir();
    }

    protected final File createTmpSubdirIfNeeded(String name) throws IOException {
        File newDir = new File(this.tmpDir, name);
        Files.createDirectories(newDir.toPath(), new FileAttribute[0]);
        return newDir;
    }

    private VolumeState analyzeVolumeState() {
        if (!this.getStorageDir().exists()) {
            return VolumeState.NON_EXISTENT;
        }
        if (!this.getStorageDir().isDirectory()) {
            LOG.warn("Volume {} exists but is not a directory, current volume state: {}.", (Object)this.getStorageDir().getPath(), (Object)VolumeState.INCONSISTENT);
            return VolumeState.INCONSISTENT;
        }
        File[] files = this.getStorageDir().listFiles();
        if (files == null || files.length == 0) {
            return VolumeState.NOT_FORMATTED;
        }
        if (!this.getVersionFile().exists()) {
            LOG.warn("VERSION file does not exist in volume {}, current volume state: {}.", (Object)this.getStorageDir().getPath(), (Object)VolumeState.INCONSISTENT);
            return VolumeState.INCONSISTENT;
        }
        return VolumeState.NOT_INITIALIZED;
    }

    private void createVersionFile() throws IOException {
        this.storageID = StorageVolumeUtil.generateUuid();
        this.cTime = Time.now();
        this.layoutVersion = HDDSVolumeLayoutVersion.getLatestVersion().getVersion();
        if (this.clusterID == null || this.datanodeUuid == null) {
            LOG.debug("ClusterID not available. Cannot format the volume {}", (Object)this.getStorageDir().getPath());
            this.setState(VolumeState.NOT_FORMATTED);
        } else {
            this.writeVersionFile();
            this.setState(VolumeState.NORMAL);
        }
    }

    private void writeVersionFile() throws IOException {
        Preconditions.checkNotNull((Object)this.storageID, (Object)"StorageID cannot be null in Version File");
        Preconditions.checkNotNull((Object)this.clusterID, (Object)"ClusterID cannot be null in Version File");
        Preconditions.checkNotNull((Object)this.datanodeUuid, (Object)"DatanodeUUID cannot be null in Version File");
        Preconditions.checkArgument((this.cTime > 0L ? 1 : 0) != 0, (Object)"Creation Time should be positive");
        Preconditions.checkArgument((this.layoutVersion == HDDSVolumeLayoutVersion.getLatestVersion().getVersion() ? 1 : 0) != 0, (Object)"Version File should have the latest LayOutVersion");
        File versionFile = this.getVersionFile();
        LOG.debug("Writing Version file to disk, {}", (Object)versionFile);
        DatanodeVersionFile dnVersionFile = new DatanodeVersionFile(this.storageID, this.clusterID, this.datanodeUuid, this.cTime, this.layoutVersion);
        dnVersionFile.createVersionFile(versionFile);
    }

    private void readVersionFile() throws IOException {
        File versionFile = this.getVersionFile();
        Properties props = DatanodeVersionFile.readFrom(versionFile);
        if (props.isEmpty()) {
            throw new InconsistentStorageStateException("Version file " + versionFile + " is missing");
        }
        LOG.debug("Reading Version file from disk, {}", (Object)versionFile);
        this.storageID = StorageVolumeUtil.getStorageID(props, versionFile);
        this.clusterID = StorageVolumeUtil.getClusterID(props, versionFile, this.clusterID);
        this.datanodeUuid = StorageVolumeUtil.getDatanodeUUID(props, versionFile, this.datanodeUuid);
        this.cTime = StorageVolumeUtil.getCreationTime(props, versionFile);
        this.layoutVersion = StorageVolumeUtil.getLayOutVersion(props, versionFile);
    }

    private File getVersionFile() {
        return StorageVolumeUtil.getVersionFile(this.getStorageDir());
    }

    public String getVolumeRootDir() {
        return this.volumeRoot;
    }

    public SpaceUsageSource getCurrentUsage() {
        return this.volumeUsage.map(VolumeUsage::getCurrentUsage).orElse(SpaceUsageSource.UNKNOWN);
    }

    protected StorageLocationReport.Builder reportBuilder() {
        StorageLocationReport.Builder builder = StorageLocationReport.newBuilder().setFailed(this.isFailed()).setId(this.getStorageID()).setStorageLocation(this.storageDirStr).setStorageType(this.storageType);
        if (!builder.isFailed()) {
            SpaceUsageSource usage = this.getCurrentUsage();
            builder.setCapacity(usage.getCapacity()).setRemaining(usage.getAvailable()).setScmUsed(usage.getUsedSpace());
        }
        return builder;
    }

    public StorageLocationReport getReport() {
        return this.reportBuilder().build();
    }

    public File getStorageDir() {
        return this.storageDir;
    }

    public String getWorkingDirName() {
        return this.workingDirName;
    }

    public File getTmpDir() {
        return this.tmpDir;
    }

    @VisibleForTesting
    public File getDiskCheckDir() {
        return this.diskCheckDir;
    }

    public void refreshVolumeUsage() {
        this.volumeUsage.ifPresent(VolumeUsage::refreshNow);
    }

    public Optional<VolumeUsage> getVolumeUsage() {
        return this.volumeUsage;
    }

    public void incrementUsedSpace(long usedSpace) {
        this.volumeUsage.ifPresent(usage -> usage.incrementUsedSpace(usedSpace));
    }

    public void decrementUsedSpace(long reclaimedSpace) {
        this.volumeUsage.ifPresent(usage -> usage.decrementUsedSpace(reclaimedSpace));
    }

    public VolumeSet getVolumeSet() {
        return this.volumeSet;
    }

    public StorageType getStorageType() {
        return this.storageType;
    }

    public String getStorageID() {
        return this.storageID;
    }

    public String getClusterID() {
        return this.clusterID;
    }

    public String getDatanodeUuid() {
        return this.datanodeUuid;
    }

    public long getCTime() {
        return this.cTime;
    }

    public int getLayoutVersion() {
        return this.layoutVersion;
    }

    public VolumeState getStorageState() {
        return this.state;
    }

    public void setState(VolumeState state) {
        this.state = state;
    }

    public boolean isFailed() {
        return this.state == VolumeState.FAILED;
    }

    public ConfigurationSource getConf() {
        return this.conf;
    }

    public DatanodeConfiguration getDatanodeConfig() {
        return this.dnConf;
    }

    public void failVolume() {
        this.setState(VolumeState.FAILED);
        this.volumeUsage.ifPresent(VolumeUsage::shutdown);
    }

    public void shutdown() {
        this.setState(VolumeState.NON_EXISTENT);
        this.volumeUsage.ifPresent(VolumeUsage::shutdown);
        this.cleanTmpDiskCheckDir();
    }

    private void cleanTmpDiskCheckDir() {
        if (this.diskCheckDir == null) {
            return;
        }
        if (!this.diskCheckDir.exists()) {
            LOG.warn("Unable to clear disk check files from {}. Directory does not exist.", (Object)this.diskCheckDir);
            return;
        }
        if (!this.diskCheckDir.isDirectory()) {
            LOG.warn("Unable to clear disk check files from {}. Location is not a directory", (Object)this.diskCheckDir);
            return;
        }
        try (Stream<Path> files = Files.list(this.diskCheckDir.toPath());){
            files.map(Path::toFile).filter(File::isFile).forEach(file -> {
                try {
                    Files.delete(file.toPath());
                }
                catch (IOException ex) {
                    LOG.warn("Failed to delete temporary volume health check file {}", file);
                }
            });
        }
        catch (IOException ex) {
            LOG.warn("Failed to list contents of volume health check directory {} for deleting.", (Object)this.diskCheckDir);
        }
    }

    public synchronized VolumeCheckResult check(@Nullable Boolean unused) throws Exception {
        boolean directoryChecksPassed;
        boolean bl = directoryChecksPassed = DiskCheckUtil.checkExistence(this.storageDir) && DiskCheckUtil.checkPermissions(this.storageDir);
        if (!directoryChecksPassed) {
            if (Thread.currentThread().isInterrupted()) {
                throw new InterruptedException("Directory check of volume " + this + " interrupted.");
            }
            return VolumeCheckResult.FAILED;
        }
        if (this.ioTestCount == 0) {
            return VolumeCheckResult.HEALTHY;
        }
        int minimumDiskSpace = this.healthCheckFileSize * 2;
        if (this.getCurrentUsage().getAvailable() < (long)minimumDiskSpace) {
            this.ioTestSlidingWindow.add(true);
            return VolumeCheckResult.HEALTHY;
        }
        boolean diskChecksPassed = DiskCheckUtil.checkReadWrite(this.storageDir, this.diskCheckDir, this.healthCheckFileSize);
        if (Thread.currentThread().isInterrupted()) {
            throw new InterruptedException("IO check of volume " + this + " interrupted.");
        }
        if (!diskChecksPassed && this.getCurrentUsage().getAvailable() < (long)minimumDiskSpace) {
            this.ioTestSlidingWindow.add(true);
            return VolumeCheckResult.HEALTHY;
        }
        this.ioTestSlidingWindow.add(diskChecksPassed);
        if (!diskChecksPassed) {
            this.currentIOFailureCount.incrementAndGet();
        }
        if (this.ioTestSlidingWindow.size() > this.ioTestCount && Objects.equals(this.ioTestSlidingWindow.poll(), Boolean.FALSE)) {
            this.currentIOFailureCount.decrementAndGet();
        }
        if (this.currentIOFailureCount.get() > this.ioFailureTolerance) {
            LOG.error("Failed IO test for volume {}: the last {} runs encountered {} out of {} tolerated failures.", new Object[]{this, this.ioTestSlidingWindow.size(), this.currentIOFailureCount, this.ioFailureTolerance});
            return VolumeCheckResult.FAILED;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("IO test results for volume {}: the last {} runs encountered {} out of {} tolerated failures", new Object[]{this, this.ioTestSlidingWindow.size(), this.currentIOFailureCount, this.ioFailureTolerance});
        }
        return VolumeCheckResult.HEALTHY;
    }

    public int hashCode() {
        return Objects.hash(this.storageDir);
    }

    public boolean equals(Object other) {
        return this == other || other instanceof StorageVolume && ((StorageVolume)other).storageDir.equals(this.storageDir);
    }

    public String toString() {
        return this.getStorageDir().toString();
    }

    private static SpaceUsageCheckParams getSpaceUsageCheckParams(Builder b, Supplier<File> exclusionProvider) throws IOException {
        boolean succeeded;
        File root = new File(b.volumeRootStr);
        boolean bl = succeeded = root.isDirectory() || root.mkdirs();
        if (!succeeded) {
            LOG.error("Unable to create the volume root dir at : {}", (Object)root);
            throw new IOException("Unable to create the volume root dir at " + root);
        }
        SpaceUsageCheckFactory usageCheckFactory = b.usageCheckFactory;
        if (usageCheckFactory == null) {
            usageCheckFactory = SpaceUsageCheckFactory.create((ConfigurationSource)b.conf);
        }
        return usageCheckFactory.paramsFor(root, exclusionProvider);
    }

    public static abstract class Builder<T extends Builder<T>> {
        private final String volumeRootStr;
        private String storageDirStr;
        private ConfigurationSource conf;
        private StorageType storageType = StorageType.DEFAULT;
        private SpaceUsageCheckFactory usageCheckFactory;
        private VolumeSet volumeSet;
        private boolean failedVolume = false;
        private String datanodeUuid;
        private String clusterID;

        public Builder(String volumeRootStr, String storageDirStr) {
            this.volumeRootStr = volumeRootStr;
            this.storageDirStr = storageDirStr;
        }

        public abstract T getThis();

        public T conf(ConfigurationSource config) {
            this.conf = config;
            return this.getThis();
        }

        public T storageType(StorageType st) {
            this.storageType = Objects.requireNonNull(st, "storageType == null");
            return this.getThis();
        }

        public T usageCheckFactory(SpaceUsageCheckFactory factory) {
            this.usageCheckFactory = factory;
            return this.getThis();
        }

        public T volumeSet(VolumeSet volSet) {
            this.volumeSet = volSet;
            return this.getThis();
        }

        public T failedVolume(boolean failed) {
            this.failedVolume = failed;
            return this.getThis();
        }

        public T datanodeUuid(String datanodeUUID) {
            this.datanodeUuid = datanodeUUID;
            return this.getThis();
        }

        public T clusterID(String cid) {
            this.clusterID = cid;
            return this.getThis();
        }

        public abstract StorageVolume build() throws IOException;

        public String getVolumeRootStr() {
            return this.volumeRootStr;
        }

        public boolean getFailedVolume() {
            return this.failedVolume;
        }

        public StorageType getStorageType() {
            return this.storageType;
        }
    }

    public static enum VolumeState {
        NORMAL,
        FAILED,
        NON_EXISTENT,
        INCONSISTENT,
        NOT_FORMATTED,
        NOT_INITIALIZED;

    }

    public static enum VolumeType {
        DATA_VOLUME,
        META_VOLUME,
        DB_VOLUME;

    }
}

