/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.impl;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.protobuf.ServiceException;
import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.hadoop.hdds.HddsUtils;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.ContainerNotOpenException;
import org.apache.hadoop.hdds.scm.container.common.helpers.InvalidContainerStateException;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.scm.protocolPB.ContainerCommandResponseBuilders;
import org.apache.hadoop.hdds.security.token.NoopTokenVerifier;
import org.apache.hadoop.hdds.security.token.TokenVerifier;
import org.apache.hadoop.hdds.server.OzoneProtocolMessageDispatcher;
import org.apache.hadoop.hdds.utils.ProtocolMessageMetrics;
import org.apache.hadoop.ozone.audit.AuditAction;
import org.apache.hadoop.ozone.audit.AuditEventStatus;
import org.apache.hadoop.ozone.audit.AuditLogger;
import org.apache.hadoop.ozone.audit.AuditLoggerType;
import org.apache.hadoop.ozone.audit.AuditMarker;
import org.apache.hadoop.ozone.audit.AuditMessage;
import org.apache.hadoop.ozone.audit.Auditor;
import org.apache.hadoop.ozone.audit.DNAction;
import org.apache.hadoop.ozone.container.common.helpers.BlockData;
import org.apache.hadoop.ozone.container.common.helpers.ContainerMetrics;
import org.apache.hadoop.ozone.container.common.helpers.ContainerUtils;
import org.apache.hadoop.ozone.container.common.impl.ContainerData;
import org.apache.hadoop.ozone.container.common.impl.ContainerSet;
import org.apache.hadoop.ozone.container.common.interfaces.Container;
import org.apache.hadoop.ozone.container.common.interfaces.ContainerDispatcher;
import org.apache.hadoop.ozone.container.common.interfaces.Handler;
import org.apache.hadoop.ozone.container.common.statemachine.StateContext;
import org.apache.hadoop.ozone.container.common.transport.server.ratis.DispatcherContext;
import org.apache.hadoop.ozone.container.common.volume.VolumeSet;
import org.apache.hadoop.ozone.container.ozoneimpl.ContainerScanError;
import org.apache.hadoop.ozone.container.ozoneimpl.DataScanResult;
import org.apache.hadoop.util.Time;
import org.apache.ratis.statemachine.StateMachine;
import org.apache.ratis.thirdparty.com.google.protobuf.ProtocolMessageEnum;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HddsDispatcher
implements ContainerDispatcher,
Auditor {
    static final Logger LOG = LoggerFactory.getLogger(HddsDispatcher.class);
    private static final AuditLogger AUDIT = new AuditLogger(AuditLoggerType.DNLOGGER);
    private static final String AUDIT_PARAM_CONTAINER_ID = "containerID";
    private static final String AUDIT_PARAM_CONTAINER_TYPE = "containerType";
    private static final String AUDIT_PARAM_FORCE_UPDATE = "forceUpdate";
    private static final String AUDIT_PARAM_FORCE_DELETE = "forceDelete";
    private static final String AUDIT_PARAM_START_CONTAINER_ID = "startContainerID";
    private static final String AUDIT_PARAM_BLOCK_DATA = "blockData";
    private static final String AUDIT_PARAM_BLOCK_DATA_OFFSET = "offset";
    private static final String AUDIT_PARAM_BLOCK_DATA_SIZE = "size";
    private static final String AUDIT_PARAM_BLOCK_DATA_STAGE = "stage";
    private static final String AUDIT_PARAM_COUNT = "count";
    private static final String AUDIT_PARAM_START_LOCAL_ID = "startLocalID";
    private static final String AUDIT_PARAM_PREV_CHUNKNAME = "prevChunkName";
    private final Map<ContainerProtos.ContainerType, Handler> handlers;
    private final ConfigurationSource conf;
    private final ContainerSet containerSet;
    private final StateContext context;
    private final float containerCloseThreshold;
    private final ProtocolMessageMetrics<ProtocolMessageEnum> protocolMetrics;
    private OzoneProtocolMessageDispatcher<ContainerProtos.ContainerCommandRequestProto, ContainerProtos.ContainerCommandResponseProto, ProtocolMessageEnum> dispatcher;
    private String clusterId;
    private ContainerMetrics metrics;
    private final TokenVerifier tokenVerifier;
    private long slowOpThresholdNs;

    public HddsDispatcher(ConfigurationSource config, ContainerSet contSet, VolumeSet volumes, Map<ContainerProtos.ContainerType, Handler> handlers, StateContext context, ContainerMetrics metrics, TokenVerifier tokenVerifier) {
        this.conf = config;
        this.containerSet = contSet;
        this.context = context;
        this.handlers = handlers;
        this.metrics = metrics;
        this.containerCloseThreshold = this.conf.getFloat("hdds.container.close.threshold", 0.9f);
        this.tokenVerifier = tokenVerifier != null ? tokenVerifier : new NoopTokenVerifier();
        this.slowOpThresholdNs = this.getSlowOpThresholdMs(this.conf) * 1000000L;
        this.protocolMetrics = new ProtocolMessageMetrics("HddsDispatcher", "HDDS dispatcher metrics", (Object[])ContainerProtos.Type.values());
        this.dispatcher = new OzoneProtocolMessageDispatcher("DatanodeClient", this.protocolMetrics, LOG, HddsUtils::processForDebug, HddsUtils::processForDebug);
    }

    @Override
    public void init() {
        this.protocolMetrics.register();
    }

    @Override
    public void shutdown() {
        this.protocolMetrics.unregister();
    }

    private boolean canIgnoreException(ContainerProtos.Result result) {
        switch (result) {
            case SUCCESS: 
            case CONTAINER_UNHEALTHY: 
            case CLOSED_CONTAINER_IO: 
            case DELETE_ON_OPEN_CONTAINER: 
            case UNSUPPORTED_REQUEST: 
            case CONTAINER_MISSING: {
                return true;
            }
        }
        return false;
    }

    @Override
    public void buildMissingContainerSetAndValidate(Map<Long, Long> container2BCSIDMap) {
        this.containerSet.buildMissingContainerSetAndValidate(container2BCSIDMap, n -> n);
    }

    @Override
    public ContainerProtos.ContainerCommandResponseProto dispatch(ContainerProtos.ContainerCommandRequestProto msg, DispatcherContext dispatcherContext) {
        try {
            return (ContainerProtos.ContainerCommandResponseProto)this.dispatcher.processRequest((Object)msg, req -> this.dispatchRequest(msg, dispatcherContext), (Object)msg.getCmdType(), msg.getTraceID());
        }
        catch (ServiceException ex) {
            throw new RuntimeException(ex);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ContainerProtos.ContainerCommandResponseProto dispatchRequest(ContainerProtos.ContainerCommandRequestProto msg, DispatcherContext dispatcherContext) {
        Handler handler;
        ContainerProtos.ContainerType containerType;
        boolean isWriteCommitStage;
        Preconditions.checkNotNull((Object)msg);
        if (LOG.isTraceEnabled()) {
            LOG.trace("Command {}, trace ID: {} ", (Object)msg.getCmdType(), (Object)msg.getTraceID());
        }
        DNAction action = HddsDispatcher.getAuditAction(msg.getCmdType());
        EventType eventType = this.getEventType(msg);
        AuditLogger.PerformanceStringBuilder perf = new AuditLogger.PerformanceStringBuilder();
        ContainerProtos.ContainerCommandResponseProto responseProto = null;
        long startTime = Time.monotonicNowNanos();
        ContainerProtos.Type cmdType = msg.getCmdType();
        long containerID = msg.getContainerID();
        Container container = this.getContainer(containerID);
        boolean isWriteStage = cmdType == ContainerProtos.Type.WriteChunk && dispatcherContext != null && dispatcherContext.getStage() == DispatcherContext.WriteChunkStage.WRITE_DATA || cmdType == ContainerProtos.Type.StreamInit;
        boolean bl = isWriteCommitStage = cmdType == ContainerProtos.Type.WriteChunk && dispatcherContext != null && dispatcherContext.getStage() == DispatcherContext.WriteChunkStage.COMMIT_DATA;
        if (dispatcherContext == null) {
            this.metrics.incContainerOpsMetrics(cmdType);
        } else if (isWriteStage) {
            this.metrics.incContainerOpsMetrics(cmdType);
        } else if (cmdType != ContainerProtos.Type.WriteChunk) {
            this.metrics.incContainerOpsMetrics(cmdType);
        }
        try {
            if (DispatcherContext.op(dispatcherContext).validateToken()) {
                this.validateToken(msg);
            }
        }
        catch (IOException ioe) {
            String s = ContainerProtos.Result.BLOCK_TOKEN_VERIFICATION_FAILED + " for " + dispatcherContext + ": " + ioe.getMessage();
            StorageContainerException sce = new StorageContainerException(s, (Throwable)ioe, ContainerProtos.Result.BLOCK_TOKEN_VERIFICATION_FAILED);
            return ContainerUtils.logAndReturnError(LOG, sce, msg);
        }
        boolean isCombinedStage = cmdType == ContainerProtos.Type.WriteChunk && (dispatcherContext == null || dispatcherContext.getStage() == DispatcherContext.WriteChunkStage.COMBINED);
        Map<Long, Long> container2BCSIDMap = null;
        if (dispatcherContext != null) {
            container2BCSIDMap = dispatcherContext.getContainer2BCSIDMap();
        }
        if (isWriteCommitStage) {
            Preconditions.checkNotNull(container2BCSIDMap);
            if (container != null && container2BCSIDMap.get(containerID) == null) {
                container2BCSIDMap.put(containerID, container.getBlockCommitSequenceId());
                this.getMissingContainerSet().remove(containerID);
            }
        }
        if (cmdType != ContainerProtos.Type.CreateContainer && !HddsUtils.isReadOnly((ContainerProtos.ContainerCommandRequestProtoOrBuilder)msg) && this.getMissingContainerSet().contains(containerID)) {
            StorageContainerException sce = new StorageContainerException("ContainerID " + containerID + " has been lost and cannot be recreated on this DataNode", ContainerProtos.Result.CONTAINER_MISSING);
            this.audit(action, eventType, msg, dispatcherContext, AuditEventStatus.FAILURE, sce);
            return ContainerUtils.logAndReturnError(LOG, sce, msg);
        }
        if (cmdType != ContainerProtos.Type.CreateContainer) {
            if (container == null && (isWriteStage || isCombinedStage || cmdType == ContainerProtos.Type.PutSmallFile || cmdType == ContainerProtos.Type.PutBlock)) {
                responseProto = this.createContainer(msg);
                this.metrics.incContainerOpsMetrics(ContainerProtos.Type.CreateContainer);
                this.metrics.incContainerOpsLatencies(ContainerProtos.Type.CreateContainer, Time.monotonicNowNanos() - startTime);
                if (responseProto.getResult() != ContainerProtos.Result.SUCCESS) {
                    StorageContainerException sce = new StorageContainerException("ContainerID " + containerID + " creation failed", responseProto.getResult());
                    this.audit(action, eventType, msg, dispatcherContext, AuditEventStatus.FAILURE, sce);
                    return ContainerUtils.logAndReturnError(LOG, sce, msg);
                }
                Preconditions.checkArgument((isWriteStage && container2BCSIDMap != null || dispatcherContext == null || cmdType == ContainerProtos.Type.PutBlock ? 1 : 0) != 0);
                if (container2BCSIDMap != null) {
                    container2BCSIDMap.putIfAbsent(containerID, 0L);
                }
                container = this.getContainer(containerID);
            }
            if (container == null) {
                StorageContainerException sce = new StorageContainerException("ContainerID " + containerID + " does not exist", ContainerProtos.Result.CONTAINER_NOT_FOUND);
                this.audit(action, eventType, msg, dispatcherContext, AuditEventStatus.FAILURE, sce);
                return ContainerUtils.logAndReturnError(LOG, sce, msg);
            }
            containerType = this.getContainerType(container);
        } else {
            if (!msg.hasCreateContainer()) {
                this.audit(action, eventType, msg, dispatcherContext, AuditEventStatus.FAILURE, new Exception("MALFORMED_REQUEST"));
                return ContainerCommandResponseBuilders.malformedRequest((ContainerProtos.ContainerCommandRequestProto)msg);
            }
            containerType = msg.getCreateContainer().getContainerType();
        }
        boolean isVolumeFullForWrite = false;
        if (!HddsUtils.isReadOnly((ContainerProtos.ContainerCommandRequestProtoOrBuilder)msg)) {
            try {
                if (container != null && container.getContainerState() == ContainerProtos.ContainerDataProto.State.OPEN) {
                    ContainerUtils.assertSpaceAvailability(containerID, ((ContainerData)container.getContainerData()).getVolume(), 0);
                }
            }
            catch (StorageContainerException e) {
                LOG.warn(e.getMessage());
                isVolumeFullForWrite = true;
                if (cmdType == ContainerProtos.Type.WriteChunk || cmdType == ContainerProtos.Type.PutBlock || cmdType == ContainerProtos.Type.PutSmallFile) {
                    this.audit(action, eventType, msg, dispatcherContext, AuditEventStatus.FAILURE, e);
                    ContainerProtos.ContainerCommandResponseProto containerCommandResponseProto = ContainerUtils.logAndReturnError(LOG, e, msg);
                    return containerCommandResponseProto;
                }
            }
            finally {
                this.sendCloseContainerActionIfNeeded(container, isVolumeFullForWrite);
            }
        }
        if ((handler = this.getHandler(containerType)) == null) {
            StorageContainerException ex = new StorageContainerException("Invalid ContainerType " + containerType, ContainerProtos.Result.CONTAINER_INTERNAL_ERROR);
            this.audit(action, eventType, msg, dispatcherContext, AuditEventStatus.FAILURE, ex);
            return ContainerUtils.logAndReturnError(LOG, ex, msg);
        }
        perf.appendPreOpLatencyNano(Time.monotonicNowNanos() - startTime);
        responseProto = handler.handle(msg, container, dispatcherContext);
        long opLatencyNs = Time.monotonicNowNanos() - startTime;
        if (responseProto != null) {
            this.metrics.incContainerOpsLatencies(cmdType, opLatencyNs);
            ContainerProtos.Result result = responseProto.getResult();
            if (!HddsUtils.isReadOnly((ContainerProtos.ContainerCommandRequestProtoOrBuilder)msg) && !this.canIgnoreException(result)) {
                if (container == null) {
                    throw new NullPointerException("Error on creating containers " + result + " " + responseProto.getMessage());
                }
                ContainerProtos.ContainerDataProto.State containerState = ((ContainerData)container.getContainerData()).getState();
                Preconditions.checkState((containerState == ContainerProtos.ContainerDataProto.State.OPEN || containerState == ContainerProtos.ContainerDataProto.State.CLOSING || containerState == ContainerProtos.ContainerDataProto.State.RECOVERING ? 1 : 0) != 0);
                try {
                    ContainerScanError error = new ContainerScanError(ContainerScanError.FailureType.WRITE_FAILURE, new File(((ContainerData)container.getContainerData()).getContainerPath()), (Exception)((Object)new StorageContainerException(result)));
                    handler.markContainerUnhealthy(container, DataScanResult.fromErrors(Collections.singletonList(error)));
                    this.containerSet.scanContainerWithoutGap(containerID, "Unhealthy container scan");
                    LOG.info("Marked Container UNHEALTHY, ContainerID: {}", (Object)containerID);
                }
                catch (IOException ioe) {
                    LOG.error("Failed to mark container " + containerID + " UNHEALTHY. ", (Throwable)ioe);
                }
                Preconditions.checkArgument((((ContainerData)container.getContainerData()).getState() == ContainerProtos.ContainerDataProto.State.UNHEALTHY ? 1 : 0) != 0);
                this.sendCloseContainerActionIfNeeded(container, isVolumeFullForWrite);
            }
            if (cmdType == ContainerProtos.Type.CreateContainer && result == ContainerProtos.Result.SUCCESS && dispatcherContext != null) {
                Preconditions.checkNotNull(dispatcherContext.getContainer2BCSIDMap());
                container2BCSIDMap.putIfAbsent(containerID, 0L);
            }
            if (result == ContainerProtos.Result.SUCCESS) {
                this.updateBCSID(container, dispatcherContext, cmdType);
                this.audit(action, eventType, msg, dispatcherContext, AuditEventStatus.SUCCESS, null);
            } else {
                this.containerSet.scanContainer(containerID, result.name());
                this.audit(action, eventType, msg, dispatcherContext, AuditEventStatus.FAILURE, new Exception(responseProto.getMessage()));
            }
            perf.appendOpLatencyNanos(opLatencyNs);
            this.performanceAudit(action, msg, dispatcherContext, perf, opLatencyNs);
            return responseProto;
        }
        this.audit(action, eventType, msg, dispatcherContext, AuditEventStatus.FAILURE, new Exception("UNSUPPORTED_REQUEST"));
        return ContainerCommandResponseBuilders.unsupportedRequest((ContainerProtos.ContainerCommandRequestProto)msg);
    }

    private long getSlowOpThresholdMs(ConfigurationSource config) {
        return config.getTimeDuration("hdds.datanode.slow.op.warning.threshold", "500ms", TimeUnit.MILLISECONDS);
    }

    private void updateBCSID(Container container, DispatcherContext dispatcherContext, ContainerProtos.Type cmdType) {
        if (dispatcherContext != null && (cmdType == ContainerProtos.Type.PutBlock || cmdType == ContainerProtos.Type.PutSmallFile)) {
            Preconditions.checkNotNull((Object)container);
            long bcsID = container.getBlockCommitSequenceId();
            long containerId = ((ContainerData)container.getContainerData()).getContainerID();
            Map<Long, Long> container2BCSIDMap = dispatcherContext.getContainer2BCSIDMap();
            Preconditions.checkNotNull(container2BCSIDMap);
            Preconditions.checkArgument((boolean)container2BCSIDMap.containsKey(containerId));
            container2BCSIDMap.computeIfPresent(containerId, (u, v) -> {
                v = bcsID;
                return v;
            });
        }
    }

    @VisibleForTesting
    ContainerProtos.ContainerCommandResponseProto createContainer(ContainerProtos.ContainerCommandRequestProto containerRequest) {
        ContainerProtos.CreateContainerRequestProto.Builder createRequest = ContainerProtos.CreateContainerRequestProto.newBuilder();
        ContainerProtos.ContainerType containerType = ContainerProtos.ContainerType.KeyValueContainer;
        createRequest.setContainerType(containerType);
        if (containerRequest.hasWriteChunk()) {
            createRequest.setReplicaIndex(containerRequest.getWriteChunk().getBlockID().getReplicaIndex());
        }
        if (containerRequest.hasPutBlock()) {
            createRequest.setReplicaIndex(containerRequest.getPutBlock().getBlockData().getBlockID().getReplicaIndex());
        }
        ContainerProtos.ContainerCommandRequestProto.Builder requestBuilder = ContainerProtos.ContainerCommandRequestProto.newBuilder().setCmdType(ContainerProtos.Type.CreateContainer).setContainerID(containerRequest.getContainerID()).setCreateContainer(createRequest.build()).setPipelineID(containerRequest.getPipelineID()).setDatanodeUuid(containerRequest.getDatanodeUuid()).setTraceID(containerRequest.getTraceID());
        Handler handler = this.getHandler(containerType);
        return handler.handle(requestBuilder.build(), null, null);
    }

    private void validateToken(ContainerProtos.ContainerCommandRequestProto msg) throws IOException {
        this.tokenVerifier.verify((ContainerProtos.ContainerCommandRequestProtoOrBuilder)msg, msg.getEncodedToken());
    }

    @Override
    public void validateContainerCommand(ContainerProtos.ContainerCommandRequestProto msg) throws StorageContainerException {
        try {
            this.validateToken(msg);
        }
        catch (IOException ioe) {
            throw new StorageContainerException(ContainerProtos.Result.BLOCK_TOKEN_VERIFICATION_FAILED + ": " + ioe.getMessage(), (Throwable)ioe, ContainerProtos.Result.BLOCK_TOKEN_VERIFICATION_FAILED);
        }
        long containerID = msg.getContainerID();
        Container container = this.getContainer(containerID);
        if (container == null) {
            return;
        }
        ContainerProtos.ContainerType containerType = container.getContainerType();
        ContainerProtos.Type cmdType = msg.getCmdType();
        DNAction action = HddsDispatcher.getAuditAction(cmdType);
        EventType eventType = this.getEventType(msg);
        Handler handler = this.getHandler(containerType);
        if (handler == null) {
            StorageContainerException ex = new StorageContainerException("Invalid ContainerType " + containerType, ContainerProtos.Result.CONTAINER_INTERNAL_ERROR);
            this.audit(action, eventType, msg, null, AuditEventStatus.FAILURE, ex);
            throw ex;
        }
        ContainerProtos.ContainerDataProto.State containerState = container.getContainerState();
        String log = "Container " + containerID + " in " + containerState + " state";
        if (!HddsUtils.isReadOnly((ContainerProtos.ContainerCommandRequestProtoOrBuilder)msg) && !HddsUtils.isOpenToWriteState((ContainerProtos.ContainerDataProto.State)containerState)) {
            switch (cmdType) {
                case CreateContainer: {
                    break;
                }
                case CloseContainer: {
                    break;
                }
                default: {
                    ContainerNotOpenException cex = new ContainerNotOpenException(log);
                    this.audit(action, eventType, msg, null, AuditEventStatus.FAILURE, (Throwable)cex);
                    throw cex;
                }
            }
        } else if (HddsUtils.isReadOnly((ContainerProtos.ContainerCommandRequestProtoOrBuilder)msg) && containerState == ContainerProtos.ContainerDataProto.State.INVALID) {
            InvalidContainerStateException iex = new InvalidContainerStateException(log);
            this.audit(action, eventType, msg, null, AuditEventStatus.FAILURE, (Throwable)iex);
            throw iex;
        }
    }

    private void sendCloseContainerActionIfNeeded(Container container, boolean isVolumeFull) {
        boolean shouldClose;
        boolean isSpaceFull = isVolumeFull || this.isContainerFull(container);
        boolean bl = shouldClose = isSpaceFull || this.isContainerUnhealthy(container);
        if (shouldClose) {
            Object containerData = container.getContainerData();
            StorageContainerDatanodeProtocolProtos.ContainerAction.Reason reason = isSpaceFull ? StorageContainerDatanodeProtocolProtos.ContainerAction.Reason.CONTAINER_FULL : StorageContainerDatanodeProtocolProtos.ContainerAction.Reason.CONTAINER_UNHEALTHY;
            StorageContainerDatanodeProtocolProtos.ContainerAction action = StorageContainerDatanodeProtocolProtos.ContainerAction.newBuilder().setContainerID(((ContainerData)containerData).getContainerID()).setAction(StorageContainerDatanodeProtocolProtos.ContainerAction.Action.CLOSE).setReason(reason).build();
            this.context.addContainerActionIfAbsent(action);
            AtomicBoolean immediateCloseActionSent = ((ContainerData)containerData).getImmediateCloseActionSent();
            if (immediateCloseActionSent.compareAndSet(false, true)) {
                this.context.getParent().triggerHeartbeat();
                if (isVolumeFull) {
                    LOG.warn("Triggered immediate heartbeat because of full volume.");
                }
            }
        }
    }

    private boolean isContainerFull(Container container) {
        boolean isOpen = Optional.ofNullable(container).map(cont -> cont.getContainerState() == ContainerProtos.ContainerDataProto.State.OPEN).orElse(Boolean.FALSE);
        if (isOpen) {
            Object containerData = container.getContainerData();
            double containerUsedPercentage = 1.0f * (float)((ContainerData)containerData).getBytesUsed() / (float)((ContainerData)containerData).getMaxSize();
            return containerUsedPercentage >= (double)this.containerCloseThreshold;
        }
        return false;
    }

    private boolean isContainerUnhealthy(Container container) {
        return Optional.ofNullable(container).map(cont -> cont.getContainerState() == ContainerProtos.ContainerDataProto.State.UNHEALTHY).orElse(Boolean.FALSE);
    }

    @Override
    public Handler getHandler(ContainerProtos.ContainerType containerType) {
        return this.handlers.get(containerType);
    }

    @Override
    public void setClusterId(String clusterId) {
        Preconditions.checkNotNull((Object)clusterId, (Object)"clusterId cannot be null");
        if (this.clusterId == null) {
            this.clusterId = clusterId;
            for (Map.Entry<ContainerProtos.ContainerType, Handler> handlerMap : this.handlers.entrySet()) {
                handlerMap.getValue().setClusterID(clusterId);
            }
        }
    }

    @VisibleForTesting
    public Container getContainer(long containerID) {
        return this.containerSet.getContainer(containerID);
    }

    @VisibleForTesting
    public Set<Long> getMissingContainerSet() {
        return this.containerSet.getMissingContainerSet();
    }

    private ContainerProtos.ContainerType getContainerType(Container container) {
        return container.getContainerType();
    }

    @VisibleForTesting
    public void setMetricsForTesting(ContainerMetrics containerMetrics) {
        this.metrics = containerMetrics;
    }

    private EventType getEventType(ContainerProtos.ContainerCommandRequestProto msg) {
        return HddsUtils.isReadOnly((ContainerProtos.ContainerCommandRequestProtoOrBuilder)msg) ? EventType.READ : EventType.WRITE;
    }

    private void audit(AuditAction action, EventType eventType, ContainerProtos.ContainerCommandRequestProto msg, DispatcherContext dispatcherContext, AuditEventStatus result, Throwable exception) {
        switch (result) {
            case SUCCESS: {
                if (!this.isAllowed(action.getAction())) break;
                Map<String, String> params = HddsDispatcher.getAuditParams(msg, dispatcherContext);
                if (eventType == EventType.READ && AUDIT.getLogger().isInfoEnabled(AuditMarker.READ.getMarker())) {
                    AuditMessage amsg = this.buildAuditMessageForSuccess(action, params);
                    AUDIT.logReadSuccess(amsg);
                    break;
                }
                if (eventType != EventType.WRITE || !AUDIT.getLogger().isInfoEnabled(AuditMarker.WRITE.getMarker())) break;
                AuditMessage amsg = this.buildAuditMessageForSuccess(action, params);
                AUDIT.logWriteSuccess(amsg);
                break;
            }
            case FAILURE: {
                Map<String, String> params = HddsDispatcher.getAuditParams(msg, dispatcherContext);
                if (eventType == EventType.READ && AUDIT.getLogger().isErrorEnabled(AuditMarker.READ.getMarker())) {
                    AuditMessage amsg = this.buildAuditMessageForFailure(action, params, exception);
                    AUDIT.logReadFailure(amsg);
                    break;
                }
                if (eventType != EventType.WRITE || !AUDIT.getLogger().isErrorEnabled(AuditMarker.WRITE.getMarker())) break;
                AuditMessage amsg = this.buildAuditMessageForFailure(action, params, exception);
                AUDIT.logWriteFailure(amsg);
                break;
            }
            default: {
                if (!LOG.isDebugEnabled()) break;
                LOG.debug("Invalid audit event status - {}", (Object)result);
            }
        }
    }

    private void performanceAudit(AuditAction action, ContainerProtos.ContainerCommandRequestProto msg, DispatcherContext dispatcherContext, AuditLogger.PerformanceStringBuilder performance, long opLatencyNs) {
        if (this.isOperationSlow(opLatencyNs)) {
            Map<String, String> params = HddsDispatcher.getAuditParams(msg, dispatcherContext);
            AuditMessage auditMessage = this.buildAuditMessageForPerformance(action, params, performance);
            AUDIT.logPerformance(auditMessage);
        }
    }

    public AuditMessage buildAuditMessageForPerformance(AuditAction op, Map<String, String> auditMap, AuditLogger.PerformanceStringBuilder performance) {
        return new AuditMessage.Builder().setUser(null).atIp(null).forOperation(op).withParams(auditMap).setPerformance(performance).build();
    }

    public AuditMessage buildAuditMessageForSuccess(AuditAction op, Map<String, String> auditMap) {
        return new AuditMessage.Builder().setUser(null).atIp(null).forOperation(op).withParams(auditMap).withResult(AuditEventStatus.SUCCESS).build();
    }

    public AuditMessage buildAuditMessageForFailure(AuditAction op, Map<String, String> auditMap, Throwable throwable) {
        return new AuditMessage.Builder().setUser(null).atIp(null).forOperation(op).withParams(auditMap).withResult(AuditEventStatus.FAILURE).withException(throwable).build();
    }

    private boolean isAllowed(String action) {
        switch (action) {
            case "CLOSE_CONTAINER": 
            case "CREATE_CONTAINER": 
            case "LIST_CONTAINER": 
            case "DELETE_CONTAINER": 
            case "READ_CONTAINER": 
            case "UPDATE_CONTAINER": 
            case "DELETE_BLOCK": {
                return true;
            }
        }
        return false;
    }

    @Override
    public StateMachine.DataChannel getStreamDataChannel(ContainerProtos.ContainerCommandRequestProto msg) throws StorageContainerException {
        long containerID = msg.getContainerID();
        Container container = this.getContainer(containerID);
        if (container != null) {
            Handler handler = this.getHandler(this.getContainerType(container));
            return handler.getStreamDataChannel(container, msg);
        }
        throw new StorageContainerException("ContainerID " + containerID + " does not exist", ContainerProtos.Result.CONTAINER_NOT_FOUND);
    }

    private static DNAction getAuditAction(ContainerProtos.Type cmdType) {
        switch (cmdType) {
            case CreateContainer: {
                return DNAction.CREATE_CONTAINER;
            }
            case ReadContainer: {
                return DNAction.READ_CONTAINER;
            }
            case UpdateContainer: {
                return DNAction.UPDATE_CONTAINER;
            }
            case DeleteContainer: {
                return DNAction.DELETE_CONTAINER;
            }
            case ListContainer: {
                return DNAction.LIST_CONTAINER;
            }
            case PutBlock: {
                return DNAction.PUT_BLOCK;
            }
            case GetBlock: {
                return DNAction.GET_BLOCK;
            }
            case DeleteBlock: {
                return DNAction.DELETE_BLOCK;
            }
            case ListBlock: {
                return DNAction.LIST_BLOCK;
            }
            case ReadChunk: {
                return DNAction.READ_CHUNK;
            }
            case DeleteChunk: {
                return DNAction.DELETE_CHUNK;
            }
            case WriteChunk: {
                return DNAction.WRITE_CHUNK;
            }
            case ListChunk: {
                return DNAction.LIST_CHUNK;
            }
            case CompactChunk: {
                return DNAction.COMPACT_CHUNK;
            }
            case PutSmallFile: {
                return DNAction.PUT_SMALL_FILE;
            }
            case GetSmallFile: {
                return DNAction.GET_SMALL_FILE;
            }
            case CloseContainer: {
                return DNAction.CLOSE_CONTAINER;
            }
            case GetCommittedBlockLength: {
                return DNAction.GET_COMMITTED_BLOCK_LENGTH;
            }
            case StreamInit: {
                return DNAction.STREAM_INIT;
            }
            case FinalizeBlock: {
                return DNAction.FINALIZE_BLOCK;
            }
            case Echo: {
                return DNAction.ECHO;
            }
            case GetContainerChecksumInfo: {
                return DNAction.GET_CONTAINER_CHECKSUM_INFO;
            }
        }
        LOG.debug("Invalid command type - {}", (Object)cmdType);
        return null;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private static Map<String, String> getAuditParams(ContainerProtos.ContainerCommandRequestProto msg, DispatcherContext dispatcherContext) {
        TreeMap<String, String> auditParams = new TreeMap<String, String>();
        ContainerProtos.Type cmdType = msg.getCmdType();
        String containerID = String.valueOf(msg.getContainerID());
        switch (cmdType) {
            case CreateContainer: {
                auditParams.put(AUDIT_PARAM_CONTAINER_ID, containerID);
                auditParams.put(AUDIT_PARAM_CONTAINER_TYPE, msg.getCreateContainer().getContainerType().toString());
                return auditParams;
            }
            case ReadContainer: {
                auditParams.put(AUDIT_PARAM_CONTAINER_ID, containerID);
                return auditParams;
            }
            case UpdateContainer: {
                auditParams.put(AUDIT_PARAM_CONTAINER_ID, containerID);
                auditParams.put(AUDIT_PARAM_FORCE_UPDATE, String.valueOf(msg.getUpdateContainer().getForceUpdate()));
                return auditParams;
            }
            case DeleteContainer: {
                auditParams.put(AUDIT_PARAM_CONTAINER_ID, containerID);
                auditParams.put(AUDIT_PARAM_FORCE_DELETE, String.valueOf(msg.getDeleteContainer().getForceDelete()));
                return auditParams;
            }
            case ListContainer: {
                auditParams.put(AUDIT_PARAM_START_CONTAINER_ID, containerID);
                auditParams.put(AUDIT_PARAM_COUNT, String.valueOf(msg.getListContainer().getCount()));
                return auditParams;
            }
            case PutBlock: {
                try {
                    auditParams.put(AUDIT_PARAM_BLOCK_DATA, BlockData.getFromProtoBuf((ContainerProtos.BlockData)msg.getPutBlock().getBlockData()).toString());
                    return auditParams;
                }
                catch (IOException ex) {
                    if (!LOG.isTraceEnabled()) return null;
                    LOG.trace("Encountered error parsing BlockData from protobuf: " + ex.getMessage());
                    return null;
                }
            }
            case GetBlock: {
                auditParams.put(AUDIT_PARAM_BLOCK_DATA, BlockID.getFromProtobuf((ContainerProtos.DatanodeBlockID)msg.getGetBlock().getBlockID()).toString());
                return auditParams;
            }
            case DeleteBlock: {
                auditParams.put(AUDIT_PARAM_BLOCK_DATA, BlockID.getFromProtobuf((ContainerProtos.DatanodeBlockID)msg.getDeleteBlock().getBlockID()).toString());
                return auditParams;
            }
            case ListBlock: {
                auditParams.put(AUDIT_PARAM_START_LOCAL_ID, String.valueOf(msg.getListBlock().getStartLocalID()));
                auditParams.put(AUDIT_PARAM_COUNT, String.valueOf(msg.getListBlock().getCount()));
                return auditParams;
            }
            case ReadChunk: {
                auditParams.put(AUDIT_PARAM_BLOCK_DATA, BlockID.getFromProtobuf((ContainerProtos.DatanodeBlockID)msg.getReadChunk().getBlockID()).toString());
                auditParams.put(AUDIT_PARAM_BLOCK_DATA_OFFSET, String.valueOf(msg.getReadChunk().getChunkData().getOffset()));
                auditParams.put(AUDIT_PARAM_BLOCK_DATA_SIZE, String.valueOf(msg.getReadChunk().getChunkData().getLen()));
                return auditParams;
            }
            case DeleteChunk: {
                auditParams.put(AUDIT_PARAM_BLOCK_DATA, BlockID.getFromProtobuf((ContainerProtos.DatanodeBlockID)msg.getDeleteChunk().getBlockID()).toString());
                return auditParams;
            }
            case WriteChunk: {
                auditParams.put(AUDIT_PARAM_BLOCK_DATA, BlockID.getFromProtobuf((ContainerProtos.DatanodeBlockID)msg.getWriteChunk().getBlockID()).toString());
                auditParams.put(AUDIT_PARAM_BLOCK_DATA_OFFSET, String.valueOf(msg.getWriteChunk().getChunkData().getOffset()));
                auditParams.put(AUDIT_PARAM_BLOCK_DATA_SIZE, String.valueOf(msg.getWriteChunk().getChunkData().getLen()));
                if (dispatcherContext == null || dispatcherContext.getStage() == null) return auditParams;
                auditParams.put(AUDIT_PARAM_BLOCK_DATA_STAGE, dispatcherContext.getStage().toString());
                return auditParams;
            }
            case ListChunk: {
                auditParams.put(AUDIT_PARAM_BLOCK_DATA, BlockID.getFromProtobuf((ContainerProtos.DatanodeBlockID)msg.getListChunk().getBlockID()).toString());
                auditParams.put(AUDIT_PARAM_PREV_CHUNKNAME, msg.getListChunk().getPrevChunkName());
                auditParams.put(AUDIT_PARAM_COUNT, String.valueOf(msg.getListChunk().getCount()));
                return auditParams;
            }
            case CompactChunk: {
                return null;
            }
            case PutSmallFile: {
                try {
                    auditParams.put(AUDIT_PARAM_BLOCK_DATA, BlockData.getFromProtoBuf((ContainerProtos.BlockData)msg.getPutSmallFile().getBlock().getBlockData()).toString());
                    auditParams.put(AUDIT_PARAM_BLOCK_DATA_OFFSET, String.valueOf(msg.getPutSmallFile().getChunkInfo().getOffset()));
                    auditParams.put(AUDIT_PARAM_BLOCK_DATA_SIZE, String.valueOf(msg.getPutSmallFile().getChunkInfo().getLen()));
                    return auditParams;
                }
                catch (IOException ex) {
                    if (!LOG.isTraceEnabled()) return auditParams;
                    LOG.trace("Encountered error parsing BlockData from protobuf: " + ex.getMessage());
                }
                return auditParams;
            }
            case GetSmallFile: {
                auditParams.put(AUDIT_PARAM_BLOCK_DATA, BlockID.getFromProtobuf((ContainerProtos.DatanodeBlockID)msg.getGetSmallFile().getBlock().getBlockID()).toString());
                return auditParams;
            }
            case CloseContainer: {
                auditParams.put(AUDIT_PARAM_CONTAINER_ID, containerID);
                return auditParams;
            }
            case GetCommittedBlockLength: {
                auditParams.put(AUDIT_PARAM_BLOCK_DATA, BlockID.getFromProtobuf((ContainerProtos.DatanodeBlockID)msg.getGetCommittedBlockLength().getBlockID()).toString());
                return auditParams;
            }
            case FinalizeBlock: {
                auditParams.put(AUDIT_PARAM_BLOCK_DATA, BlockID.getFromProtobuf((ContainerProtos.DatanodeBlockID)msg.getFinalizeBlock().getBlockID()).toString());
                return auditParams;
            }
        }
        LOG.debug("Invalid command type - {}", (Object)cmdType);
        return null;
    }

    private boolean isOperationSlow(long opLatencyNs) {
        return opLatencyNs >= this.slowOpThresholdNs;
    }

    static enum EventType {
        READ,
        WRITE;

    }
}

