/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.impl;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.protobuf.Message;
import jakarta.annotation.Nullable;
import java.io.IOException;
import java.time.Clock;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.ToLongFunction;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.ozone.container.common.impl.ContainerData;
import org.apache.hadoop.ozone.container.common.impl.ContainerDataScanOrder;
import org.apache.hadoop.ozone.container.common.interfaces.Container;
import org.apache.hadoop.ozone.container.common.statemachine.StateContext;
import org.apache.hadoop.ozone.container.common.utils.ContainerLogger;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.metadata.ContainerCreateInfo;
import org.apache.hadoop.ozone.container.metadata.WitnessedContainerMetadataStore;
import org.apache.hadoop.ozone.container.ozoneimpl.OnDemandContainerScanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerSet
implements Iterable<Container<?>> {
    private static final Logger LOG = LoggerFactory.getLogger(ContainerSet.class);
    private final ConcurrentSkipListMap<Long, Container<?>> containerMap = new ConcurrentSkipListMap();
    private final ConcurrentSkipListSet<Long> missingContainerSet = new ConcurrentSkipListSet();
    private final ConcurrentSkipListMap<Long, Long> recoveringContainerMap = new ConcurrentSkipListMap();
    private final Clock clock;
    private long recoveringTimeout;
    @Nullable
    private final WitnessedContainerMetadataStore containerMetadataStore;
    private OnDemandContainerScanner containerScanner;

    public static ContainerSet newReadOnlyContainerSet(long recoveringTimeout) {
        return new ContainerSet(null, recoveringTimeout);
    }

    public static ContainerSet newRwContainerSet(WitnessedContainerMetadataStore metadataStore, long recoveringTimeout) {
        Objects.requireNonNull(metadataStore, "WitnessedContainerMetadataStore == null");
        return new ContainerSet(metadataStore, recoveringTimeout);
    }

    private ContainerSet(WitnessedContainerMetadataStore containerMetadataStore, long recoveringTimeout) {
        this(containerMetadataStore, recoveringTimeout, null);
    }

    ContainerSet(WitnessedContainerMetadataStore containerMetadataStore, long recoveringTimeout, Clock clock) {
        this.clock = clock != null ? clock : Clock.systemUTC();
        this.containerMetadataStore = containerMetadataStore;
        this.recoveringTimeout = recoveringTimeout;
    }

    public long getCurrentTime() {
        return this.clock.millis();
    }

    @Nullable
    public WitnessedContainerMetadataStore getContainerMetadataStore() {
        return this.containerMetadataStore;
    }

    @VisibleForTesting
    public void setRecoveringTimeout(long recoveringTimeout) {
        this.recoveringTimeout = recoveringTimeout;
    }

    public boolean addContainer(Container<?> container) throws StorageContainerException {
        return this.addContainer(container, false);
    }

    public boolean addContainerByOverwriteMissingContainer(Container<?> container) throws StorageContainerException {
        return this.addContainer(container, true);
    }

    public void ensureContainerNotMissing(long containerId, ContainerProtos.ContainerDataProto.State state) throws StorageContainerException {
        if (this.missingContainerSet.contains(containerId)) {
            throw new StorageContainerException(String.format("Container with container Id %d with state : %s is missing in the DN.", containerId, state), ContainerProtos.Result.CONTAINER_MISSING);
        }
    }

    public void registerOnDemandScanner(OnDemandContainerScanner scanner) {
        this.containerScanner = scanner;
    }

    public void scanContainer(long containerID, String reasonForScan) {
        if (this.containerScanner != null) {
            Container<?> container = this.getContainer(containerID);
            if (container != null) {
                this.containerScanner.scanContainer(container, reasonForScan);
            } else {
                LOG.warn("Request to scan container {} which was not found in the container set", (Object)containerID);
            }
        }
    }

    public void scanContainerWithoutGap(long containerID, String reasonForScan) {
        if (this.containerScanner != null) {
            Container<?> container = this.getContainer(containerID);
            if (container != null) {
                this.containerScanner.scanContainerWithoutGap(container, reasonForScan);
            } else {
                LOG.warn("Request to scan container {} which was not found in the container set", (Object)containerID);
            }
        }
    }

    private boolean addContainer(Container<?> container, boolean overwrite) throws StorageContainerException {
        Preconditions.checkNotNull(container, (Object)"container cannot be null");
        long containerId = ((ContainerData)container.getContainerData()).getContainerID();
        ContainerProtos.ContainerDataProto.State containerState = ((ContainerData)container.getContainerData()).getState();
        if (!overwrite) {
            this.ensureContainerNotMissing(containerId, containerState);
        }
        if (this.containerMap.putIfAbsent(containerId, container) == null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Container with container Id {} is added to containerMap", (Object)containerId);
            }
            this.updateContainerIdTable(containerId, (ContainerData)container.getContainerData());
            this.missingContainerSet.remove(containerId);
            if (((ContainerData)container.getContainerData()).getState() == ContainerProtos.ContainerDataProto.State.RECOVERING) {
                this.recoveringContainerMap.put(this.clock.millis() + this.recoveringTimeout, containerId);
            }
            return true;
        }
        LOG.warn("Container already exists with container Id {}", (Object)containerId);
        throw new StorageContainerException("Container already exists with container Id " + containerId, ContainerProtos.Result.CONTAINER_EXISTS);
    }

    private void updateContainerIdTable(long containerId, ContainerData containerData) throws StorageContainerException {
        if (null != this.containerMetadataStore) {
            try {
                ContainerID containerIdObj = ContainerID.valueOf((long)containerId);
                Table<ContainerID, ContainerCreateInfo> containerCreateInfoTable = this.containerMetadataStore.getContainerCreateInfoTable();
                ContainerCreateInfo containerCreateInfo = (ContainerCreateInfo)containerCreateInfoTable.get((Object)containerIdObj);
                if (containerCreateInfo == null || containerCreateInfo.getReplicaIndex() == -1) {
                    containerCreateInfoTable.put((Object)containerIdObj, (Object)ContainerCreateInfo.valueOf(containerData.getState(), containerData.getReplicaIndex()));
                }
            }
            catch (IOException e) {
                throw new StorageContainerException((Throwable)e, ContainerProtos.Result.IO_EXCEPTION);
            }
        }
    }

    public Container<?> getContainer(long containerId) {
        Preconditions.checkState((containerId >= 0L ? 1 : 0) != 0, (Object)"Container Id cannot be negative.");
        return this.containerMap.get(containerId);
    }

    public boolean removeContainer(long containerId) throws StorageContainerException {
        return this.removeContainer(containerId, false, true);
    }

    public boolean removeContainerOnlyFromMemory(long containerId) throws StorageContainerException {
        return this.removeContainer(containerId, false, false);
    }

    public boolean removeMissingContainer(long containerId) throws StorageContainerException {
        return this.removeContainer(containerId, true, false);
    }

    private boolean removeContainer(long containerId, boolean markMissing, boolean removeFromDB) throws StorageContainerException {
        Preconditions.checkState((containerId >= 0L ? 1 : 0) != 0, (Object)"Container Id cannot be negative.");
        if (markMissing) {
            this.missingContainerSet.add(containerId);
        }
        Container<?> removed = this.containerMap.remove(containerId);
        if (removeFromDB) {
            this.deleteFromContainerTable(containerId);
        }
        if (removed == null) {
            LOG.debug("Container with containerId {} is not present in containerMap", (Object)containerId);
            return false;
        }
        LOG.debug("Container with containerId {} is removed from containerMap", (Object)containerId);
        return true;
    }

    private void deleteFromContainerTable(long containerId) throws StorageContainerException {
        if (null != this.containerMetadataStore) {
            try {
                this.containerMetadataStore.getContainerCreateInfoTable().delete((Object)ContainerID.valueOf((long)containerId));
            }
            catch (IOException e) {
                throw new StorageContainerException((Throwable)e, ContainerProtos.Result.IO_EXCEPTION);
            }
        }
    }

    public boolean removeRecoveringContainer(long containerId) {
        Preconditions.checkState((containerId >= 0L ? 1 : 0) != 0, (Object)"Container Id cannot be negative.");
        Iterator<Map.Entry<Long, Long>> it = this.getRecoveringContainerIterator();
        while (it.hasNext()) {
            Map.Entry<Long, Long> entry = it.next();
            if (entry.getValue() != containerId) continue;
            it.remove();
            return true;
        }
        return false;
    }

    @VisibleForTesting
    public int containerCount() {
        return this.containerMap.size();
    }

    public void handleVolumeFailures(StateContext context) throws StorageContainerException {
        AtomicBoolean failedVolume = new AtomicBoolean(false);
        AtomicInteger containerCount = new AtomicInteger(0);
        for (Container<?> c : this.containerMap.values()) {
            Object data = c.getContainerData();
            if (!((ContainerData)data).getVolume().isFailed()) continue;
            this.removeMissingContainer(((ContainerData)data).getContainerID());
            LOG.debug("Removing Container {} as the Volume {} has failed", (Object)((ContainerData)data).getContainerID(), (Object)((ContainerData)data).getVolume());
            failedVolume.set(true);
            containerCount.incrementAndGet();
            ContainerLogger.logLost(data, "Volume failure");
        }
        if (failedVolume.get()) {
            try {
                LOG.info("Removed {} containers on failed volumes", (Object)containerCount.get());
                StorageContainerDatanodeProtocolProtos.ContainerReportsProto report = context.getFullContainerReportDiscardPendingICR();
                context.refreshFullReport((Message)report);
                context.getParent().triggerHeartbeat();
            }
            catch (Exception e) {
                LOG.error("Failed to send FCR in Volume failure", (Throwable)e);
            }
        }
    }

    @Override
    public Iterator<Container<?>> iterator() {
        return this.containerMap.values().iterator();
    }

    public Iterator<Map.Entry<Long, Long>> getRecoveringContainerIterator() {
        return this.recoveringContainerMap.entrySet().iterator();
    }

    public Iterator<Container<?>> getContainerIterator(HddsVolume volume) {
        Preconditions.checkNotNull((Object)volume);
        Preconditions.checkNotNull((Object)volume.getStorageID());
        String volumeUuid = volume.getStorageID();
        return this.containerMap.values().stream().filter(x -> volumeUuid.equals(((ContainerData)x.getContainerData()).getVolume().getStorageID())).sorted(ContainerDataScanOrder.INSTANCE).iterator();
    }

    public long containerCount(HddsVolume volume) {
        Preconditions.checkNotNull((Object)volume);
        Preconditions.checkNotNull((Object)volume.getStorageID());
        String volumeUuid = volume.getStorageID();
        return this.containerMap.values().stream().filter(x -> volumeUuid.equals(((ContainerData)x.getContainerData()).getVolume().getStorageID())).count();
    }

    public Iterator<Map.Entry<Long, Container<?>>> getContainerMapIterator() {
        return this.containerMap.entrySet().iterator();
    }

    @VisibleForTesting
    public Map<Long, Container<?>> getContainerMapCopy() {
        return ImmutableMap.copyOf(this.containerMap);
    }

    public Map<Long, Container<?>> getContainerMap() {
        return Collections.unmodifiableMap(this.containerMap);
    }

    public void listContainer(long startContainerId, long count, List<ContainerData> data) throws StorageContainerException {
        Preconditions.checkNotNull(data, (Object)"Internal assertion: data cannot be null");
        Preconditions.checkState((startContainerId >= 0L ? 1 : 0) != 0, (Object)"Start container Id cannot be negative");
        Preconditions.checkState((count > 0L ? 1 : 0) != 0, (Object)"max number of containers returned must be positive");
        LOG.debug("listContainer returns containerData starting from {} of count {}", (Object)startContainerId, (Object)count);
        NavigableMap<Long, Container<Object>> map = startContainerId == 0L ? this.containerMap : this.containerMap.tailMap((Object)startContainerId, true);
        int currentCount = 0;
        for (Container entry : map.values()) {
            if ((long)currentCount < count) {
                data.add((ContainerData)entry.getContainerData());
                ++currentCount;
                continue;
            }
            return;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public StorageContainerDatanodeProtocolProtos.ContainerReportsProto getContainerReport() throws IOException {
        LOG.debug("Starting container report iteration.");
        StorageContainerDatanodeProtocolProtos.ContainerReportsProto.Builder crBuilder = StorageContainerDatanodeProtocolProtos.ContainerReportsProto.newBuilder();
        ArrayList containers = new ArrayList(this.containerMap.values());
        ContainerSet containerSet = this;
        synchronized (containerSet) {
            for (Container container : containers) {
                if (container.getContainerState() == ContainerProtos.ContainerDataProto.State.RECOVERING) continue;
                crBuilder.addReports(container.getContainerReport());
            }
        }
        return crBuilder.build();
    }

    public Set<Long> getMissingContainerSet() {
        return this.missingContainerSet;
    }

    public <T> void buildMissingContainerSetAndValidate(Map<T, Long> container2BCSIDMap, ToLongFunction<T> getId) {
        container2BCSIDMap.entrySet().parallelStream().forEach(mapEntry -> {
            long id = getId.applyAsLong(mapEntry.getKey());
            if (!this.containerMap.containsKey(id)) {
                LOG.warn("Adding container {} to missing container set.", (Object)id);
                this.missingContainerSet.add(id);
            } else {
                long snapshotBCSID;
                Container<?> container = this.containerMap.get(id);
                long containerBCSID = container.getBlockCommitSequenceId();
                if (containerBCSID < (snapshotBCSID = ((Long)mapEntry.getValue()).longValue())) {
                    LOG.warn("Marking container {} unhealthy as reported BCSID {} is smaller than ratis snapshot recorded value {}", new Object[]{id, containerBCSID, snapshotBCSID});
                    try {
                        container.markContainerUnhealthy();
                    }
                    catch (StorageContainerException sce) {
                        LOG.error("Unable to persist unhealthy state for container {}", (Object)id);
                    }
                }
            }
        });
    }
}

