/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.impl;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.conf.ReconfigurableConfig;
import org.apache.hadoop.hdds.conf.ReconfigurationHandler;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.utils.BackgroundService;
import org.apache.hadoop.hdds.utils.BackgroundTask;
import org.apache.hadoop.hdds.utils.BackgroundTaskQueue;
import org.apache.hadoop.ozone.container.checksum.ContainerChecksumTreeManager;
import org.apache.hadoop.ozone.container.common.helpers.BlockDeletingServiceMetrics;
import org.apache.hadoop.ozone.container.common.helpers.ContainerUtils;
import org.apache.hadoop.ozone.container.common.impl.ContainerData;
import org.apache.hadoop.ozone.container.common.impl.TopNOrderedContainerDeletionChoosingPolicy;
import org.apache.hadoop.ozone.container.common.interfaces.Container;
import org.apache.hadoop.ozone.container.common.interfaces.ContainerDeletionChoosingPolicy;
import org.apache.hadoop.ozone.container.common.statemachine.DatanodeConfiguration;
import org.apache.hadoop.ozone.container.common.transport.server.ratis.XceiverServerRatis;
import org.apache.hadoop.ozone.container.keyvalue.statemachine.background.BlockDeletingTask;
import org.apache.hadoop.ozone.container.ozoneimpl.OzoneContainer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlockDeletingService
extends BackgroundService {
    private static final Logger LOG = LoggerFactory.getLogger(BlockDeletingService.class);
    private final OzoneContainer ozoneContainer;
    private final ContainerDeletionChoosingPolicy containerDeletionPolicy;
    private final ConfigurationSource conf;
    private final DatanodeConfiguration dnConf;
    private final BlockDeletingServiceMetrics metrics;
    private static final int TASK_PRIORITY_DEFAULT = 1;
    private final Duration blockDeletingMaxLockHoldingTime;
    private final ContainerChecksumTreeManager checksumTreeManager;

    @VisibleForTesting
    public BlockDeletingService(OzoneContainer ozoneContainer, long serviceInterval, long serviceTimeout, TimeUnit timeUnit, int workerSize, ConfigurationSource conf, ContainerChecksumTreeManager checksumTreeManager) {
        this(ozoneContainer, serviceInterval, serviceTimeout, timeUnit, workerSize, conf, "", checksumTreeManager, null);
    }

    public BlockDeletingService(OzoneContainer ozoneContainer, long serviceInterval, long serviceTimeout, TimeUnit timeUnit, int workerSize, ConfigurationSource conf, String threadNamePrefix, ContainerChecksumTreeManager checksumTreeManager, ReconfigurationHandler reconfigurationHandler) {
        super("BlockDeletingService", serviceInterval, timeUnit, workerSize, serviceTimeout, threadNamePrefix);
        this.ozoneContainer = ozoneContainer;
        this.checksumTreeManager = checksumTreeManager;
        try {
            this.containerDeletionPolicy = (ContainerDeletionChoosingPolicy)conf.getClass("ozone.scm.keyvalue.container.deletion-choosing.policy", TopNOrderedContainerDeletionChoosingPolicy.class, ContainerDeletionChoosingPolicy.class).newInstance();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        this.conf = conf;
        this.dnConf = (DatanodeConfiguration)((Object)conf.getObject(DatanodeConfiguration.class));
        if (reconfigurationHandler != null) {
            reconfigurationHandler.register((ReconfigurableConfig)this.dnConf);
            this.registerReconfigCallbacks(reconfigurationHandler);
        }
        this.blockDeletingMaxLockHoldingTime = this.dnConf.getBlockDeletingMaxLockHoldingTime();
        this.metrics = BlockDeletingServiceMetrics.create();
    }

    public void registerReconfigCallbacks(ReconfigurationHandler handler) {
        handler.registerCompleteCallback((changedKeys, newConf) -> {
            if (changedKeys.containsKey("ozone.block.deleting.service.interval") || changedKeys.containsKey("ozone.block.deleting.service.timeout") || changedKeys.containsKey("ozone.block.deleting.service.workers")) {
                this.updateAndRestart((OzoneConfiguration)newConf);
            }
        });
    }

    public synchronized void updateAndRestart(OzoneConfiguration ozoneConf) {
        long newInterval = ozoneConf.getTimeDuration("ozone.block.deleting.service.interval", "60s", TimeUnit.SECONDS);
        int newCorePoolSize = ozoneConf.getInt("ozone.block.deleting.service.workers", 10);
        long newTimeout = ozoneConf.getTimeDuration("ozone.block.deleting.service.timeout", "300s", TimeUnit.NANOSECONDS);
        LOG.info("Updating and restarting BlockDeletingService with interval {} {}, core pool size {} and timeout {} {}", new Object[]{newInterval, TimeUnit.SECONDS.name().toLowerCase(), newCorePoolSize, newTimeout, TimeUnit.NANOSECONDS.name().toLowerCase()});
        this.shutdown();
        this.setInterval(newInterval, TimeUnit.SECONDS);
        this.setPoolSize(newCorePoolSize);
        this.setServiceTimeoutInNanos(newTimeout);
        this.start();
    }

    public BackgroundTaskQueue getTasks() {
        BackgroundTaskQueue queue = new BackgroundTaskQueue();
        try {
            List<ContainerBlockInfo> containers = this.chooseContainerForBlockDeletion(this.getBlockLimitPerInterval(), this.containerDeletionPolicy);
            BackgroundTask containerBlockInfos = null;
            long totalBlocks = 0L;
            for (ContainerBlockInfo containerBlockInfo : containers) {
                BlockDeletingTaskBuilder builder = new BlockDeletingTaskBuilder();
                builder.setBlockDeletingService(this).setContainerBlockInfo(containerBlockInfo).setChecksumTreeManager(this.checksumTreeManager).setPriority(1);
                containerBlockInfos = builder.build();
                queue.add(containerBlockInfos);
                totalBlocks += containerBlockInfo.getNumBlocksToDelete().longValue();
                LOG.debug("Queued- Container: {}, deleted blocks: {}", (Object)containerBlockInfo.getContainerData().getContainerID(), (Object)containerBlockInfo.getNumBlocksToDelete());
            }
            this.metrics.incrTotalBlockChosenCount(totalBlocks);
            this.metrics.incrTotalContainerChosenCount(containers.size());
        }
        catch (StorageContainerException e) {
            LOG.warn("Failed to initiate block deleting tasks, caused by unable to get containers info. Retry in next interval. ", (Throwable)e);
        }
        catch (Exception e) {
            LOG.error("Unexpected error occurs during deleting blocks.", (Throwable)e);
        }
        return queue;
    }

    public List<ContainerBlockInfo> chooseContainerForBlockDeletion(int blockLimit, ContainerDeletionChoosingPolicy deletionPolicy) throws StorageContainerException {
        AtomicLong totalPendingBlockCount = new AtomicLong(0L);
        Map<Long, ContainerData> containerDataMap = this.ozoneContainer.getContainerSet().getContainerMap().entrySet().stream().filter(e -> this.checkPendingDeletionBlocks((ContainerData)((Container)e.getValue()).getContainerData())).filter(e -> this.isDeletionAllowed((ContainerData)((Container)e.getValue()).getContainerData(), deletionPolicy)).collect(Collectors.toMap(Map.Entry::getKey, e -> {
            Object containerData = ((Container)e.getValue()).getContainerData();
            totalPendingBlockCount.addAndGet(ContainerUtils.getPendingDeletionBlocks(containerData));
            return containerData;
        }));
        this.metrics.setTotalPendingBlockCount(totalPendingBlockCount.get());
        return deletionPolicy.chooseContainerForBlockDeletion(blockLimit, containerDataMap);
    }

    private boolean checkPendingDeletionBlocks(ContainerData containerData) {
        return ContainerUtils.getPendingDeletionBlocks(containerData) > 0L;
    }

    private boolean isDeletionAllowed(ContainerData containerData, ContainerDeletionChoosingPolicy deletionPolicy) {
        if (!deletionPolicy.isValidContainerType(containerData.getContainerType())) {
            LOG.debug("Container with type {} is not valid for block deletion.", (Object)containerData.getContainerType());
            return false;
        }
        if (!containerData.isClosed() && !containerData.isQuasiClosed()) {
            LOG.info("Skipping block deletion for container {}. State: {} (only CLOSED or QUASI_CLOSED are allowed).", (Object)containerData.getContainerID(), (Object)containerData.getState());
            return false;
        }
        if (this.ozoneContainer.getWriteChannel() instanceof XceiverServerRatis) {
            PipelineID pipelineID;
            XceiverServerRatis ratisServer = (XceiverServerRatis)this.ozoneContainer.getWriteChannel();
            String originPipelineId = containerData.getOriginPipelineId();
            if (originPipelineId == null || originPipelineId.isEmpty()) {
                return true;
            }
            try {
                pipelineID = PipelineID.valueOf((String)originPipelineId);
            }
            catch (IllegalArgumentException e) {
                LOG.warn("Invalid pipelineID {} for container {}", (Object)originPipelineId, (Object)containerData.getContainerID());
                return false;
            }
            if (!ratisServer.isExist(pipelineID.getProtobuf())) {
                return true;
            }
            try {
                long minReplicatedIndex = ratisServer.getMinReplicatedIndex(pipelineID);
                long containerBCSID = containerData.getBlockCommitSequenceId();
                if (minReplicatedIndex < containerBCSID) {
                    LOG.warn("Close Container log Index {} is not replicated across all servers in the pipeline {} (min replicated index {}). Deletion is not allowed yet.", new Object[]{containerBCSID, containerData.getOriginPipelineId(), minReplicatedIndex});
                    return false;
                }
                return true;
            }
            catch (IOException ioe) {
                if (!ratisServer.isExist(pipelineID.getProtobuf())) {
                    return true;
                }
                LOG.info("Skipping deletes for container {} due to exception: {}", (Object)containerData.getContainerID(), (Object)ioe.getMessage());
                return false;
            }
        }
        return true;
    }

    public OzoneContainer getOzoneContainer() {
        return this.ozoneContainer;
    }

    public ConfigurationSource getConf() {
        return this.conf;
    }

    public BlockDeletingServiceMetrics getMetrics() {
        return this.metrics;
    }

    public Duration getBlockDeletingMaxLockHoldingTime() {
        return this.blockDeletingMaxLockHoldingTime;
    }

    public int getBlockLimitPerInterval() {
        return this.dnConf.getBlockDeletionLimit();
    }

    public static class ContainerBlockInfo {
        private final ContainerData containerData;
        private final Long numBlocksToDelete;

        public ContainerBlockInfo(ContainerData containerData, Long blocks) {
            this.containerData = containerData;
            this.numBlocksToDelete = blocks;
        }

        public ContainerData getContainerData() {
            return this.containerData;
        }

        public Long getNumBlocksToDelete() {
            return this.numBlocksToDelete;
        }
    }

    private static class BlockDeletingTaskBuilder {
        private BlockDeletingService blockDeletingService;
        private ContainerBlockInfo containerBlockInfo;
        private int priority;
        private ContainerChecksumTreeManager checksumTreeManager;

        private BlockDeletingTaskBuilder() {
        }

        public BlockDeletingTaskBuilder setBlockDeletingService(BlockDeletingService blockDeletingService) {
            this.blockDeletingService = blockDeletingService;
            return this;
        }

        public BlockDeletingTaskBuilder setContainerBlockInfo(ContainerBlockInfo containerBlockInfo) {
            this.containerBlockInfo = containerBlockInfo;
            return this;
        }

        public BlockDeletingTaskBuilder setChecksumTreeManager(ContainerChecksumTreeManager treeManager) {
            this.checksumTreeManager = treeManager;
            return this;
        }

        public BlockDeletingTaskBuilder setPriority(int priority) {
            this.priority = priority;
            return this;
        }

        public BackgroundTask build() {
            ContainerProtos.ContainerType containerType = this.containerBlockInfo.getContainerData().getContainerType();
            if (containerType.equals((Object)ContainerProtos.ContainerType.KeyValueContainer)) {
                return new BlockDeletingTask(this.blockDeletingService, this.containerBlockInfo, this.checksumTreeManager, this.priority);
            }
            throw new IllegalArgumentException("BlockDeletingTask for ContainerType: " + containerType + "doesn't exist.");
        }
    }
}

