/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.keyvalue;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.List;
import java.util.function.BooleanSupplier;
import java.util.stream.Stream;
import org.apache.commons.io.FileUtils;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.server.JsonUtils;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.hdds.utils.db.TableIterator;
import org.apache.hadoop.ozone.container.common.helpers.BlockData;
import org.apache.hadoop.ozone.container.common.impl.ContainerData;
import org.apache.hadoop.ozone.container.common.interfaces.BlockIterator;
import org.apache.hadoop.ozone.container.common.interfaces.ContainerInspector;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.container.keyvalue.helpers.KeyValueContainerUtil;
import org.apache.hadoop.ozone.container.metadata.DatanodeStore;
import org.apache.hadoop.ozone.container.metadata.DatanodeStoreSchemaThreeImpl;
import org.apache.hadoop.ozone.container.metadata.DatanodeStoreSchemaTwoImpl;
import org.apache.hadoop.ozone.container.metadata.DatanodeStoreWithIncrementalChunkList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeyValueContainerMetadataInspector
implements ContainerInspector {
    public static final Logger LOG = LoggerFactory.getLogger(KeyValueContainerMetadataInspector.class);
    public static final Logger REPORT_LOG = LoggerFactory.getLogger((String)"ContainerMetadataInspectorReport");
    public static final String SYSTEM_PROPERTY = "ozone.datanode.container.metadata.inspector";
    private Mode mode;

    public KeyValueContainerMetadataInspector(Mode mode) {
        this.mode = mode;
    }

    public KeyValueContainerMetadataInspector() {
        this.mode = Mode.OFF;
    }

    @Override
    public boolean load() {
        String propertyValue = System.getProperty(SYSTEM_PROPERTY);
        boolean propertyPresent = propertyValue != null && !propertyValue.isEmpty();
        boolean propertyValid = false;
        if (propertyPresent) {
            if (propertyValue.equals(Mode.REPAIR.toString())) {
                this.mode = Mode.REPAIR;
                propertyValid = true;
            } else if (propertyValue.equals(Mode.INSPECT.toString())) {
                this.mode = Mode.INSPECT;
                propertyValid = true;
            }
            if (propertyValid) {
                LOG.info("Container metadata inspector enabled in {} mode. Reportwill be output to the {} log.", (Object)this.mode, (Object)REPORT_LOG.getName());
            } else {
                this.mode = Mode.OFF;
                LOG.error("{} system property specified with invalid mode {}. Valid options are {} and {}. Container metadata inspection will not be run.", new Object[]{SYSTEM_PROPERTY, propertyValue, Mode.REPAIR, Mode.INSPECT});
            }
        } else {
            this.mode = Mode.OFF;
        }
        return propertyPresent && propertyValid;
    }

    @Override
    public void unload() {
        this.mode = Mode.OFF;
    }

    @Override
    public boolean isReadOnly() {
        return this.mode != Mode.REPAIR;
    }

    @Override
    public void process(ContainerData containerData, DatanodeStore store) {
        this.process(containerData, store, REPORT_LOG);
    }

    public String process(ContainerData containerData, DatanodeStore store, Logger log) {
        if (this.mode == Mode.OFF) {
            return null;
        }
        KeyValueContainerData kvData = null;
        if (!(containerData instanceof KeyValueContainerData)) {
            LOG.error("This inspector only works on KeyValueContainers. Inspection will not be run for container {}", (Object)containerData.getContainerID());
            return null;
        }
        kvData = (KeyValueContainerData)containerData;
        ObjectNode containerJson = KeyValueContainerMetadataInspector.inspectContainer(kvData, store);
        boolean correct = this.checkAndRepair(containerJson, kvData, store);
        String jsonReport = JsonUtils.toJsonStringWIthIndent((Object)containerJson);
        if (log != null) {
            if (correct) {
                log.trace(jsonReport);
            } else {
                log.error(jsonReport);
            }
        }
        return jsonReport;
    }

    static ObjectNode inspectContainer(KeyValueContainerData containerData, DatanodeStore store) {
        ObjectNode containerJson = JsonUtils.createObjectNode(null);
        try {
            containerJson.put("containerID", containerData.getContainerID());
            String schemaVersion = containerData.getSchemaVersion();
            containerJson.put("schemaVersion", schemaVersion);
            containerJson.put("containerState", containerData.getState().toString());
            containerJson.put("currentDatanodeID", containerData.getVolume().getDatanodeUuid());
            containerJson.put("originDatanodeID", containerData.getOriginNodeId());
            ObjectNode dBMetadata = KeyValueContainerMetadataInspector.getDBMetadataJson(store.getMetadataTable(), containerData);
            containerJson.set("dBMetadata", (JsonNode)dBMetadata);
            ObjectNode aggregates = KeyValueContainerMetadataInspector.getAggregateValues(store, containerData, schemaVersion);
            containerJson.set("aggregates", (JsonNode)aggregates);
            ObjectNode chunksDirectory = KeyValueContainerMetadataInspector.getChunksDirectoryJson(new File(containerData.getChunksPath()));
            containerJson.set("chunksDirectory", (JsonNode)chunksDirectory);
        }
        catch (IOException ex) {
            LOG.error("Inspecting container {} failed", (Object)containerData.getContainerID(), (Object)ex);
        }
        return containerJson;
    }

    static ObjectNode getDBMetadataJson(Table<String, Long> metadataTable, KeyValueContainerData containerData) throws IOException {
        ObjectNode dBMetadata = JsonUtils.createObjectNode(null);
        dBMetadata.put("#BLOCKCOUNT", (Long)metadataTable.get((Object)containerData.getBlockCountKey()));
        dBMetadata.put("#BYTESUSED", (Long)metadataTable.get((Object)containerData.getBytesUsedKey()));
        dBMetadata.put("#PENDINGDELETEBLOCKCOUNT", (Long)metadataTable.get((Object)containerData.getPendingDeleteBlockCountKey()));
        dBMetadata.put("#delTX", (Long)metadataTable.get((Object)containerData.getLatestDeleteTxnKey()));
        dBMetadata.put("#BCSID", (Long)metadataTable.get((Object)containerData.getBcsIdKey()));
        return dBMetadata;
    }

    static ObjectNode getAggregateValues(DatanodeStore store, KeyValueContainerData containerData, String schemaVersion) throws IOException {
        PendingDelete pendingDelete;
        ObjectNode aggregates = JsonUtils.createObjectNode(null);
        long usedBytesTotal = 0L;
        long blockCountTotal = 0L;
        try (BlockIterator<BlockData> blockIter = store.getBlockIterator(containerData.getContainerID(), containerData.getUnprefixedKeyFilter());){
            while (blockIter.hasNext()) {
                ++blockCountTotal;
                usedBytesTotal += KeyValueContainerMetadataInspector.getBlockLength(blockIter.nextBlock());
            }
        }
        if (KeyValueContainerUtil.isSameSchemaVersion(schemaVersion, "1")) {
            long pendingDeleteBlockCountTotal = 0L;
            long pendingDeleteBytes = 0L;
            try (BlockIterator<BlockData> blockIter = store.getBlockIterator(containerData.getContainerID(), containerData.getDeletingBlockKeyFilter());){
                while (blockIter.hasNext()) {
                    ++blockCountTotal;
                    ++pendingDeleteBlockCountTotal;
                    long bytes = KeyValueContainerMetadataInspector.getBlockLength(blockIter.nextBlock());
                    usedBytesTotal += bytes;
                    pendingDeleteBytes += bytes;
                }
            }
            pendingDelete = new PendingDelete(pendingDeleteBlockCountTotal, pendingDeleteBytes);
        } else if (KeyValueContainerUtil.isSameSchemaVersion(schemaVersion, "2")) {
            DatanodeStoreSchemaTwoImpl schemaTwoStore = (DatanodeStoreSchemaTwoImpl)store;
            pendingDelete = KeyValueContainerMetadataInspector.countPendingDeletesSchemaV2(schemaTwoStore, containerData);
        } else if (KeyValueContainerUtil.isSameSchemaVersion(schemaVersion, "3")) {
            DatanodeStoreSchemaThreeImpl schemaThreeStore = (DatanodeStoreSchemaThreeImpl)store;
            pendingDelete = KeyValueContainerMetadataInspector.countPendingDeletesSchemaV3(schemaThreeStore, containerData);
        } else {
            throw new IOException("Failed to process deleted blocks for unknown container schema " + schemaVersion);
        }
        aggregates.put("blockCount", blockCountTotal);
        aggregates.put("usedBytes", usedBytesTotal);
        pendingDelete.addToJson(aggregates);
        return aggregates;
    }

    static ObjectNode getChunksDirectoryJson(File chunksDir) throws IOException {
        ObjectNode chunksDirectory = JsonUtils.createObjectNode(null);
        chunksDirectory.put("path", chunksDir.getAbsolutePath());
        boolean chunksDirPresent = FileUtils.isDirectory((File)chunksDir, (LinkOption[])new LinkOption[0]);
        chunksDirectory.put("present", chunksDirPresent);
        long fileCount = 0L;
        if (chunksDirPresent) {
            try (Stream<Path> stream = Files.list(chunksDir.toPath());){
                fileCount = stream.count();
            }
        }
        chunksDirectory.put("fileCount", fileCount);
        return chunksDirectory;
    }

    private boolean checkAndRepair(ObjectNode parent, KeyValueContainerData containerData, DatanodeStore store) {
        JsonNode chunksDirPresent;
        JsonNode pendingDeleteCountAggregate;
        long deleteTransactionCount;
        JsonNode pendingDeleteCountDB;
        long dbDeleteCount;
        long blockCountDBLong;
        ArrayNode errors = JsonUtils.createArrayNode();
        boolean passed = true;
        Table<String, Long> metadataTable = store.getMetadataTable();
        ObjectNode dBMetadata = (ObjectNode)parent.get("dBMetadata");
        ObjectNode aggregates = (ObjectNode)parent.get("aggregates");
        JsonNode blockCountDB = dBMetadata.get("#BLOCKCOUNT");
        JsonNode blockCountAggregate = aggregates.get("blockCount");
        long l = blockCountDBLong = blockCountDB.isNull() ? 0L : blockCountDB.asLong();
        if (blockCountDBLong != blockCountAggregate.asLong()) {
            passed = false;
            BooleanSupplier keyRepairAction = () -> {
                boolean repaired = false;
                try {
                    metadataTable.put((Object)containerData.getBlockCountKey(), (Object)blockCountAggregate.asLong());
                    repaired = true;
                }
                catch (IOException ex) {
                    LOG.error("Error repairing block count for container {}.", (Object)containerData.getContainerID(), (Object)ex);
                }
                return repaired;
            };
            ObjectNode blockCountError = this.buildErrorAndRepair("dBMetadata.#BLOCKCOUNT", blockCountAggregate, blockCountDB, keyRepairAction);
            errors.add((JsonNode)blockCountError);
        }
        JsonNode usedBytesDB = parent.path("dBMetadata").path("#BYTESUSED");
        JsonNode usedBytesAggregate = parent.path("aggregates").path("usedBytes");
        long usedBytesDBLong = 0L;
        if (!usedBytesDB.isNull()) {
            usedBytesDBLong = usedBytesDB.asLong();
        }
        if (usedBytesDBLong != usedBytesAggregate.asLong()) {
            passed = false;
            BooleanSupplier keyRepairAction = () -> {
                boolean repaired = false;
                try {
                    metadataTable.put((Object)containerData.getBytesUsedKey(), (Object)usedBytesAggregate.asLong());
                    repaired = true;
                }
                catch (IOException ex) {
                    LOG.error("Error repairing used bytes for container {}.", (Object)containerData.getContainerID(), (Object)ex);
                }
                return repaired;
            };
            ObjectNode usedBytesError = this.buildErrorAndRepair("dBMetadata.#BYTESUSED", usedBytesAggregate, usedBytesDB, keyRepairAction);
            errors.add((JsonNode)usedBytesError);
        }
        if ((dbDeleteCount = KeyValueContainerMetadataInspector.jsonToLong(pendingDeleteCountDB = dBMetadata.path("#PENDINGDELETEBLOCKCOUNT"))) != (deleteTransactionCount = KeyValueContainerMetadataInspector.jsonToLong(pendingDeleteCountAggregate = aggregates.path("pendingDeleteBlocks")))) {
            passed = false;
            BooleanSupplier deleteCountRepairAction = () -> {
                String key = containerData.getPendingDeleteBlockCountKey();
                try {
                    metadataTable.put((Object)key, (Object)deleteTransactionCount);
                    return true;
                }
                catch (IOException ex) {
                    LOG.error("Failed to reset {} for container {}.", new Object[]{key, containerData.getContainerID(), ex});
                    return false;
                }
            };
            ObjectNode deleteCountError = this.buildErrorAndRepair("dBMetadata.#PENDINGDELETEBLOCKCOUNT", pendingDeleteCountAggregate, pendingDeleteCountDB, deleteCountRepairAction);
            errors.add((JsonNode)deleteCountError);
        }
        if (!(chunksDirPresent = parent.path("chunksDirectory").path("present")).asBoolean()) {
            passed = false;
            BooleanSupplier dirRepairAction = () -> {
                boolean repaired = false;
                try {
                    File chunksDir = new File(containerData.getChunksPath());
                    Files.createDirectories(chunksDir.toPath(), new FileAttribute[0]);
                    repaired = true;
                }
                catch (IOException ex) {
                    LOG.error("Error recreating empty chunks directory for container {}.", (Object)containerData.getContainerID(), (Object)ex);
                }
                return repaired;
            };
            ObjectNode chunksDirError = this.buildErrorAndRepair("chunksDirectory.present", (JsonNode)JsonNodeFactory.instance.booleanNode(true), chunksDirPresent, dirRepairAction);
            errors.add((JsonNode)chunksDirError);
        }
        parent.put("correct", passed);
        parent.set("errors", (JsonNode)errors);
        return passed;
    }

    private static long jsonToLong(JsonNode e) {
        return e == null || e.isNull() ? 0L : e.asLong();
    }

    private ObjectNode buildErrorAndRepair(String property, JsonNode expected, JsonNode actual, BooleanSupplier repairAction) {
        ObjectNode error = JsonUtils.createObjectNode(null);
        error.put("property", property);
        error.set("expected", expected);
        error.set("actual", actual);
        boolean repaired = false;
        if (this.mode == Mode.REPAIR) {
            repaired = repairAction.getAsBoolean();
        }
        error.put("repaired", repaired);
        return error;
    }

    static PendingDelete countPendingDeletesSchemaV2(DatanodeStoreSchemaTwoImpl schemaTwoStore, KeyValueContainerData containerData) throws IOException {
        long pendingDeleteBlockCountTotal = 0L;
        long pendingDeleteBytes = 0L;
        Table<Long, StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction> delTxTable = schemaTwoStore.getDeleteTransactionTable();
        try (TableIterator iterator = delTxTable.iterator();){
            while (iterator.hasNext()) {
                StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction txn = (StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction)((Table.KeyValue)iterator.next()).getValue();
                List localIDs = txn.getLocalIDList();
                pendingDeleteBlockCountTotal += (long)localIDs.size();
                pendingDeleteBytes += KeyValueContainerMetadataInspector.computePendingDeleteBytes(localIDs, containerData, schemaTwoStore);
            }
        }
        return new PendingDelete(pendingDeleteBlockCountTotal, pendingDeleteBytes);
    }

    static long computePendingDeleteBytes(List<Long> localIDs, KeyValueContainerData containerData, DatanodeStoreWithIncrementalChunkList store) {
        long pendingDeleteBytes = 0L;
        for (long id : localIDs) {
            try {
                String blockKey = containerData.getBlockKey(id);
                BlockData blockData = store.getBlockByID(null, blockKey);
                if (blockData == null) continue;
                pendingDeleteBytes += blockData.getSize();
            }
            catch (IOException e) {
                LOG.error("Failed to get block " + id + " in container " + containerData.getContainerID() + " from blockDataTable", (Throwable)e);
            }
        }
        return pendingDeleteBytes;
    }

    static PendingDelete countPendingDeletesSchemaV3(DatanodeStoreSchemaThreeImpl store, KeyValueContainerData containerData) throws IOException {
        long pendingDeleteBlockCountTotal = 0L;
        long pendingDeleteBytes = 0L;
        try (TableIterator iter = store.getDeleteTransactionTable().iterator((Object)containerData.containerPrefix());){
            while (iter.hasNext()) {
                StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction delTx = (StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction)((Table.KeyValue)iter.next()).getValue();
                List localIDs = delTx.getLocalIDList();
                pendingDeleteBlockCountTotal += (long)localIDs.size();
                pendingDeleteBytes += KeyValueContainerMetadataInspector.computePendingDeleteBytes(localIDs, containerData, store);
            }
        }
        return new PendingDelete(pendingDeleteBlockCountTotal, pendingDeleteBytes);
    }

    private static long getBlockLength(BlockData block) {
        long blockLen = 0L;
        List chunkInfoList = block.getChunks();
        for (ContainerProtos.ChunkInfo chunk : chunkInfoList) {
            blockLen += chunk.getLen();
        }
        return blockLen;
    }

    static class PendingDelete {
        static final String COUNT = "pendingDeleteBlocks";
        static final String BYTES = "pendingDeleteBytes";
        private final long count;
        private final long bytes;

        PendingDelete(long count, long bytes) {
            this.count = count;
            this.bytes = bytes;
        }

        void addToJson(ObjectNode json) {
            json.put(COUNT, this.count);
            json.put(BYTES, this.bytes);
        }
    }

    public static enum Mode {
        REPAIR("repair"),
        INSPECT("inspect"),
        OFF("off");

        private final String name;

        private Mode(String name) {
            this.name = name;
        }

        public String toString() {
            return this.name;
        }
    }
}

