/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.authorization;

import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import org.apache.nifi.authorization.AuthorizerConfigurationContext;
import org.apache.nifi.authorization.Group;
import org.apache.nifi.authorization.NssShellCommands;
import org.apache.nifi.authorization.OsxShellCommands;
import org.apache.nifi.authorization.ShellCommandsProvider;
import org.apache.nifi.authorization.User;
import org.apache.nifi.authorization.UserAndGroups;
import org.apache.nifi.authorization.UserGroupProvider;
import org.apache.nifi.authorization.UserGroupProviderInitializationContext;
import org.apache.nifi.authorization.exception.AuthorizationAccessException;
import org.apache.nifi.authorization.exception.AuthorizerCreationException;
import org.apache.nifi.authorization.exception.AuthorizerDestructionException;
import org.apache.nifi.authorization.util.ShellRunner;
import org.apache.nifi.components.PropertyValue;
import org.apache.nifi.util.FormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ShellUserGroupProvider
implements UserGroupProvider {
    private static final Logger logger = LoggerFactory.getLogger(ShellUserGroupProvider.class);
    private static final String OS_TYPE_ERROR = "Unsupported operating system.";
    private static final String SYS_CHECK_ERROR = "System check failed - cannot provide users and groups.";
    private static final Map<String, User> usersById = new HashMap<String, User>();
    private static final Map<String, User> usersByName = new HashMap<String, User>();
    private static final Map<String, Group> groupsById = new HashMap<String, Group>();
    public static final String REFRESH_DELAY_PROPERTY = "Refresh Delay";
    private static final long MINIMUM_SYNC_INTERVAL_MILLISECONDS = 10000L;
    public static final String EXCLUDE_USER_PROPERTY = "Exclude Users";
    public static final String EXCLUDE_GROUP_PROPERTY = "Exclude Groups";
    private long fixedDelay;
    private Pattern excludeUsers;
    private Pattern excludeGroups;
    private final ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(2);
    private final Integer shellTimeout = 10;
    private ShellCommandsProvider selectedShellCommands;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Set<User> getUsers() throws AuthorizationAccessException {
        Map<String, User> map = usersById;
        synchronized (map) {
            logger.debug("getUsers has user set of size: " + usersById.size());
            return new HashSet<User>(usersById.values());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public User getUser(String identifier) throws AuthorizationAccessException {
        User user;
        Map<String, User> map = usersById;
        synchronized (map) {
            user = usersById.get(identifier);
        }
        if (user == null) {
            logger.debug("getUser (by id) user not found: " + identifier);
        } else {
            logger.debug("getUser (by id) found user: " + user + " for id: " + identifier);
        }
        return user;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public User getUserByIdentity(String identity) throws AuthorizationAccessException {
        User user;
        Map<String, User> map = usersByName;
        synchronized (map) {
            user = usersByName.get(identity);
        }
        if (user == null) {
            this.refreshOneUser(this.selectedShellCommands.getUserByName(identity), "Get Single User by Name");
            user = usersByName.get(identity);
        }
        if (user == null) {
            logger.debug("getUser (by name) user not found: " + identity);
        } else {
            logger.debug("getUser (by name) found user: " + user + " for name: " + identity);
        }
        return user;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Set<Group> getGroups() throws AuthorizationAccessException {
        Map<String, Group> map = groupsById;
        synchronized (map) {
            logger.debug("getGroups has group set of size: " + groupsById.size());
            return new HashSet<Group>(groupsById.values());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Group getGroup(String identifier) throws AuthorizationAccessException {
        Group group;
        Map<String, Group> map = groupsById;
        synchronized (map) {
            group = groupsById.get(identifier);
        }
        if (group == null) {
            this.refreshOneGroup(this.selectedShellCommands.getGroupById(identifier), "Get Single Group by Id");
            group = groupsById.get(identifier);
        }
        if (group == null) {
            logger.debug("getGroup (by id) group not found: " + identifier);
        } else {
            logger.debug("getGroup (by id) found group: " + group + " for id: " + identifier);
        }
        return group;
    }

    public UserAndGroups getUserAndGroups(String identity) throws AuthorizationAccessException {
        final User user = this.getUserByIdentity(identity);
        logger.debug("Retrieved user {} for identity {}", new Object[]{user, identity});
        final HashSet<Group> groups = new HashSet<Group>();
        for (Group g : this.getGroups()) {
            if (user == null || !g.getUsers().contains(user.getIdentity())) continue;
            groups.add(g);
        }
        if (groups.isEmpty()) {
            logger.debug("User {} belongs to no groups", (Object)user);
        }
        return new UserAndGroups(){

            public User getUser() {
                return user;
            }

            public Set<Group> getGroups() {
                return groups;
            }
        };
    }

    public void initialize(UserGroupProviderInitializationContext initializationContext) throws AuthorizerCreationException {
    }

    public void onConfigured(AuthorizerConfigurationContext configurationContext) throws AuthorizerCreationException {
        this.fixedDelay = this.getDelayProperty(configurationContext, REFRESH_DELAY_PROPERTY, "5 mins");
        ShellCommandsProvider commands = this.getCommandsProvider();
        if (commands == null) {
            commands = ShellUserGroupProvider.getCommandsProviderFromName(null);
            this.setCommandsProvider(commands);
        }
        try {
            ShellRunner.runShell(commands.getSystemCheck());
        }
        catch (IOException ioexc) {
            logger.error("initialize exception: " + ioexc + " system check command: " + commands.getSystemCheck());
            throw new AuthorizerCreationException(SYS_CHECK_ERROR, ioexc.getCause());
        }
        try {
            this.excludeGroups = Pattern.compile(this.getProperty(configurationContext, EXCLUDE_GROUP_PROPERTY, ""));
            this.excludeUsers = Pattern.compile(this.getProperty(configurationContext, EXCLUDE_USER_PROPERTY, ""));
        }
        catch (PatternSyntaxException e) {
            throw new AuthorizerCreationException((Throwable)e);
        }
        this.refreshUsersAndGroups();
        this.scheduler.scheduleWithFixedDelay(() -> {
            try {
                this.refreshUsersAndGroups();
            }
            catch (Throwable t) {
                logger.error("", t);
            }
        }, this.fixedDelay, this.fixedDelay, TimeUnit.SECONDS);
    }

    private static ShellCommandsProvider getCommandsProviderFromName(String osName) {
        ShellCommandsProvider commands;
        if (osName == null) {
            osName = System.getProperty("os.name");
        }
        if (osName.startsWith("Linux")) {
            logger.debug("Selected Linux command set.");
            commands = new NssShellCommands();
        } else if (osName.startsWith("Mac OS X")) {
            logger.debug("Selected OSX command set.");
            commands = new OsxShellCommands();
        } else {
            throw new AuthorizerCreationException(OS_TYPE_ERROR);
        }
        return commands;
    }

    private String getProperty(AuthorizerConfigurationContext authContext, String propertyName, String defaultValue) {
        PropertyValue property = authContext.getProperty(propertyName);
        String value = property != null && property.isSet() ? property.getValue() : defaultValue;
        return value;
    }

    private long getDelayProperty(AuthorizerConfigurationContext authContext, String propertyName, String defaultValue) {
        long syncInterval;
        PropertyValue intervalProperty = authContext.getProperty(propertyName);
        String propertyValue = intervalProperty.isSet() ? intervalProperty.getValue() : defaultValue;
        try {
            syncInterval = Math.round(FormatUtils.getPreciseTimeDuration((String)propertyValue, (TimeUnit)TimeUnit.MILLISECONDS));
        }
        catch (IllegalArgumentException ignored) {
            throw new AuthorizerCreationException(String.format("The %s '%s' is not a valid time interval.", propertyName, propertyValue));
        }
        if (syncInterval < 10000L) {
            throw new AuthorizerCreationException(String.format("The %s '%s' is below the minimum value of '%d ms'", propertyName, propertyValue, 10000L));
        }
        return syncInterval;
    }

    public void preDestruction() throws AuthorizerDestructionException {
        try {
            this.scheduler.shutdownNow();
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    public ShellCommandsProvider getCommandsProvider() {
        return this.selectedShellCommands;
    }

    public void setCommandsProvider(ShellCommandsProvider commandsProvider) {
        this.selectedShellCommands = commandsProvider;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void refreshOneUser(String command, String description) {
        if (command != null) {
            Map<String, User> map;
            HashMap<String, User> idToUser = new HashMap<String, User>();
            HashMap<String, User> usernameToUser = new HashMap<String, User>();
            HashMap<String, User> gidToUser = new HashMap<String, User>();
            try {
                List<String> userLines = ShellRunner.runShell(command, description);
                this.rebuildUsers(userLines, idToUser, usernameToUser, gidToUser);
            }
            catch (IOException ioexc) {
                logger.error("refreshOneUser shell exception: " + ioexc);
            }
            if (idToUser.size() > 0) {
                map = usersById;
                synchronized (map) {
                    usersById.putAll(idToUser);
                }
            }
            if (usernameToUser.size() > 0) {
                map = usersByName;
                synchronized (map) {
                    usersByName.putAll(usernameToUser);
                }
            }
        } else {
            logger.info("Get Single User not supported on this system.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void refreshOneGroup(String command, String description) {
        if (command != null) {
            HashMap<String, Group> gidToGroup = new HashMap<String, Group>();
            try {
                List<String> groupLines = ShellRunner.runShell(command, description);
                this.rebuildGroups(groupLines, gidToGroup);
            }
            catch (IOException ioexc) {
                logger.error("refreshOneGroup shell exception: " + ioexc);
            }
            if (gidToGroup.size() > 0) {
                Map<String, Group> map = groupsById;
                synchronized (map) {
                    groupsById.putAll(gidToGroup);
                }
            }
        } else {
            logger.info("Get Single Group not supported on this system.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void refreshUsersAndGroups() {
        List<String> groupLines;
        List<String> userLines;
        HashMap<String, User> uidToUser = new HashMap<String, User>();
        HashMap<String, User> usernameToUser = new HashMap<String, User>();
        HashMap<String, User> gidToUser = new HashMap<String, User>();
        HashMap<String, Group> gidToGroup = new HashMap<String, Group>();
        try {
            userLines = ShellRunner.runShell(this.selectedShellCommands.getUsersList(), "Get Users List");
            groupLines = ShellRunner.runShell(this.selectedShellCommands.getGroupsList(), "Get Groups List");
        }
        catch (IOException ioexc) {
            logger.error("refreshUsersAndGroups shell exception: " + ioexc);
            return;
        }
        this.rebuildUsers(userLines, uidToUser, usernameToUser, gidToUser);
        this.rebuildGroups(groupLines, gidToGroup);
        this.reconcilePrimaryGroups(gidToUser, gidToGroup);
        Map<String, User> map = usersById;
        synchronized (map) {
            usersById.clear();
            usersById.putAll(uidToUser);
        }
        map = usersByName;
        synchronized (map) {
            usersByName.clear();
            usersByName.putAll(usernameToUser);
            logger.debug("users now size: " + usersByName.size());
        }
        map = groupsById;
        synchronized (map) {
            groupsById.clear();
            groupsById.putAll(gidToGroup);
            logger.debug("groups now size: " + groupsById.size());
        }
    }

    private void rebuildUsers(List<String> userLines, Map<String, User> idToUser, Map<String, User> usernameToUser, Map<String, User> gidToUser) {
        userLines.forEach(line -> {
            String[] record = line.split(":");
            if (record.length > 2) {
                String name = record[0];
                String id = record[1];
                String gid = record[2];
                if (!(name == null || id == null || name.equals("") || id.equals("") || this.excludeUsers.matcher(name).matches())) {
                    User user = new User.Builder().identity(name).identifierGenerateFromSeed(id).build();
                    idToUser.put(user.getIdentifier(), user);
                    usernameToUser.put(name, user);
                    if (gid != null && !gid.equals("")) {
                        Group group = new Group.Builder().name(gid).identifierGenerateFromSeed(gid).build();
                        gidToUser.put(group.getIdentifier(), user);
                    } else {
                        logger.warn("Null or empty primary group id for: " + name);
                    }
                } else {
                    logger.warn("Null, empty, or skipped user name: " + name + " or id: " + id);
                }
            } else {
                logger.warn("Unexpected record format.  Expected 3 or more colon separated values per line.");
            }
        });
    }

    private void rebuildGroups(List<String> groupLines, Map<String, Group> groupsById) {
        groupLines.forEach(line -> {
            String[] record = line.split(":");
            if (record.length > 1) {
                HashSet<String> users = new HashSet<String>();
                String name = record[0];
                String id = record[1];
                try {
                    List<String> memberLines = ShellRunner.runShell(this.selectedShellCommands.getGroupMembers(name));
                    if (!memberLines.isEmpty()) {
                        users.addAll(Arrays.asList(memberLines.get(0).split(",")));
                    } else {
                        logger.debug("list membership returned zero lines.");
                    }
                    if (memberLines.size() > 1) {
                        logger.error("list membership returned too many lines, only used the first.");
                    }
                }
                catch (IOException ioexc) {
                    logger.error("list membership shell exception: " + ioexc);
                }
                if (!(name == null || id == null || name.equals("") || id.equals("") || this.excludeGroups.matcher(name).matches())) {
                    Group group = new Group.Builder().name(name).identifierGenerateFromSeed(id).addUsers(users).build();
                    groupsById.put(group.getIdentifier(), group);
                    logger.debug("Refreshed group: " + group);
                } else {
                    logger.warn("Null, empty, or skipped group name: " + name + " or id: " + id);
                }
            } else {
                logger.warn("Unexpected record format.  Expected 1 or more comma separated values.");
            }
        });
    }

    private void reconcilePrimaryGroups(Map<String, User> uidToUser, Map<String, Group> gidToGroup) {
        uidToUser.forEach((primaryGid, primaryUser) -> {
            Set groupUsers;
            Group primaryGroup = (Group)gidToGroup.get(primaryGid);
            if (primaryGroup == null) {
                logger.warn("user: " + primaryUser + " primary group not found");
            } else if (!this.excludeGroups.matcher(primaryGroup.getName()).matches() && !(groupUsers = primaryGroup.getUsers()).contains(primaryUser.getIdentity())) {
                HashSet<String> secondSet = new HashSet<String>(groupUsers);
                secondSet.add(primaryUser.getIdentity());
                Group group = new Group.Builder().name(primaryGroup.getName()).identifierGenerateFromSeed(primaryGid).addUsers(secondSet).build();
                gidToGroup.put(group.getIdentifier(), group);
            }
        });
    }

    public long getRefreshDelay() {
        return this.fixedDelay;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void clearCaches() {
        Map<String, User> map = usersById;
        synchronized (map) {
            usersById.clear();
        }
        map = usersByName;
        synchronized (map) {
            usersByName.clear();
        }
        map = groupsById;
        synchronized (map) {
            groupsById.clear();
        }
    }

    public int userCacheSize() {
        return usersById.size();
    }

    public int groupCacheSize() {
        return groupsById.size();
    }
}

