/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.flow.encryptor.command;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.List;
import java.util.Objects;
import java.util.Properties;
import java.util.stream.Collectors;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import org.apache.nifi.encrypt.PropertyEncryptionMethod;
import org.apache.nifi.encrypt.PropertyEncryptor;
import org.apache.nifi.encrypt.PropertyEncryptorBuilder;
import org.apache.nifi.flow.encryptor.JsonFlowEncryptor;

class FlowEncryptorCommand
implements Runnable {
    protected static final String PROPERTIES_FILE_PATH = "nifi.properties.file.path";
    protected static final String PROPS_KEY = "nifi.sensitive.props.key";
    protected static final String PROPS_ALGORITHM = "nifi.sensitive.props.algorithm";
    protected static final String CONFIGURATION_FILE = "nifi.flow.configuration.file";
    private static final String FLOW_PREFIX = "nifi.flow.";
    private static final String GZ_EXTENSION = ".gz";
    private static final String DEFAULT_PROPERTIES_ALGORITHM = PropertyEncryptionMethod.NIFI_PBKDF2_AES_GCM_256.name();
    private static final String SENSITIVE_PROPERTIES_KEY = String.format("%s=", "nifi.sensitive.props.key");
    private static final String SENSITIVE_PROPERTIES_ALGORITHM = String.format("%s=", "nifi.sensitive.props.algorithm");
    private String requestedPropertiesKey;
    private String requestedPropertiesAlgorithm;

    FlowEncryptorCommand() {
    }

    void setRequestedPropertiesKey(String requestedPropertiesKey) {
        this.requestedPropertiesKey = Objects.requireNonNull(requestedPropertiesKey, "Key required");
    }

    void setRequestedPropertiesAlgorithm(String requestedPropertiesAlgorithm) {
        this.requestedPropertiesAlgorithm = Objects.requireNonNull(requestedPropertiesAlgorithm, "Algorithm required");
    }

    @Override
    public void run() {
        String propertiesFilePath = System.getProperty(PROPERTIES_FILE_PATH);
        if (propertiesFilePath == null) {
            throw new IllegalStateException(String.format("System property not defined [%s]", PROPERTIES_FILE_PATH));
        }
        File propertiesFile = new File(propertiesFilePath);
        Properties properties = this.loadProperties(propertiesFile);
        this.processFlowConfigurationFiles(properties);
        try {
            this.storeProperties(propertiesFile);
            System.out.printf("NiFi Properties Processed [%s]%n", propertiesFilePath);
        }
        catch (IOException e) {
            String message = String.format("Failed to Process NiFi Properties [%s]", propertiesFilePath);
            throw new UncheckedIOException(message, e);
        }
    }

    private void processFlowConfigurationFiles(Properties properties) {
        String outputKey;
        String outputAlgorithm;
        String string = outputAlgorithm = this.requestedPropertiesAlgorithm == null ? this.getAlgorithm(properties) : this.requestedPropertiesAlgorithm;
        if (this.requestedPropertiesKey == null) {
            String inputKey = properties.getProperty(PROPS_KEY);
            if (inputKey == null || inputKey.isBlank()) {
                throw new IllegalStateException("Sensitive Properties Key [%s] not specified".formatted(PROPS_KEY));
            }
            outputKey = inputKey;
        } else {
            outputKey = this.requestedPropertiesKey;
        }
        PropertyEncryptor outputEncryptor = this.getPropertyEncryptor(outputKey, outputAlgorithm);
        String configurationFileProperty = properties.getProperty(CONFIGURATION_FILE);
        if (configurationFileProperty == null || configurationFileProperty.isEmpty()) {
            throw new IllegalStateException("Flow Configuration Property not specified [%s]".formatted(configurationFileProperty));
        }
        File configurationFile = new File(configurationFileProperty);
        if (!configurationFile.exists()) {
            throw new IllegalStateException("Flow Configuration File not found [%s]".formatted(configurationFileProperty));
        }
        this.processFlowConfiguration(configurationFile, properties, outputEncryptor);
    }

    private void processFlowConfiguration(File flowConfigurationFile, Properties properties, PropertyEncryptor outputEncryptor) {
        try (GZIPInputStream flowInputStream = new GZIPInputStream(new FileInputStream(flowConfigurationFile));){
            File flowOutputFile = this.getFlowOutputFile();
            Path flowOutputPath = flowOutputFile.toPath();
            try (GZIPOutputStream flowOutputStream = new GZIPOutputStream(new FileOutputStream(flowOutputFile));){
                String inputAlgorithm = this.getAlgorithm(properties);
                String inputPropertiesKey = this.getInputPropertiesKey(properties);
                PropertyEncryptor inputEncryptor = this.getPropertyEncryptor(inputPropertiesKey, inputAlgorithm);
                JsonFlowEncryptor flowEncryptor = new JsonFlowEncryptor();
                flowEncryptor.processFlow(flowInputStream, flowOutputStream, inputEncryptor, outputEncryptor);
            }
            Path flowConfigurationPath = flowConfigurationFile.toPath();
            Files.move(flowOutputPath, flowConfigurationPath, StandardCopyOption.REPLACE_EXISTING);
            System.out.printf("Flow Configuration Processed [%s]%n", flowConfigurationPath);
        }
        catch (IOException | RuntimeException e) {
            throw new IllegalStateException("Failed to process Flow Configuration [%s]".formatted(flowConfigurationFile), e);
        }
    }

    private String getAlgorithm(Properties properties) {
        String algorithm = properties.getProperty(PROPS_ALGORITHM, DEFAULT_PROPERTIES_ALGORITHM);
        if (algorithm.isEmpty()) {
            algorithm = DEFAULT_PROPERTIES_ALGORITHM;
        }
        return algorithm;
    }

    private String getInputPropertiesKey(Properties properties) {
        String key = properties.getProperty(PROPS_KEY);
        if (key == null || key.isBlank()) {
            throw new IllegalStateException("Sensitive Properties Key [%s] not found".formatted(PROPS_KEY));
        }
        return key;
    }

    private File getFlowOutputFile() throws IOException {
        File flowOutputFile = File.createTempFile(FLOW_PREFIX, GZ_EXTENSION);
        flowOutputFile.deleteOnExit();
        return flowOutputFile;
    }

    private Properties loadProperties(File propertiesFile) {
        Properties properties = new Properties();
        try (FileReader reader = new FileReader(propertiesFile);){
            properties.load(reader);
        }
        catch (IOException e) {
            String message = String.format("Failed to read NiFi Properties [%s]", propertiesFile);
            throw new UncheckedIOException(message, e);
        }
        return properties;
    }

    private void storeProperties(File propertiesFile) throws IOException {
        Path propertiesFilePath = propertiesFile.toPath();
        List<String> lines = Files.readAllLines(propertiesFilePath);
        List updatedLines = lines.stream().map(line -> {
            if (line.startsWith(SENSITIVE_PROPERTIES_KEY)) {
                return this.requestedPropertiesKey == null ? line : SENSITIVE_PROPERTIES_KEY + this.requestedPropertiesKey;
            }
            if (line.startsWith(SENSITIVE_PROPERTIES_ALGORITHM)) {
                return this.requestedPropertiesAlgorithm == null ? line : SENSITIVE_PROPERTIES_ALGORITHM + this.requestedPropertiesAlgorithm;
            }
            return line;
        }).collect(Collectors.toList());
        Files.write(propertiesFilePath, updatedLines, new OpenOption[0]);
    }

    private PropertyEncryptor getPropertyEncryptor(String propertiesKey, String propertiesAlgorithm) {
        return new PropertyEncryptorBuilder(propertiesKey).setAlgorithm(propertiesAlgorithm).build();
    }
}

