/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.azure.storage.queue;

import com.azure.core.http.rest.PagedIterable;
import com.azure.core.util.Context;
import com.azure.storage.queue.QueueClient;
import com.azure.storage.queue.models.QueueMessageItem;
import com.azure.storage.queue.models.QueueStorageException;
import java.time.Duration;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.SeeAlso;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.processors.azure.storage.queue.AbstractAzureQueueStorage_v12;
import org.apache.nifi.processors.azure.storage.queue.PutAzureQueueStorage_v12;
import org.apache.nifi.proxy.ProxyConfiguration;
import org.apache.nifi.proxy.ProxySpec;

@SeeAlso(value={PutAzureQueueStorage_v12.class})
@InputRequirement(value=InputRequirement.Requirement.INPUT_FORBIDDEN)
@Tags(value={"azure", "queue", "microsoft", "storage", "dequeue", "cloud"})
@CapabilityDescription(value="Retrieves the messages from an Azure Queue Storage. The retrieved messages will be deleted from the queue by default. If the requirement is to consume messages without deleting them, set 'Auto Delete Messages' to 'false'. Note: There might be chances of receiving duplicates in situations like when a message is received but was unable to be deleted from the queue due to some unexpected situations.")
@WritesAttributes(value={@WritesAttribute(attribute="azure.queue.uri", description="The absolute URI of the configured Azure Queue Storage"), @WritesAttribute(attribute="azure.queue.insertionTime", description="The time when the message was inserted into the queue storage"), @WritesAttribute(attribute="azure.queue.expirationTime", description="The time when the message will expire from the queue storage"), @WritesAttribute(attribute="azure.queue.messageId", description="The ID of the retrieved message"), @WritesAttribute(attribute="azure.queue.popReceipt", description="The pop receipt of the retrieved message")})
public class GetAzureQueueStorage_v12
extends AbstractAzureQueueStorage_v12 {
    public static final PropertyDescriptor AUTO_DELETE = new PropertyDescriptor.Builder().name("Auto Delete Messages").description("Specifies whether the received message is to be automatically deleted from the queue.").required(true).allowableValues(new String[]{"true", "false"}).defaultValue("true").addValidator(StandardValidators.BOOLEAN_VALIDATOR).build();
    public static final PropertyDescriptor MESSAGE_BATCH_SIZE = new PropertyDescriptor.Builder().name("Message Batch Size").description("The number of messages to be retrieved from the queue.").required(true).addValidator(StandardValidators.createLongValidator((long)1L, (long)32L, (boolean)true)).defaultValue("32").build();
    public static final PropertyDescriptor VISIBILITY_TIMEOUT = new PropertyDescriptor.Builder().name("Visibility Timeout").description("The duration during which the retrieved message should be invisible to other consumers.").required(true).defaultValue("30 secs").addValidator(StandardValidators.TIME_PERIOD_VALIDATOR).build();
    private static final ProxySpec[] PROXY_SPECS = new ProxySpec[]{ProxySpec.HTTP, ProxySpec.SOCKS};
    private static final List<PropertyDescriptor> PROPERTY_DESCRIPTORS = List.of(QUEUE_NAME, ENDPOINT_SUFFIX, STORAGE_CREDENTIALS_SERVICE, AUTO_DELETE, MESSAGE_BATCH_SIZE, VISIBILITY_TIMEOUT, REQUEST_TIMEOUT, ProxyConfiguration.createProxyConfigPropertyDescriptor((ProxySpec[])PROXY_SPECS));
    private static final Set<Relationship> RELATIONSHIPS = Set.of(REL_SUCCESS);
    private static final Duration MAX_VISIBILITY_TIMEOUT = Duration.ofDays(7L);

    public List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return PROPERTY_DESCRIPTORS;
    }

    @Override
    protected Collection<ValidationResult> customValidate(ValidationContext validationContext) {
        List results = (List)super.customValidate(validationContext);
        Duration visibilityTimeout = Duration.ofSeconds(validationContext.getProperty(VISIBILITY_TIMEOUT).asTimePeriod(TimeUnit.SECONDS));
        if (visibilityTimeout.getSeconds() <= 0L) {
            results.add(new ValidationResult.Builder().valid(false).subject(VISIBILITY_TIMEOUT.getDisplayName()).explanation(VISIBILITY_TIMEOUT.getDisplayName() + " should be greater than 0 secs").build());
        }
        if (MAX_VISIBILITY_TIMEOUT.compareTo(visibilityTimeout) < 0) {
            results.add(new ValidationResult.Builder().valid(false).subject(VISIBILITY_TIMEOUT.getDisplayName()).explanation(VISIBILITY_TIMEOUT.getDisplayName() + " should not be greater than 7 days").build());
        }
        return results;
    }

    @Override
    public Set<Relationship> getRelationships() {
        return RELATIONSHIPS;
    }

    public void onTrigger(ProcessContext context, ProcessSession session) throws ProcessException {
        PagedIterable retrievedMessagesIterable;
        int batchSize = context.getProperty(MESSAGE_BATCH_SIZE).asInteger();
        int visibilityTimeoutInSecs = context.getProperty(VISIBILITY_TIMEOUT).asTimePeriod(TimeUnit.SECONDS).intValue();
        int requestTimeoutInSecs = context.getProperty(REQUEST_TIMEOUT).asTimePeriod(TimeUnit.SECONDS).intValue();
        boolean autoDelete = context.getProperty(AUTO_DELETE).asBoolean();
        QueueClient queueClient = this.createQueueClient(context, null);
        try {
            retrievedMessagesIterable = queueClient.receiveMessages(Integer.valueOf(batchSize), Duration.ofSeconds(visibilityTimeoutInSecs), Duration.ofSeconds(requestTimeoutInSecs), Context.NONE);
        }
        catch (QueueStorageException e) {
            this.getLogger().error("Failed to retrieve messages from Azure Storage Queue", (Throwable)e);
            context.yield();
            return;
        }
        LinkedHashMap<String, String> messagesToDelete = new LinkedHashMap<String, String>();
        for (QueueMessageItem message : retrievedMessagesIterable) {
            FlowFile flowFile = session.create();
            LinkedHashMap<String, String> attributes = new LinkedHashMap<String, String>();
            attributes.put("azure.queue.uri", queueClient.getQueueUrl());
            attributes.put("azure.queue.insertionTime", message.getInsertionTime().toString());
            attributes.put("azure.queue.expirationTime", message.getExpirationTime().toString());
            attributes.put("azure.queue.messageId", message.getMessageId());
            attributes.put("azure.queue.popReceipt", message.getPopReceipt());
            if (autoDelete) {
                messagesToDelete.put(message.getMessageId(), message.getPopReceipt());
            }
            flowFile = session.putAllAttributes(flowFile, attributes);
            flowFile = session.write(flowFile, out -> out.write(message.getBody().toString().getBytes()));
            session.transfer(flowFile, REL_SUCCESS);
            session.getProvenanceReporter().receive(flowFile, queueClient.getQueueUrl());
        }
        if (autoDelete) {
            session.commitAsync(() -> {
                for (Map.Entry entry : messagesToDelete.entrySet()) {
                    String messageId = (String)entry.getKey();
                    String popReceipt = (String)entry.getValue();
                    queueClient.deleteMessage(messageId, popReceipt);
                }
            });
        }
    }
}

