/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.parameter.aws;

import com.amazonaws.ClientConfiguration;
import com.amazonaws.Protocol;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.http.conn.ssl.SdkTLSSocketFactory;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.secretsmanager.AWSSecretsManager;
import com.amazonaws.services.secretsmanager.AWSSecretsManagerClientBuilder;
import com.amazonaws.services.secretsmanager.model.AWSSecretsManagerException;
import com.amazonaws.services.secretsmanager.model.GetSecretValueRequest;
import com.amazonaws.services.secretsmanager.model.GetSecretValueResult;
import com.amazonaws.services.secretsmanager.model.ListSecretsRequest;
import com.amazonaws.services.secretsmanager.model.ListSecretsResult;
import com.amazonaws.services.secretsmanager.model.ResourceNotFoundException;
import com.amazonaws.services.secretsmanager.model.SecretListEntry;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.ssl.DefaultHostnameVerifier;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.components.AllowableValue;
import org.apache.nifi.components.ConfigVerificationResult;
import org.apache.nifi.components.DescribedValue;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.controller.ConfigurationContext;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.parameter.AbstractParameterProvider;
import org.apache.nifi.parameter.Parameter;
import org.apache.nifi.parameter.ParameterGroup;
import org.apache.nifi.parameter.VerifiableParameterProvider;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.processors.aws.credentials.provider.service.AWSCredentialsProviderService;
import org.apache.nifi.ssl.SSLContextProvider;

@Tags(value={"aws", "secretsmanager", "secrets", "manager"})
@CapabilityDescription(value="Fetches parameters from AWS SecretsManager.  Each secret becomes a Parameter group, which can map to a Parameter Context, with key/value pairs in the secret mapping to Parameters in the group.")
public class AwsSecretsManagerParameterProvider
extends AbstractParameterProvider
implements VerifiableParameterProvider {
    public static final PropertyDescriptor SECRET_LISTING_STRATEGY = new PropertyDescriptor.Builder().name("secret-listing-strategy").displayName("Secret Listing Strategy").description("Strategy to use for listing secrets.").required(true).allowableValues(ListingStrategy.class).defaultValue((DescribedValue)ListingStrategy.PATTERN).build();
    public static final PropertyDescriptor SECRET_NAME_PATTERN = new PropertyDescriptor.Builder().name("secret-name-pattern").displayName("Secret Name Pattern").description("A Regular Expression matching on Secret Name that identifies Secrets whose parameters should be fetched. Any secrets whose names do not match this pattern will not be fetched.").addValidator(StandardValidators.REGULAR_EXPRESSION_VALIDATOR).dependsOn(SECRET_LISTING_STRATEGY, (DescribedValue)ListingStrategy.PATTERN, new DescribedValue[0]).required(true).defaultValue(".*").build();
    public static final PropertyDescriptor SECRET_NAMES = new PropertyDescriptor.Builder().name("secret-names").displayName("Secret Names").description("Comma-separated list of secret names to fetch.").dependsOn(SECRET_LISTING_STRATEGY, (DescribedValue)ListingStrategy.ENUMERATION, new DescribedValue[0]).required(true).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    public static final PropertyDescriptor AWS_CREDENTIALS_PROVIDER_SERVICE = new PropertyDescriptor.Builder().name("aws-credentials-provider-service").displayName("AWS Credentials Provider Service").description("Service used to obtain an Amazon Web Services Credentials Provider").required(true).identifiesControllerService(AWSCredentialsProviderService.class).build();
    public static final PropertyDescriptor REGION = new PropertyDescriptor.Builder().name("aws-region").displayName("Region").required(true).allowableValues((DescribedValue[])AwsSecretsManagerParameterProvider.getAvailableRegions()).defaultValue(AwsSecretsManagerParameterProvider.createAllowableValue(Regions.DEFAULT_REGION).getValue()).build();
    public static final PropertyDescriptor TIMEOUT = new PropertyDescriptor.Builder().name("aws-communications-timeout").displayName("Communications Timeout").required(true).addValidator(StandardValidators.TIME_PERIOD_VALIDATOR).defaultValue("30 secs").build();
    public static final PropertyDescriptor SSL_CONTEXT_SERVICE = new PropertyDescriptor.Builder().name("aws-ssl-context-service").displayName("SSL Context Service").description("Specifies an optional SSL Context Service that, if provided, will be used to create connections").required(false).identifiesControllerService(SSLContextProvider.class).build();
    private static final String DEFAULT_USER_AGENT = "NiFi";
    private static final Protocol DEFAULT_PROTOCOL = Protocol.HTTPS;
    private static final List<PropertyDescriptor> PROPERTY_DESCRIPTORS = List.of(SECRET_LISTING_STRATEGY, SECRET_NAME_PATTERN, SECRET_NAMES, REGION, AWS_CREDENTIALS_PROVIDER_SERVICE, TIMEOUT, SSL_CONTEXT_SERVICE);
    private final ObjectMapper objectMapper = new ObjectMapper();

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return PROPERTY_DESCRIPTORS;
    }

    public List<ParameterGroup> fetchParameters(ConfigurationContext context) {
        AWSSecretsManager secretsManager = this.configureClient(context);
        ArrayList<ParameterGroup> groups = new ArrayList<ParameterGroup>();
        ListingStrategy listingStrategy = (ListingStrategy)context.getProperty(SECRET_LISTING_STRATEGY).asAllowableValue(ListingStrategy.class);
        HashSet<String> fetchSecretNames = new HashSet<String>();
        if (listingStrategy == ListingStrategy.ENUMERATION) {
            String secretNames = context.getProperty(SECRET_NAMES).getValue();
            fetchSecretNames.addAll(Arrays.asList(secretNames.split(",")));
        } else {
            Pattern secretNamePattern = Pattern.compile(context.getProperty(SECRET_NAME_PATTERN).getValue());
            ListSecretsRequest listSecretsRequest = new ListSecretsRequest();
            ListSecretsResult listSecretsResult = secretsManager.listSecrets(listSecretsRequest);
            while (!listSecretsResult.getSecretList().isEmpty()) {
                for (SecretListEntry entry : listSecretsResult.getSecretList()) {
                    String secretName = entry.getName();
                    if (!secretNamePattern.matcher(secretName).matches()) {
                        this.getLogger().debug("Secret [{}] does not match the secret name pattern {}", new Object[]{secretName, secretNamePattern});
                        continue;
                    }
                    fetchSecretNames.add(secretName);
                }
                String nextToken = listSecretsResult.getNextToken();
                if (nextToken == null) break;
                listSecretsRequest.setNextToken(nextToken);
                listSecretsResult = secretsManager.listSecrets(listSecretsRequest);
            }
        }
        for (String secretName : fetchSecretNames) {
            List<ParameterGroup> secretParameterGroups = this.fetchSecret(secretsManager, secretName);
            groups.addAll(secretParameterGroups);
        }
        return groups;
    }

    public List<ConfigVerificationResult> verify(ConfigurationContext context, ComponentLog verificationLogger) {
        ArrayList<ConfigVerificationResult> results = new ArrayList<ConfigVerificationResult>();
        try {
            List<ParameterGroup> parameterGroups = this.fetchParameters(context);
            int parameterCount = 0;
            for (ParameterGroup group : parameterGroups) {
                parameterCount += group.getParameters().size();
            }
            results.add(new ConfigVerificationResult.Builder().outcome(ConfigVerificationResult.Outcome.SUCCESSFUL).verificationStepName("Fetch Parameters").explanation(String.format("Fetched secret keys [%d] as parameters, across groups [%d]", parameterCount, parameterGroups.size())).build());
        }
        catch (Exception e) {
            verificationLogger.error("Failed to fetch parameters", (Throwable)e);
            results.add(new ConfigVerificationResult.Builder().outcome(ConfigVerificationResult.Outcome.FAILED).verificationStepName("Fetch Parameters").explanation("Failed to fetch parameters: " + e.getMessage()).build());
        }
        return results;
    }

    private List<ParameterGroup> fetchSecret(AWSSecretsManager secretsManager, String secretName) {
        ArrayList<ParameterGroup> groups = new ArrayList<ParameterGroup>();
        ArrayList<Parameter> parameters = new ArrayList<Parameter>();
        GetSecretValueRequest getSecretValueRequest = new GetSecretValueRequest().withSecretId(secretName);
        try {
            GetSecretValueResult getSecretValueResult = secretsManager.getSecretValue(getSecretValueRequest);
            if (getSecretValueResult.getSecretString() == null) {
                this.getLogger().debug("Secret [{}] is not configured", new Object[]{secretName});
                return groups;
            }
            ObjectNode secretObject = this.parseSecret(getSecretValueResult.getSecretString());
            if (secretObject == null) {
                this.getLogger().debug("Secret [{}] is not in the expected JSON key/value format", new Object[]{secretName});
                return groups;
            }
            Iterator it = secretObject.fields();
            while (it.hasNext()) {
                Map.Entry field = (Map.Entry)it.next();
                String parameterName = (String)field.getKey();
                String parameterValue = ((JsonNode)field.getValue()).textValue();
                if (parameterValue == null) {
                    this.getLogger().debug("Secret [{}] Parameter [{}] has no value", new Object[]{secretName, parameterName});
                    continue;
                }
                parameters.add(this.createParameter(parameterName, parameterValue));
            }
            groups.add(new ParameterGroup(secretName, parameters));
            return groups;
        }
        catch (ResourceNotFoundException e) {
            throw new IllegalStateException(String.format("Secret %s not found", secretName), e);
        }
        catch (AWSSecretsManagerException e) {
            throw new IllegalStateException("Error retrieving secret " + secretName, e);
        }
    }

    private Parameter createParameter(String parameterName, String parameterValue) {
        return new Parameter.Builder().name(parameterName).value(parameterValue).provided(Boolean.valueOf(true)).build();
    }

    protected ClientConfiguration createConfiguration(ConfigurationContext context) {
        ClientConfiguration config = new ClientConfiguration();
        config.setMaxErrorRetry(0);
        config.setUserAgentPrefix(DEFAULT_USER_AGENT);
        config.setProtocol(DEFAULT_PROTOCOL);
        int commsTimeout = context.getProperty(TIMEOUT).asTimePeriod(TimeUnit.MILLISECONDS).intValue();
        config.setConnectionTimeout(commsTimeout);
        config.setSocketTimeout(commsTimeout);
        SSLContextProvider sslContextProvider = (SSLContextProvider)context.getProperty(SSL_CONTEXT_SERVICE).asControllerService(SSLContextProvider.class);
        if (sslContextProvider != null) {
            SSLContext sslContext = sslContextProvider.createContext();
            SdkTLSSocketFactory sdkTLSSocketFactory = new SdkTLSSocketFactory(sslContext, (HostnameVerifier)new DefaultHostnameVerifier());
            config.getApacheHttpClientConfig().setSslSocketFactory((ConnectionSocketFactory)sdkTLSSocketFactory);
        }
        return config;
    }

    private ObjectNode parseSecret(String secretString) {
        try {
            JsonNode root = this.objectMapper.readTree(secretString);
            if (root instanceof ObjectNode) {
                return (ObjectNode)root;
            }
            return null;
        }
        catch (JsonProcessingException e) {
            this.getLogger().debug("Error parsing JSON", (Throwable)e);
            return null;
        }
    }

    AWSSecretsManager configureClient(ConfigurationContext context) {
        return (AWSSecretsManager)((AWSSecretsManagerClientBuilder)((AWSSecretsManagerClientBuilder)((AWSSecretsManagerClientBuilder)AWSSecretsManagerClientBuilder.standard().withRegion(context.getProperty(REGION).getValue())).withClientConfiguration(this.createConfiguration(context))).withCredentials(this.getCredentialsProvider(context))).build();
    }

    protected AWSCredentialsProvider getCredentialsProvider(ConfigurationContext context) {
        AWSCredentialsProviderService awsCredentialsProviderService = (AWSCredentialsProviderService)context.getProperty(AWS_CREDENTIALS_PROVIDER_SERVICE).asControllerService(AWSCredentialsProviderService.class);
        return awsCredentialsProviderService.getCredentialsProvider();
    }

    private static AllowableValue createAllowableValue(Regions region) {
        return new AllowableValue(region.getName(), region.getDescription(), "AWS Region Code : " + region.getName());
    }

    private static AllowableValue[] getAvailableRegions() {
        ArrayList<AllowableValue> values = new ArrayList<AllowableValue>();
        for (Regions region : Regions.values()) {
            values.add(AwsSecretsManagerParameterProvider.createAllowableValue(region));
        }
        return values.toArray(new AllowableValue[0]);
    }

    static enum ListingStrategy implements DescribedValue
    {
        ENUMERATION("Enumerate Secret Names", "Requires a set of secret names to fetch. AWS actions required: GetSecretValue."),
        PATTERN("Match Pattern", "Requires a regular expression pattern to match secret names. AWS actions required: ListSecrets and GetSecretValue.");

        private final String displayName;
        private final String description;

        private ListingStrategy(String displayName, String description) {
            this.displayName = displayName;
            this.description = description;
        }

        public String getValue() {
            return this.name();
        }

        public String getDisplayName() {
            return this.displayName;
        }

        public String getDescription() {
            return this.description;
        }
    }
}

