/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kylin.tool.kerberos;

import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.security.InvalidParameterException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.security.Credentials;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.security.token.TokenIdentifier;
import org.apache.hadoop.security.token.delegation.AbstractDelegationTokenIdentifier;
import org.apache.hadoop.util.Shell;
import org.apache.kylin.common.KapConfig;
import org.apache.kylin.common.util.Unsafe;
import org.apache.kylin.engine.spark.utils.ThreadUtils;
import org.apache.kylin.guava30.shaded.common.base.Preconditions;
import org.apache.kylin.guava30.shaded.common.collect.Maps;
import org.apache.kylin.tool.kerberos.KerberosLoginUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.ReflectionUtils;

public class DelegationTokenManager {
    private String principal;
    private String keytab;
    private ScheduledExecutorService renewalExecutor;
    private final KapConfig kapConf;
    private static final String CONTEXT_NAME = "Client";
    private static final Configuration CONFIGURATION = new Configuration();
    private static final Logger logger = LoggerFactory.getLogger(DelegationTokenManager.class);

    public DelegationTokenManager() {
        this(KapConfig.getInstanceFromEnv());
    }

    public DelegationTokenManager(KapConfig kapConf) {
        this.kapConf = kapConf;
    }

    public void start() {
        if (Boolean.FALSE.equals(this.kapConf.isKerberosEnabled())) {
            logger.info("Kerberos is not enabled.");
            return;
        }
        this.principal = this.kapConf.getKerberosPrincipal();
        this.keytab = this.kapConf.getKerberosKeytabPath();
        this.preCheck();
        this.renewalExecutor = ThreadUtils.newDaemonSingleThreadScheduledExecutor((String)"Kylin Credential Renewal Thread");
        Runtime.getRuntime().addShutdownHook(new Thread(this::stop));
        this.tryLogin();
        this.scheduleTGTRenewal();
        this.scheduleTGTCacheRenewal();
    }

    private void tryLogin() {
        try {
            this.doLogin();
        }
        catch (IOException ioe) {
            long retryInterval = this.kapConf.getKerberosTGTRetryInterval();
            logger.error("Failed to login kerberos from principal: {}, keytab: {}, will try again in {} minutes. If this happens too often tasks will fail.", new Object[]{this.principal, this.keytab, retryInterval, ioe});
            this.renewalExecutor.schedule(this::tryLogin, Math.max(0L, retryInterval), TimeUnit.MINUTES);
        }
    }

    private void scheduleTGTRenewal() {
        Runnable tgtRenewalTask = () -> {
            try {
                this.updateCredentials();
            }
            catch (Exception e) {
                logger.error("Failed to update UGI credentials.", (Throwable)e);
            }
        };
        long renewalInternal = this.kapConf.getKerberosTGTRenewalInterval();
        this.renewalExecutor.scheduleWithFixedDelay(tgtRenewalTask, renewalInternal, renewalInternal, TimeUnit.MINUTES);
    }

    private void scheduleTGTCacheRenewal() {
        Runnable tgtCacheRenewalTask = () -> {
            try {
                this.doRenewTGTCache();
            }
            catch (IOException ioe) {
                logger.error("Failed to renew kerberos tgt cache at KRB5CCNAME.", (Throwable)ioe);
            }
        };
        long renewalInternal = this.kapConf.getKerberosTicketRefreshInterval();
        this.renewalExecutor.scheduleWithFixedDelay(tgtCacheRenewalTask, renewalInternal, renewalInternal, TimeUnit.MINUTES);
    }

    private void updateCredentials() throws IOException, NoSuchFieldException {
        UserGroupInformation current = UserGroupInformation.getCurrentUser();
        current.checkTGTAndReloginFromKeytab();
        Object fsCache = this.getFileSystemCache();
        if (Objects.isNull(fsCache)) {
            return;
        }
        Collection<Object> cacheKeys = this.getCacheKeys(fsCache);
        Credentials creds = current.getCredentials();
        Collection<Token<? extends TokenIdentifier>> tokens = this.getTokens(creds);
        for (Object key : cacheKeys) {
            UserGroupInformation ugi = this.getUGI(key);
            if (!this.updatable(ugi, current)) continue;
            try {
                this.updateTokens(ugi, tokens);
            }
            catch (Exception e) {
                logger.debug("Failed to update private tokens, hadoop version not supported.", (Throwable)e);
            }
            ugi.addCredentials(creds);
        }
    }

    private Collection<Token<? extends TokenIdentifier>> getTokens(Credentials creds) {
        return creds.getAllTokens().stream().filter(t -> Objects.nonNull(t.getKind()) && Objects.nonNull(t.getService())).collect(Collectors.collectingAndThen(Collectors.toList(), Collections::unmodifiableCollection));
    }

    private Object getFileSystemCache() throws NoSuchFieldException {
        Field cacheField = FileSystem.class.getDeclaredField("CACHE");
        ReflectionUtils.makeAccessible((Field)cacheField);
        return ReflectionUtils.getField((Field)cacheField, null);
    }

    private Collection<Object> getCacheKeys(Object fsCache) throws NoSuchFieldException {
        Field mapField = fsCache.getClass().getDeclaredField("map");
        ReflectionUtils.makeAccessible((Field)mapField);
        Map cacheMap = (Map)ReflectionUtils.getField((Field)mapField, (Object)fsCache);
        if (Objects.isNull(cacheMap) || cacheMap.isEmpty()) {
            return Collections.emptyList();
        }
        return Collections.unmodifiableSet(cacheMap.keySet());
    }

    private UserGroupInformation getUGI(Object key) throws NoSuchFieldException {
        Field ugiField = key.getClass().getDeclaredField("ugi");
        ReflectionUtils.makeAccessible((Field)ugiField);
        return (UserGroupInformation)ReflectionUtils.getField((Field)ugiField, (Object)key);
    }

    private boolean updatable(UserGroupInformation ugi, UserGroupInformation current) {
        if (Objects.isNull(ugi) || Objects.equals(ugi, current) || !Objects.equals(ugi.getUserName(), current.getUserName())) {
            return false;
        }
        return this.getTokens(current.getCredentials()).stream().anyMatch(token -> this.getDelegationTokenIdentifier((Token<? extends TokenIdentifier>)token).map(AbstractDelegationTokenIdentifier::getSequenceNumber).filter(dtSeq -> this.getTokens(ugi.getCredentials()).stream().filter(otk -> Objects.equals(token.getKind(), otk.getKind()) && Objects.equals(token.getService(), otk.getService())).anyMatch(otk -> this.getDelegationTokenIdentifier((Token<? extends TokenIdentifier>)otk).map(AbstractDelegationTokenIdentifier::getSequenceNumber).filter(odtSeq -> odtSeq < dtSeq).isPresent())).isPresent());
    }

    private Optional<AbstractDelegationTokenIdentifier> getDelegationTokenIdentifier(Token<? extends TokenIdentifier> token) {
        try {
            TokenIdentifier ti = token.decodeIdentifier();
            if (ti instanceof AbstractDelegationTokenIdentifier) {
                return Optional.of((AbstractDelegationTokenIdentifier)ti);
            }
        }
        catch (IOException e) {
            logger.debug("Failed to decode token {}", token, (Object)e);
        }
        return Optional.empty();
    }

    private void updateTokens(UserGroupInformation ugi, Collection<Token<? extends TokenIdentifier>> tokens) throws NoSuchMethodException, NoSuchFieldException {
        Credentials creds = this.getCredentialsInternal(ugi);
        if (Objects.isNull(creds)) {
            return;
        }
        Map<Text, Token<? extends TokenIdentifier>> oldInternalTokens = this.getTokenMapInternal(creds);
        tokens.forEach(token -> this.updateTokensInternal((Token<? extends TokenIdentifier>)token, ugi, oldInternalTokens));
    }

    private void updateTokensInternal(Token<? extends TokenIdentifier> token, UserGroupInformation ugi, Map<Text, Token<? extends TokenIdentifier>> oldInternalTokens) {
        this.getDelegationTokenIdentifier(token).map(AbstractDelegationTokenIdentifier::getSequenceNumber).ifPresent(dtSeq -> oldInternalTokens.forEach((key, otk) -> {
            if (!Objects.equals(token.getKind(), otk.getKind())) {
                return;
            }
            this.getDelegationTokenIdentifier((Token<? extends TokenIdentifier>)otk).map(AbstractDelegationTokenIdentifier::getSequenceNumber).ifPresent(odtSeq -> {
                if (odtSeq < dtSeq && this.isPrivateCloneOf((Token<? extends TokenIdentifier>)otk, token.getService())) {
                    this.privateClone(token, otk.getService()).ifPresent(tk -> ugi.addToken(key, tk));
                }
            });
        }));
    }

    private boolean isPrivateCloneOf(Token<? extends TokenIdentifier> token, Text service) {
        try {
            Method privateMethod = token.getClass().getDeclaredMethod("isPrivateCloneOf", new Class[0]);
            ReflectionUtils.makeAccessible((Method)privateMethod);
            Object boolObj = ReflectionUtils.invokeMethod((Method)privateMethod, token, (Object[])new Object[]{service});
            if (Objects.isNull(boolObj)) {
                return false;
            }
            return (Boolean)boolObj;
        }
        catch (NoSuchMethodException e) {
            logger.debug("Failed to get method 'isPrivateCloneOf', hadoop version not supported (since 2.8.2).");
            return false;
        }
    }

    private Optional<Token<? extends TokenIdentifier>> privateClone(Token<? extends TokenIdentifier> token, Text service) {
        try {
            Method cloneMethod = token.getClass().getDeclaredMethod("privateClone", new Class[0]);
            ReflectionUtils.makeAccessible((Method)cloneMethod);
            Object tkObj = ReflectionUtils.invokeMethod((Method)cloneMethod, token, (Object[])new Object[]{service});
            if (Objects.isNull(tkObj)) {
                return Optional.empty();
            }
            return Optional.of((Token)tkObj);
        }
        catch (NoSuchMethodException e) {
            logger.debug("Failed to get method 'privateClone', hadoop version not supported (since 2.8.2).");
            return Optional.empty();
        }
    }

    private Credentials getCredentialsInternal(UserGroupInformation ugi) throws NoSuchMethodException {
        Method credsMethod = UserGroupInformation.class.getDeclaredMethod("getCredentialsInternal", new Class[0]);
        ReflectionUtils.makeAccessible((Method)credsMethod);
        return (Credentials)ReflectionUtils.invokeMethod((Method)credsMethod, (Object)ugi);
    }

    private Map<Text, Token<? extends TokenIdentifier>> getTokenMapInternal(Credentials creds) throws NoSuchFieldException {
        Field mapFiled = Credentials.class.getDeclaredField("tokenMap");
        ReflectionUtils.makeAccessible((Field)mapFiled);
        Map internalTokenMap = (Map)ReflectionUtils.getField((Field)mapFiled, (Object)creds);
        if (Objects.isNull(internalTokenMap)) {
            return Collections.emptyMap();
        }
        HashMap tokenMap = Maps.newHashMap((Map)internalTokenMap);
        return Collections.unmodifiableMap(tokenMap);
    }

    private void doRenewTGTCache() throws IOException {
        logger.info("Prepare credential cache by 'kinit -kt {} {}' at KRB5CCNAME: {}", new Object[]{this.keytab, this.principal, System.getenv("KRB5CCNAME")});
        Shell.execCommand((String[])new String[]{"kinit", "-kt", this.keytab, this.principal});
    }

    private void doLogin() throws IOException {
        String platform;
        this.doRenewTGTCache();
        logger.info("Login kerberos from principal: {}, keytab: {}.", (Object)this.principal, (Object)this.keytab);
        switch (platform = this.kapConf.getKerberosPlatform()) {
            case "Standard": {
                this.loginStandardPlatform();
                break;
            }
            case "FI": 
            case "TDH": {
                this.loginNonStandardPlatform();
                break;
            }
            default: {
                throw new InvalidParameterException("Unknown platform: " + platform + ", please check 'kylin.kerberos.platform'.");
            }
        }
    }

    private void loginNonStandardPlatform() throws IOException {
        Unsafe.setProperty((String)"zookeeper.sasl.client", (String)"true");
        Unsafe.setProperty((String)"java.security.auth.login.config", (String)this.kapConf.getKerberosJaasConfPath());
        Unsafe.setProperty((String)"java.security.krb5.conf", (String)this.kapConf.getKerberosKrb5ConfPath());
        KerberosLoginUtil.setJaasConf(CONTEXT_NAME, this.principal, this.keytab);
        KerberosLoginUtil.setZookeeperServerPrincipal(this.kapConf.getKerberosZKPrincipal());
        KerberosLoginUtil.login(this.principal, this.keytab, this.kapConf.getKerberosKrb5ConfPath(), CONFIGURATION);
    }

    private void loginStandardPlatform() throws IOException {
        UserGroupInformation.loginUserFromKeytab((String)this.principal, (String)this.keytab);
        logger.info("Login kerberos success.");
    }

    private void preCheck() {
        Preconditions.checkState((boolean)KerberosLoginUtil.checkKeyTabIsExist(this.keytab), (String)"The key tab is not exist : %s", (Object)this.keytab);
        Preconditions.checkState((boolean)KerberosLoginUtil.checkKeyTabIsValid(this.keytab), (String)"The key tab is invalid : %s", (Object)this.keytab);
    }

    private void stop() {
        if (this.renewalExecutor != null) {
            this.renewalExecutor.shutdownNow();
        }
    }
}

