/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.document.rdb;

import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import java.io.IOException;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.SQLTransientException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Locale;
import java.util.TreeMap;
import org.apache.jackrabbit.oak.plugins.document.DocumentStoreException;
import org.apache.jackrabbit.oak.plugins.document.util.UTF8Encoder;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RDBJDBCTools {
    private static final Logger LOG = LoggerFactory.getLogger(RDBJDBCTools.class);
    public static final int MAX_IN_CLAUSE = Integer.getInteger("org.apache.jackrabbit.oak.plugins.document.rdb.RDBJDBCTools.MAX_IN_CLAUSE", 2048);

    protected static String jdbctype(String jdbcurl) {
        if (jdbcurl == null) {
            return null;
        }
        String t = jdbcurl.toLowerCase(Locale.ENGLISH);
        if (!t.startsWith("jdbc:")) {
            return null;
        }
        int p = (t = t.substring("jbdc:".length())).indexOf(":");
        if (p <= 0) {
            return t;
        }
        return t.substring(0, p);
    }

    protected static String driverForDBType(String type) {
        if ("h2".equals(type)) {
            return "org.h2.Driver";
        }
        if ("derby".equals(type)) {
            return "org.apache.derby.jdbc.EmbeddedDriver";
        }
        if ("postgresql".equals(type)) {
            return "org.postgresql.Driver";
        }
        if ("db2".equals(type)) {
            return "com.ibm.db2.jcc.DB2Driver";
        }
        if ("mysql".equals(type)) {
            return "com.mysql.jdbc.Driver";
        }
        if ("oracle".equals(type)) {
            return "oracle.jdbc.OracleDriver";
        }
        if ("sqlserver".equals(type)) {
            return "com.microsoft.sqlserver.jdbc.SQLServerDriver";
        }
        return "";
    }

    @NotNull
    private static String checkLegalTableName(@NotNull String tableName) throws IllegalArgumentException {
        for (int i = 0; i < tableName.length(); ++i) {
            char c = tableName.charAt(i);
            if (c >= 'a' && c <= 'z' || c >= 'A' && c <= 'Z' || c >= '0' && c <= '9' || c == '_') continue;
            throw new IllegalArgumentException("Invalid character '" + c + "' in table name '" + tableName + "'");
        }
        return tableName;
    }

    @NotNull
    protected static String createTableName(@NotNull String prefix, @NotNull String basename) throws IllegalArgumentException {
        String p = RDBJDBCTools.checkLegalTableName(prefix);
        String b = RDBJDBCTools.checkLegalTableName(basename);
        if (p.length() != 0 && !p.endsWith("_")) {
            p = p + "_";
        }
        return p + b;
    }

    @NotNull
    protected static String isolationLevelToString(int isolationLevel) {
        String name;
        switch (isolationLevel) {
            case 0: {
                name = "TRANSACTION_NONE";
                break;
            }
            case 2: {
                name = "TRANSACTION_READ_COMMITTED";
                break;
            }
            case 1: {
                name = "TRANSACTION_READ_UNCOMMITTED";
                break;
            }
            case 4: {
                name = "TRANSACTION_REPEATABLE_READ";
                break;
            }
            case 8: {
                name = "TRANSACTION_SERIALIZABLE";
                break;
            }
            default: {
                name = "unknown";
            }
        }
        return String.format("%s (%d)", name, isolationLevel);
    }

    private static String dumpColumnMeta(String columnName, int type, String typeName, int precision) {
        boolean skipPrecision = precision == 0 || type == 5 && precision == 5 || type == -5 && precision == 19;
        return skipPrecision ? String.format("%s %s", columnName, typeName) : String.format("%s %s(%d)", columnName, typeName, precision);
    }

    protected static String dumpResultSetMeta(ResultSetMetaData met) {
        try {
            StringBuilder sb = new StringBuilder();
            sb.append(String.format("%s.%s: ", met.getSchemaName(1).trim(), met.getTableName(1).trim()));
            TreeMap<String, Integer> types = new TreeMap<String, Integer>();
            for (int i = 1; i <= met.getColumnCount(); ++i) {
                if (i > 1) {
                    sb.append(", ");
                }
                sb.append(RDBJDBCTools.dumpColumnMeta(met.getColumnName(i), met.getColumnType(i), met.getColumnTypeName(i), met.getPrecision(i)));
                types.put(met.getColumnTypeName(i), met.getColumnType(i));
            }
            sb.append(" /* " + ((Object)types).toString() + " */");
            return sb.toString();
        }
        catch (SQLException ex) {
            return "Column metadata unavailable: " + ex.getMessage();
        }
    }

    @NotNull
    protected static String getAdditionalMessages(SQLException ex) {
        ArrayList<String> messages = new ArrayList<String>();
        String message = ex.getMessage();
        for (SQLException next = ex.getNextException(); next != null; next = next.getNextException()) {
            String m = next.getMessage();
            if (message.equals(m)) continue;
            messages.add(m);
        }
        return messages.isEmpty() ? "" : ((Object)messages).toString();
    }

    protected static boolean matchesSQLState(SQLException ex, String ... statePrefix) {
        String state = ex.getSQLState();
        if (state != null) {
            for (String sp : statePrefix) {
                if (!state.startsWith(sp)) continue;
                return true;
            }
        }
        return false;
    }

    protected static String versionCheck(DatabaseMetaData md, int dbmax, int dbmin, int drmax, int drmin, String dbname) throws SQLException {
        int min;
        int maj;
        StringBuilder result = new StringBuilder();
        if (dbmax != -1) {
            maj = md.getDatabaseMajorVersion();
            min = md.getDatabaseMinorVersion();
            if (maj < dbmax || maj == dbmax && min < dbmin) {
                result.append("Unsupported " + dbname + " version: " + maj + "." + min + ", expected at least " + dbmax + "." + dbmin);
            }
        }
        if (drmax != -1) {
            maj = md.getDriverMajorVersion();
            min = md.getDriverMinorVersion();
            if (maj < drmax || maj == drmax && min < drmin) {
                if (result.length() != 0) {
                    result.append(", ");
                }
                result.append("Unsupported " + dbname + " driver version: " + md.getDriverName() + " " + maj + "." + min + ", expected at least " + drmax + "." + drmin);
            }
        }
        return result.toString();
    }

    protected static String versionCheck(DatabaseMetaData md, int dbmax, int dbmin, String dbname) throws SQLException {
        return RDBJDBCTools.versionCheck(md, dbmax, dbmin, -1, -1, dbname);
    }

    protected static <T extends Statement> T closeStatement(@Nullable T stmt) {
        if (stmt != null) {
            try {
                stmt.close();
            }
            catch (SQLException ex) {
                LOG.debug("Closing statement", (Throwable)ex);
            }
        }
        return null;
    }

    protected static ResultSet closeResultSet(@Nullable ResultSet rs) {
        if (rs != null) {
            try {
                rs.close();
            }
            catch (SQLException ex) {
                LOG.debug("Closing result set", (Throwable)ex);
            }
        }
        return null;
    }

    protected static void appendInCondition(StringBuilder builder, String field, int placeholdersCount, int maxListLength) {
        if (placeholdersCount == 1) {
            builder.append(field).append(" = ?");
        } else if (placeholdersCount > 0) {
            int i;
            if (placeholdersCount > maxListLength) {
                builder.append('(');
            }
            for (i = 0; i < placeholdersCount / maxListLength; ++i) {
                if (i > 0) {
                    builder.append(" or ");
                }
                RDBJDBCTools.appendInCondition(builder, field, maxListLength);
            }
            if (placeholdersCount % maxListLength > 0) {
                if (i > 0) {
                    builder.append(" or ");
                }
                RDBJDBCTools.appendInCondition(builder, field, placeholdersCount % maxListLength);
            }
            if (placeholdersCount > maxListLength) {
                builder.append(')');
            }
        }
    }

    private static void appendInCondition(StringBuilder builder, String field, int placeholdersCount) {
        builder.append(field).append(" in (");
        Joiner.on((char)',').appendTo(builder, Iterables.limit((Iterable)Iterables.cycle((Object[])new Character[]{Character.valueOf('?')}), (int)placeholdersCount));
        builder.append(')');
    }

    public static PreparedStatementComponent createInStatement(final String fieldName, final Collection<String> values, final boolean binary) {
        if (values.size() > MAX_IN_CLAUSE) {
            throw new IllegalArgumentException("Maximum size of IN clause allowed is " + MAX_IN_CLAUSE + ", but " + values.size() + " was requested");
        }
        return new PreparedStatementComponent(){

            @Override
            public String getStatementComponent() {
                StringBuilder sb = new StringBuilder(values.size() * 3);
                RDBJDBCTools.appendInCondition(sb, fieldName, values.size(), 1000);
                return sb.toString();
            }

            @Override
            public int setParameters(PreparedStatement stmt, int startIndex) throws SQLException {
                for (String value : values) {
                    try {
                        if (binary) {
                            stmt.setBytes(startIndex++, UTF8Encoder.encodeAsByteArray(value));
                            continue;
                        }
                        if (!UTF8Encoder.canEncode(value)) {
                            throw new IOException("can not encode as UTF-8");
                        }
                        stmt.setString(startIndex++, value);
                    }
                    catch (IOException ex) {
                        LOG.warn("Invalid ID: " + value, (Throwable)ex);
                        throw RDBJDBCTools.asDocumentStoreException(ex, "Invalid ID: " + value);
                    }
                }
                return startIndex;
            }
        };
    }

    private static DocumentStoreException.Type exceptionTypeFor(Exception cause) {
        return cause instanceof SQLTransientException ? DocumentStoreException.Type.TRANSIENT : DocumentStoreException.Type.GENERIC;
    }

    public static DocumentStoreException asDocumentStoreException(@NotNull Exception cause, @NotNull String message) {
        return new DocumentStoreException(message, cause, RDBJDBCTools.exceptionTypeFor(cause));
    }

    public static interface PreparedStatementComponent {
        @NotNull
        public String getStatementComponent();

        public int setParameters(PreparedStatement var1, int var2) throws SQLException;
    }
}

