/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.storageengine.dataregion.compaction.execute.utils;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.iotdb.commons.conf.CommonDescriptor;
import org.apache.iotdb.commons.exception.IllegalPathException;
import org.apache.iotdb.commons.path.MeasurementPath;
import org.apache.iotdb.commons.service.metric.MetricService;
import org.apache.iotdb.commons.service.metric.enums.Tag;
import org.apache.iotdb.commons.utils.TestOnly;
import org.apache.iotdb.db.service.metrics.CompactionMetrics;
import org.apache.iotdb.db.service.metrics.FileMetrics;
import org.apache.iotdb.db.storageengine.dataregion.compaction.constant.CompactionTaskType;
import org.apache.iotdb.db.storageengine.dataregion.compaction.schedule.CompactionTaskManager;
import org.apache.iotdb.db.storageengine.dataregion.modification.DeletionPredicate;
import org.apache.iotdb.db.storageengine.dataregion.modification.IDPredicate;
import org.apache.iotdb.db.storageengine.dataregion.modification.ModEntry;
import org.apache.iotdb.db.storageengine.dataregion.modification.ModificationFile;
import org.apache.iotdb.db.storageengine.dataregion.modification.TableDeletionEntry;
import org.apache.iotdb.db.storageengine.dataregion.modification.TreeDeletionEntry;
import org.apache.iotdb.db.storageengine.dataregion.tsfile.TsFileResource;
import org.apache.iotdb.db.storageengine.dataregion.tsfile.timeindex.ArrayDeviceTimeIndex;
import org.apache.iotdb.metrics.utils.MetricLevel;
import org.apache.iotdb.metrics.utils.SystemMetric;
import org.apache.tsfile.file.metadata.ChunkMetadata;
import org.apache.tsfile.file.metadata.IDeviceID;
import org.apache.tsfile.fileSystem.FSFactoryProducer;
import org.apache.tsfile.read.common.TimeRange;
import org.apache.tsfile.write.writer.TsFileIOWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompactionUtils {
    private static final Logger logger = LoggerFactory.getLogger((String)"COMPACTION");
    private static final String SYSTEM = "system";

    private CompactionUtils() {
    }

    public static void moveTargetFile(List<TsFileResource> targetResources, CompactionTaskType type, String fullStorageGroupName) throws IOException {
        String fileSuffix = CompactionUtils.getTmpFileSuffix(type);
        for (TsFileResource targetResource : targetResources) {
            if (targetResource == null) continue;
            CompactionUtils.moveOneTargetFile(targetResource, fileSuffix, fullStorageGroupName);
        }
    }

    public static String getTmpFileSuffix(CompactionTaskType type) {
        switch (type) {
            case INNER_UNSEQ: 
            case INNER_SEQ: 
            case REPAIR: {
                return ".inner";
            }
            case CROSS: {
                return ".cross";
            }
            case SETTLE: {
                return ".settle";
            }
        }
        logger.error("Current task type {} does not have tmp file suffix.", (Object)type);
        return "";
    }

    private static void moveOneTargetFile(TsFileResource targetResource, String tmpFileSuffix, String fullStorageGroupName) throws IOException {
        if (!targetResource.getTsFile().exists()) {
            logger.info("{} [Compaction] Tmp target tsfile {} may be deleted after compaction.", (Object)fullStorageGroupName, (Object)targetResource.getTsFilePath());
            return;
        }
        File newFile = new File(targetResource.getTsFilePath().replace(tmpFileSuffix, ".tsfile"));
        if (!newFile.exists()) {
            FSFactoryProducer.getFSFactory().moveFile(targetResource.getTsFile(), newFile);
        }
        targetResource.setFile(newFile);
        targetResource.serialize();
        targetResource.closeWithoutSettingStatus();
    }

    public static void combineModsInCrossCompaction(List<TsFileResource> seqResources, List<TsFileResource> unseqResources, List<TsFileResource> targetResources) throws IOException {
        HashSet<ModEntry> modifications = new HashSet<ModEntry>();
        for (TsFileResource unseqFile : unseqResources) {
            modifications.addAll(ModificationFile.readAllCompactionModifications(unseqFile.getTsFile()));
        }
        for (int i = 0; i < targetResources.size(); ++i) {
            TsFileResource targetResource = targetResources.get(i);
            if (targetResource == null) continue;
            HashSet<ModEntry> seqModifications = new HashSet<ModEntry>(ModificationFile.readAllCompactionModifications(seqResources.get(i).getTsFile()));
            modifications.addAll(seqModifications);
            CompactionUtils.updateOneTargetMods(targetResource, modifications);
            modifications.removeAll(seqModifications);
        }
    }

    @SafeVarargs
    public static void prepareCompactionModFiles(List<TsFileResource> targets, List<TsFileResource> ... sourceLists) throws IOException {
        Set<ModificationFile> compactionModFileSet = targets.stream().map(tsFileResource -> {
            try {
                return tsFileResource.getModFileForWrite();
            }
            catch (IOException e) {
                logger.error("Can not get mod file of {}", tsFileResource, (Object)e);
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toSet());
        for (List<TsFileResource> sourceList : sourceLists) {
            for (TsFileResource tsFileResource2 : sourceList) {
                tsFileResource2.getModFileForWrite().setCascadeFile(compactionModFileSet);
            }
        }
    }

    public static void combineModsInInnerCompaction(Collection<TsFileResource> sourceFiles, TsFileResource targetTsFile) throws IOException {
        HashSet<ModEntry> modifications = new HashSet<ModEntry>();
        for (TsFileResource mergeTsFile : sourceFiles) {
            ModificationFile sourceCompactionModificationFile = mergeTsFile.getCompactionModFile();
            try {
                modifications.addAll(sourceCompactionModificationFile.getAllMods());
            }
            finally {
                if (sourceCompactionModificationFile == null) continue;
                sourceCompactionModificationFile.close();
            }
        }
        CompactionUtils.updateOneTargetMods(targetTsFile, modifications);
    }

    public static void combineModsInInnerCompaction(Collection<TsFileResource> sourceFiles, List<TsFileResource> targetTsFiles) throws IOException {
        HashSet<ModEntry> modifications = new HashSet<ModEntry>();
        for (TsFileResource mergeTsFile : sourceFiles) {
            ModificationFile sourceCompactionModificationFile = mergeTsFile.getCompactionModFile();
            try {
                modifications.addAll(sourceCompactionModificationFile.getAllMods());
            }
            finally {
                if (sourceCompactionModificationFile == null) continue;
                sourceCompactionModificationFile.close();
            }
        }
        for (TsFileResource targetTsFile : targetTsFiles) {
            CompactionUtils.updateOneTargetMods(targetTsFile, modifications);
        }
    }

    public static void addFilesToFileMetrics(TsFileResource resource) {
        FileMetrics.getInstance().addTsFile(resource.getDatabaseName(), resource.getDataRegionId(), resource.getTsFile().length(), resource.isSeq(), resource.getTsFile().getName());
    }

    private static void updateOneTargetMods(TsFileResource targetFile, Set<ModEntry> modifications) throws IOException {
        if (!modifications.isEmpty()) {
            try (ModificationFile modificationFile = targetFile.getModFileForWrite();){
                for (ModEntry modification : modifications) {
                    modificationFile.write(modification);
                }
            }
        }
    }

    public static void deleteCompactionModsFile(List<TsFileResource> selectedSeqTsFileResourceList, List<TsFileResource> selectedUnSeqTsFileResourceList) throws IOException {
        for (TsFileResource seqFile : selectedSeqTsFileResourceList) {
            seqFile.removeCompactionModFile();
        }
        for (TsFileResource unseqFile : selectedUnSeqTsFileResourceList) {
            unseqFile.removeCompactionModFile();
        }
    }

    public static boolean deleteTsFilesInDisk(Collection<TsFileResource> mergeTsFiles, String storageGroupName) {
        logger.info("{} [Compaction] Compaction starts to delete real file ", (Object)storageGroupName);
        boolean result = true;
        for (TsFileResource mergeTsFile : mergeTsFiles) {
            if (!mergeTsFile.remove()) {
                result = false;
            }
            logger.info("{} [Compaction] delete TsFile {}", (Object)storageGroupName, (Object)mergeTsFile.getTsFilePath());
        }
        return result;
    }

    @TestOnly
    public static void deleteModificationForSourceFile(Collection<TsFileResource> sourceFiles, String storageGroupName) throws IOException {
        logger.info("{} [Compaction] Start to delete modifications of source files", (Object)storageGroupName);
        for (TsFileResource tsFileResource : sourceFiles) {
            tsFileResource.removeModFile();
        }
    }

    public static void updateResource(TsFileResource resource, TsFileIOWriter tsFileIoWriter, IDeviceID deviceId) {
        List chunkMetadatasOfCurrentDevice = tsFileIoWriter.getChunkMetadataListOfCurrentDeviceInMemory();
        if (chunkMetadatasOfCurrentDevice != null) {
            for (ChunkMetadata chunkMetadata : chunkMetadatasOfCurrentDevice) {
                if (chunkMetadata.getMask() == 64) continue;
                resource.updateStartTime(deviceId, chunkMetadata.getStatistics().getStartTime());
                resource.updateEndTime(deviceId, chunkMetadata.getStatistics().getEndTime());
            }
        }
    }

    public static void updateProgressIndexAndMark(List<TsFileResource> targetResources, List<TsFileResource> seqResources, List<TsFileResource> unseqResources) {
        for (TsFileResource targetResource : targetResources) {
            for (TsFileResource unseqResource : unseqResources) {
                targetResource.updateProgressIndex(unseqResource.getMaxProgressIndexAfterClose());
                targetResource.setGeneratedByPipe(unseqResource.isGeneratedByPipe() && targetResource.isGeneratedByPipe());
                targetResource.setGeneratedByPipeConsensus(unseqResource.isGeneratedByPipeConsensus() && targetResource.isGeneratedByPipeConsensus());
            }
            for (TsFileResource seqResource : seqResources) {
                targetResource.updateProgressIndex(seqResource.getMaxProgressIndexAfterClose());
                targetResource.setGeneratedByPipe(seqResource.isGeneratedByPipe() && targetResource.isGeneratedByPipe());
                targetResource.setGeneratedByPipeConsensus(seqResource.isGeneratedByPipeConsensus() && targetResource.isGeneratedByPipeConsensus());
            }
        }
    }

    public static void updatePlanIndexes(List<TsFileResource> targetResources, List<TsFileResource> seqResources, List<TsFileResource> unseqResources) {
        for (TsFileResource targetResource : targetResources) {
            for (TsFileResource unseqResource : unseqResources) {
                targetResource.updatePlanIndexes(unseqResource);
            }
            for (TsFileResource seqResource : seqResources) {
                targetResource.updatePlanIndexes(seqResource);
            }
        }
    }

    public static void deleteSourceTsFileAndUpdateFileMetrics(List<TsFileResource> sourceSeqResourceList, List<TsFileResource> sourceUnseqResourceList) {
        CompactionUtils.deleteSourceTsFileAndUpdateFileMetrics(sourceSeqResourceList, true);
        CompactionUtils.deleteSourceTsFileAndUpdateFileMetrics(sourceUnseqResourceList, false);
    }

    public static void deleteSourceTsFileAndUpdateFileMetrics(List<TsFileResource> resources, boolean seq) {
        for (TsFileResource resource : resources) {
            CompactionUtils.deleteTsFileResourceWithoutLock(resource);
        }
        FileMetrics.getInstance().deleteTsFile(seq, resources);
    }

    public static void deleteTsFileResourceWithoutLock(TsFileResource resource) {
        if (!resource.remove()) {
            logger.warn("[Compaction] delete file failed, file path is {}", (Object)resource.getTsFile().getAbsolutePath());
        } else {
            logger.info("[Compaction] delete file: {}", (Object)resource.getTsFile().getAbsolutePath());
        }
    }

    public static boolean isDiskHasSpace() {
        return CompactionUtils.isDiskHasSpace(0.0);
    }

    public static boolean isDiskHasSpace(double redundancy) {
        double availableDisk = MetricService.getInstance().getAutoGauge(SystemMetric.SYS_DISK_AVAILABLE_SPACE.toString(), MetricLevel.CORE, new String[]{Tag.NAME.toString(), SYSTEM}).getValue();
        double totalDisk = MetricService.getInstance().getAutoGauge(SystemMetric.SYS_DISK_TOTAL_SPACE.toString(), MetricLevel.CORE, new String[]{Tag.NAME.toString(), SYSTEM}).getValue();
        if (availableDisk != 0.0 && totalDisk != 0.0) {
            return availableDisk / totalDisk > CommonDescriptor.getInstance().getConfig().getDiskSpaceWarningThreshold() + redundancy;
        }
        return true;
    }

    public static ArrayDeviceTimeIndex buildDeviceTimeIndex(TsFileResource resource) throws IOException {
        int readSize;
        long resourceFileSize;
        CompactionTaskManager.getInstance().getCompactionReadOperationRateLimiter().acquire(1);
        CompactionMetrics.getInstance().recordDeserializeResourceInfo(resourceFileSize);
        for (resourceFileSize = new File(new StringBuilder().append(resource.getTsFilePath()).append(".resource").toString()).length(); resourceFileSize > 0L; resourceFileSize -= (long)readSize) {
            readSize = (int)Math.min(resourceFileSize, Integer.MAX_VALUE);
            CompactionTaskManager.getInstance().getCompactionReadRateLimiter().acquire(readSize);
        }
        return resource.buildDeviceTimeIndex();
    }

    public static ModEntry convertTtlToDeletion(IDeviceID deviceID, long timeLowerBound) throws IllegalPathException {
        if (!deviceID.isTableModel()) {
            return new TreeDeletionEntry(new MeasurementPath(deviceID, "*"), Long.MIN_VALUE, timeLowerBound);
        }
        return new TableDeletionEntry(new DeletionPredicate(deviceID.getTableName(), new IDPredicate.FullExactMatch(deviceID)), new TimeRange(Long.MIN_VALUE, timeLowerBound));
    }
}

