/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.data.orc;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import org.apache.iceberg.Schema;
import org.apache.iceberg.data.GenericRecord;
import org.apache.iceberg.data.Record;
import org.apache.iceberg.orc.ORCSchemaUtil;
import org.apache.iceberg.orc.OrcValueReader;
import org.apache.iceberg.types.Types;
import org.apache.orc.TypeDescription;
import org.apache.orc.storage.ql.exec.vector.BytesColumnVector;
import org.apache.orc.storage.ql.exec.vector.ColumnVector;
import org.apache.orc.storage.ql.exec.vector.DecimalColumnVector;
import org.apache.orc.storage.ql.exec.vector.DoubleColumnVector;
import org.apache.orc.storage.ql.exec.vector.ListColumnVector;
import org.apache.orc.storage.ql.exec.vector.LongColumnVector;
import org.apache.orc.storage.ql.exec.vector.MapColumnVector;
import org.apache.orc.storage.ql.exec.vector.StructColumnVector;
import org.apache.orc.storage.ql.exec.vector.TimestampColumnVector;
import org.apache.orc.storage.ql.exec.vector.VectorizedRowBatch;

public class GenericOrcReader
implements OrcValueReader<Record> {
    private final Schema schema;
    private final List<TypeDescription> columns;
    private final Converter[] converters;
    private static final OffsetDateTime EPOCH = Instant.ofEpochSecond(0L).atOffset(ZoneOffset.UTC);
    private static final LocalDate EPOCH_DAY = EPOCH.toLocalDate();

    private GenericOrcReader(Schema expectedSchema, TypeDescription readSchema) {
        this.schema = expectedSchema;
        this.columns = readSchema.getChildren();
        this.converters = this.buildConverters();
    }

    private Converter[] buildConverters() {
        Preconditions.checkState((this.schema.columns().size() == this.columns.size() ? 1 : 0) != 0, (Object)"Expected schema must have same number of columns as projection.");
        Converter[] newConverters = new Converter[this.columns.size()];
        List icebergCols = this.schema.columns();
        for (int c = 0; c < newConverters.length; ++c) {
            newConverters[c] = GenericOrcReader.buildConverter((Types.NestedField)icebergCols.get(c), this.columns.get(c));
        }
        return newConverters;
    }

    public static OrcValueReader<Record> buildReader(Schema expectedSchema, TypeDescription fileSchema) {
        return new GenericOrcReader(expectedSchema, fileSchema);
    }

    public Record read(VectorizedRowBatch batch, int row) {
        GenericRecord rowRecord = GenericRecord.create(this.schema);
        for (int c = 0; c < batch.cols.length; ++c) {
            rowRecord.set(c, this.converters[c].convert(batch.cols[c], row));
        }
        return rowRecord;
    }

    private static Converter buildConverter(Types.NestedField icebergField, TypeDescription schema) {
        switch (schema.getCategory()) {
            case BOOLEAN: {
                return new BooleanConverter();
            }
            case BYTE: {
                return new ByteConverter();
            }
            case SHORT: {
                return new ShortConverter();
            }
            case DATE: {
                return new DateConverter();
            }
            case INT: {
                return new IntConverter();
            }
            case LONG: {
                String longAttributeValue = schema.getAttributeValue("iceberg.long-type");
                ORCSchemaUtil.LongType longType = longAttributeValue == null ? ORCSchemaUtil.LongType.LONG : ORCSchemaUtil.LongType.valueOf((String)longAttributeValue);
                switch (longType) {
                    case TIME: {
                        return new TimeConverter();
                    }
                    case LONG: {
                        return new LongConverter();
                    }
                }
                throw new IllegalStateException("Unhandled Long type found in ORC type attribute: " + longType);
            }
            case FLOAT: {
                return new FloatConverter();
            }
            case DOUBLE: {
                return new DoubleConverter();
            }
            case TIMESTAMP: {
                return new TimestampConverter();
            }
            case TIMESTAMP_INSTANT: {
                return new TimestampTzConverter();
            }
            case DECIMAL: {
                return new DecimalConverter();
            }
            case BINARY: {
                String binaryAttributeValue = schema.getAttributeValue("iceberg.binary-type");
                ORCSchemaUtil.BinaryType binaryType = binaryAttributeValue == null ? ORCSchemaUtil.BinaryType.BINARY : ORCSchemaUtil.BinaryType.valueOf((String)binaryAttributeValue);
                switch (binaryType) {
                    case UUID: {
                        return new UUIDConverter();
                    }
                    case FIXED: {
                        return new FixedConverter();
                    }
                    case BINARY: {
                        return new BinaryConverter();
                    }
                }
                throw new IllegalStateException("Unhandled Binary type found in ORC type attribute: " + binaryType);
            }
            case STRING: 
            case CHAR: 
            case VARCHAR: {
                return new StringConverter();
            }
            case STRUCT: {
                return new StructConverter(icebergField, schema);
            }
            case LIST: {
                return new ListConverter(icebergField, schema);
            }
            case MAP: {
                return new MapConverter(icebergField, schema);
            }
        }
        throw new IllegalArgumentException("Unhandled type " + schema);
    }

    private static class StructConverter
    implements Converter<Record> {
        private final Converter[] children;
        private final Schema icebergStructSchema;

        StructConverter(Types.NestedField icebergField, TypeDescription schema) {
            Preconditions.checkArgument((boolean)icebergField.type().isStructType());
            this.icebergStructSchema = new Schema(icebergField.type().asStructType().fields());
            List icebergChildren = icebergField.type().asStructType().fields();
            this.children = new Converter[schema.getChildren().size()];
            Preconditions.checkState((icebergChildren.size() == this.children.length ? 1 : 0) != 0, (Object)"Expected schema must have same number of columns as projection.");
            for (int c = 0; c < this.children.length; ++c) {
                this.children[c] = GenericOrcReader.buildConverter((Types.NestedField)icebergChildren.get(c), (TypeDescription)schema.getChildren().get(c));
            }
        }

        @Override
        public Record convertNonNullValue(ColumnVector vector, int row) {
            StructColumnVector structVector = (StructColumnVector)vector;
            GenericRecord data = GenericRecord.create(this.icebergStructSchema);
            for (int c = 0; c < this.children.length; ++c) {
                data.set(c, this.children[c].convert(structVector.fields[c], row));
            }
            return data;
        }
    }

    private static class MapConverter
    implements Converter<Map<?, ?>> {
        private final Converter keyConvert;
        private final Converter valueConvert;

        MapConverter(Types.NestedField icebergField, TypeDescription schema) {
            Preconditions.checkArgument((boolean)icebergField.type().isMapType());
            TypeDescription keyType = (TypeDescription)schema.getChildren().get(0);
            TypeDescription valueType = (TypeDescription)schema.getChildren().get(1);
            List mapFields = icebergField.type().asMapType().fields();
            this.keyConvert = GenericOrcReader.buildConverter((Types.NestedField)mapFields.get(0), keyType);
            this.valueConvert = GenericOrcReader.buildConverter((Types.NestedField)mapFields.get(1), valueType);
        }

        @Override
        public Map<?, ?> convertNonNullValue(ColumnVector vector, int row) {
            MapColumnVector mapVector = (MapColumnVector)vector;
            int offset = (int)mapVector.offsets[row];
            int length = (int)mapVector.lengths[row];
            HashMap map = Maps.newHashMapWithExpectedSize((int)length);
            for (int c = 0; c < length; ++c) {
                Object key = this.keyConvert.convert(mapVector.keys, offset + c);
                Object value = this.valueConvert.convert(mapVector.values, offset + c);
                map.put(key, value);
            }
            return map;
        }
    }

    private static class ListConverter
    implements Converter<List<?>> {
        private final Converter childConverter;

        ListConverter(Types.NestedField icebergField, TypeDescription schema) {
            Preconditions.checkArgument((boolean)icebergField.type().isListType());
            TypeDescription child = (TypeDescription)schema.getChildren().get(0);
            this.childConverter = GenericOrcReader.buildConverter((Types.NestedField)icebergField.type().asListType().fields().get(0), child);
        }

        @Override
        public List<?> convertNonNullValue(ColumnVector vector, int row) {
            ListColumnVector listVector = (ListColumnVector)vector;
            int offset = (int)listVector.offsets[row];
            int length = (int)listVector.lengths[row];
            ArrayList list = Lists.newArrayListWithExpectedSize((int)length);
            for (int c = 0; c < length; ++c) {
                list.add(this.childConverter.convert(listVector.child, offset + c));
            }
            return list;
        }
    }

    private static class DecimalConverter
    implements Converter<BigDecimal> {
        private DecimalConverter() {
        }

        @Override
        public BigDecimal convertNonNullValue(ColumnVector vector, int row) {
            DecimalColumnVector cv = (DecimalColumnVector)vector;
            return cv.vector[row].getHiveDecimal().bigDecimalValue().setScale(cv.scale);
        }
    }

    private static class StringConverter
    implements Converter<String> {
        private StringConverter() {
        }

        @Override
        public String convertNonNullValue(ColumnVector vector, int row) {
            BytesColumnVector bytesVector = (BytesColumnVector)vector;
            return new String(bytesVector.vector[row], bytesVector.start[row], bytesVector.length[row], StandardCharsets.UTF_8);
        }
    }

    private static class UUIDConverter
    implements Converter<UUID> {
        private UUIDConverter() {
        }

        @Override
        public UUID convertNonNullValue(ColumnVector vector, int row) {
            BytesColumnVector bytesVector = (BytesColumnVector)vector;
            ByteBuffer buf = ByteBuffer.wrap(bytesVector.vector[row], bytesVector.start[row], bytesVector.length[row]);
            long mostSigBits = buf.getLong();
            long leastSigBits = buf.getLong();
            return new UUID(mostSigBits, leastSigBits);
        }
    }

    private static class BinaryConverter
    implements Converter<ByteBuffer> {
        private BinaryConverter() {
        }

        @Override
        public ByteBuffer convertNonNullValue(ColumnVector vector, int row) {
            BytesColumnVector bytesVector = (BytesColumnVector)vector;
            return ByteBuffer.wrap(bytesVector.vector[row], bytesVector.start[row], bytesVector.length[row]);
        }
    }

    private static class FixedConverter
    implements Converter<byte[]> {
        private FixedConverter() {
        }

        @Override
        public byte[] convertNonNullValue(ColumnVector vector, int row) {
            BytesColumnVector bytesVector = (BytesColumnVector)vector;
            return Arrays.copyOfRange(bytesVector.vector[row], bytesVector.start[row], bytesVector.start[row] + bytesVector.length[row]);
        }
    }

    private static class TimestampConverter
    implements Converter<LocalDateTime> {
        private TimestampConverter() {
        }

        @Override
        public LocalDateTime convertNonNullValue(ColumnVector vector, int row) {
            TimestampColumnVector tcv = (TimestampColumnVector)vector;
            return Instant.ofEpochSecond(Math.floorDiv(tcv.time[row], 1000L), tcv.nanos[row]).atOffset(ZoneOffset.UTC).toLocalDateTime();
        }
    }

    private static class TimestampTzConverter
    implements Converter<OffsetDateTime> {
        private TimestampTzConverter() {
        }

        @Override
        public OffsetDateTime convertNonNullValue(ColumnVector vector, int row) {
            TimestampColumnVector tcv = (TimestampColumnVector)vector;
            return Instant.ofEpochSecond(Math.floorDiv(tcv.time[row], 1000L), tcv.nanos[row]).atOffset(ZoneOffset.UTC);
        }
    }

    private static class DoubleConverter
    implements Converter<Double> {
        private DoubleConverter() {
        }

        @Override
        public Double convertNonNullValue(ColumnVector vector, int row) {
            return ((DoubleColumnVector)vector).vector[row];
        }
    }

    private static class FloatConverter
    implements Converter<Float> {
        private FloatConverter() {
        }

        @Override
        public Float convertNonNullValue(ColumnVector vector, int row) {
            return Float.valueOf((float)((DoubleColumnVector)vector).vector[row]);
        }
    }

    private static class LongConverter
    implements Converter<Long> {
        private LongConverter() {
        }

        @Override
        public Long convertNonNullValue(ColumnVector vector, int row) {
            return ((LongColumnVector)vector).vector[row];
        }
    }

    private static class DateConverter
    implements Converter<LocalDate> {
        private DateConverter() {
        }

        @Override
        public LocalDate convertNonNullValue(ColumnVector vector, int row) {
            return EPOCH_DAY.plusDays((int)((LongColumnVector)vector).vector[row]);
        }
    }

    private static class TimeConverter
    implements Converter<LocalTime> {
        private TimeConverter() {
        }

        @Override
        public LocalTime convertNonNullValue(ColumnVector vector, int row) {
            return LocalTime.ofNanoOfDay(((LongColumnVector)vector).vector[row] * 1000L);
        }
    }

    private static class IntConverter
    implements Converter<Integer> {
        private IntConverter() {
        }

        @Override
        public Integer convertNonNullValue(ColumnVector vector, int row) {
            return (int)((LongColumnVector)vector).vector[row];
        }
    }

    private static class ShortConverter
    implements Converter<Short> {
        private ShortConverter() {
        }

        @Override
        public Short convertNonNullValue(ColumnVector vector, int row) {
            return (short)((LongColumnVector)vector).vector[row];
        }
    }

    private static class ByteConverter
    implements Converter<Byte> {
        private ByteConverter() {
        }

        @Override
        public Byte convertNonNullValue(ColumnVector vector, int row) {
            return (byte)((LongColumnVector)vector).vector[row];
        }
    }

    private static class BooleanConverter
    implements Converter<Boolean> {
        private BooleanConverter() {
        }

        @Override
        public Boolean convertNonNullValue(ColumnVector vector, int row) {
            return ((LongColumnVector)vector).vector[row] != 0L;
        }
    }

    static interface Converter<T> {
        default public T convert(ColumnVector vector, int row) {
            int rowIndex;
            int n = rowIndex = vector.isRepeating ? 0 : row;
            if (!vector.noNulls && vector.isNull[rowIndex]) {
                return null;
            }
            return this.convertNonNullValue(vector, rowIndex);
        }

        public T convertNonNullValue(ColumnVector var1, int var2);
    }
}

