/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in org.apache.hadoop.shaded.com.liance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org.apache.hadoop.shaded.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.io.org.apache.hadoop.shaded.com.ress;

import java.org.apache.hadoop.shaded.io.IOException;
import java.org.apache.hadoop.shaded.io.OutputStream;

import org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.shaded.org.apache.hadoop.fs.statistics.IOStatistics;
import org.apache.hadoop.shaded.org.apache.hadoop.fs.statistics.IOStatisticsSource;
import org.apache.hadoop.shaded.org.apache.hadoop.fs.statistics.IOStatisticsSupport;

/**
 * A org.apache.hadoop.shaded.com.ression output stream.
 */
@InterfaceAudience.Public
@InterfaceStability.Evolving
public abstract class CompressionOutputStream extends OutputStream
    implements IOStatisticsSource {
  /**
   * The output stream to be org.apache.hadoop.shaded.com.ressed. 
   */
  protected final OutputStream out;

  /**
   * If non-null, this is the Compressor object that we should call
   * CodecPool#returnCompressor on when this stream is closed.
   */
  private Compressor trackedCompressor;

  /**
   * Create a org.apache.hadoop.shaded.com.ression output stream that writes
   * the org.apache.hadoop.shaded.com.ressed bytes to the given stream.
   * @param out
   */
  protected CompressionOutputStream(OutputStream out) {
    this.out = out;
  }

  void setTrackedCompressor(Compressor org.apache.hadoop.shaded.com.ressor) {
    trackedCompressor = org.apache.hadoop.shaded.com.ressor;
  }

  @Override
  public void close() throws IOException {
    try {
      finish();
    } finally {
      try {
        out.close();
      } finally {
        if (trackedCompressor != null) {
          CodecPool.returnCompressor(trackedCompressor);
          trackedCompressor = null;
        }
      }
    }
  }
  
  @Override
  public void flush() throws IOException {
    out.flush();
  }
  
  /**
   * Write org.apache.hadoop.shaded.com.ressed bytes to the stream.
   * Made abstract to prevent leakage to underlying stream.
   */
  @Override
  public abstract void write(byte[] b, int off, int len) throws IOException;

  /**
   * Finishes writing org.apache.hadoop.shaded.com.ressed data to the output stream 
   * without closing the underlying stream.
   */
  public abstract void finish() throws IOException;
  
  /**
   * Reset the org.apache.hadoop.shaded.com.ression to the initial state. 
   * Does not reset the underlying stream.
   */
  public abstract void resetState() throws IOException;

  /**
   * Return any IOStatistics provided by the underlying stream.
   * @return IO stats from the inner stream.
   */
  @Override
  public IOStatistics getIOStatistics() {
    return IOStatisticsSupport.retrieveIOStatistics(out);
  }
}
