/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in org.apache.hadoop.shaded.com.liance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org.apache.hadoop.shaded.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.io.org.apache.hadoop.shaded.com.ress.zstd;

import org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.io.org.apache.hadoop.shaded.com.ress.Decompressor;
import org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.io.org.apache.hadoop.shaded.com.ress.DirectDecompressor;
import org.apache.hadoop.shaded.org.apache.hadoop.util.NativeCodeLoader;
import org.apache.hadoop.shaded.org.slf4j.Logger;
import org.apache.hadoop.shaded.org.slf4j.LoggerFactory;

import java.org.apache.hadoop.shaded.io.IOException;
import java.nio.ByteBuffer;

/**
 * A {@link Decompressor} based on the zStandard org.apache.hadoop.shaded.com.ression algorithm.
 * https://github.org.apache.hadoop.shaded.com.facebook/zstd
 */
public class ZStandardDecompressor implements Decompressor {
  private static final Logger LOG =
      LoggerFactory.getLogger(ZStandardDecompressor.class);

  private long stream;
  private int directBufferSize;
  private ByteBuffer org.apache.hadoop.shaded.com.ressedDirectBuf = null;
  private int org.apache.hadoop.shaded.com.ressedDirectBufOff, bytesInCompressedBuffer;
  private ByteBuffer uncompressedDirectBuf = null;
  private byte[] userBuf = null;
  private int userBufOff = 0, userBufferBytesToConsume = 0;
  private boolean finished;
  private int remaining = 0;

  private static boolean nativeZStandardLoaded = false;

  static {
    if (NativeCodeLoader.isNativeCodeLoaded()) {
      try {
        // Initialize the native library
        initIDs();
        nativeZStandardLoaded = true;
      } catch (Throwable t) {
        LOG.warn("Error loading zstandard native libraries: " + t);
      }
    }
  }

  public static boolean isNativeCodeLoaded() {
    return nativeZStandardLoaded;
  }

  public static int getRecommendedBufferSize() {
    return getStreamSize();
  }

  public ZStandardDecompressor() {
    this(getStreamSize());
  }

  /**
   * Creates a new decompressor.
   */
  public ZStandardDecompressor(int bufferSize) {
    this.directBufferSize = bufferSize;
    org.apache.hadoop.shaded.com.ressedDirectBuf = ByteBuffer.allocateDirect(directBufferSize);
    uncompressedDirectBuf = ByteBuffer.allocateDirect(directBufferSize);
    uncompressedDirectBuf.position(directBufferSize);
    stream = create();
    reset();
  }

  @Override
  public void setInput(byte[] b, int off, int len) {
    if (b == null) {
      throw new NullPointerException();
    }
    if (off < 0 || len < 0 || off > b.length - len) {
      throw new ArrayIndexOutOfBoundsException();
    }

    this.userBuf = b;
    this.userBufOff = off;
    this.userBufferBytesToConsume = len;

    setInputFromSavedData();

    uncompressedDirectBuf.limit(directBufferSize);
    uncompressedDirectBuf.position(directBufferSize);
  }

  private void setInputFromSavedData() {
    org.apache.hadoop.shaded.com.ressedDirectBufOff = 0;
    bytesInCompressedBuffer = userBufferBytesToConsume;
    if (bytesInCompressedBuffer > directBufferSize) {
      bytesInCompressedBuffer = directBufferSize;
    }

    org.apache.hadoop.shaded.com.ressedDirectBuf.rewind();
    org.apache.hadoop.shaded.com.ressedDirectBuf.put(
        userBuf, userBufOff, bytesInCompressedBuffer);

    // Set the finished to false when org.apache.hadoop.shaded.com.ressedDirectBuf still
    // contains some bytes.
    if (org.apache.hadoop.shaded.com.ressedDirectBuf.position() > 0 && finished) {
      finished = false;
    }

    userBufOff += bytesInCompressedBuffer;
    userBufferBytesToConsume -= bytesInCompressedBuffer;
  }

  // dictionary is not supported
  @Override
  public void setDictionary(byte[] b, int off, int len) {
    throw new UnsupportedOperationException(
        "Dictionary support is not enabled");
  }

  @Override
  public boolean needsInput() {
    // Consume remaining org.apache.hadoop.shaded.com.ressed data?
    if (uncompressedDirectBuf.remaining() > 0) {
      return false;
    }

    // Check if we have consumed all input
    if (bytesInCompressedBuffer - org.apache.hadoop.shaded.com.ressedDirectBufOff <= 0) {
      // Check if we have consumed all user-input
      if (userBufferBytesToConsume <= 0) {
        return true;
      } else {
        setInputFromSavedData();
      }
    }
    return false;
  }

  // dictionary is not supported.
  @Override
  public boolean needsDictionary() {
    return false;
  }

  @Override
  public boolean finished() {
    // finished == true if ZSTD_decompressStream() returns 0
    // also check we have nothing left in our buffer
    return (finished && uncompressedDirectBuf.remaining() == 0);
  }

  @Override
  public int decompress(byte[] b, int off, int len)
      throws IOException {
    checkStream();
    if (b == null) {
      throw new NullPointerException();
    }
    if (off < 0 || len < 0 || off > b.length - len) {
      throw new ArrayIndexOutOfBoundsException();
    }

    // Check if there is uncompressed data
    int n = uncompressedDirectBuf.remaining();
    if (n > 0) {
      return populateUncompressedBuffer(b, off, len, n);
    }

    // Re-initialize the output direct buffer
    uncompressedDirectBuf.rewind();
    uncompressedDirectBuf.limit(directBufferSize);

    // Decompress data
    n = inflateBytesDirect(
        org.apache.hadoop.shaded.com.ressedDirectBuf,
        org.apache.hadoop.shaded.com.ressedDirectBufOff,
        bytesInCompressedBuffer,
        uncompressedDirectBuf,
        0,
        directBufferSize
    );

    // Set the finished to false when org.apache.hadoop.shaded.com.ressedDirectBuf still
    // contains some bytes.
    if (remaining > 0 && finished) {
      finished = false;
    }

    uncompressedDirectBuf.limit(n);

    // Get at most 'len' bytes
    return populateUncompressedBuffer(b, off, len, n);
  }

  /**
   * <p>Returns the number of bytes remaining in the input buffers;
   * normally called when finished() is true to determine amount of post-stream
   * data.</p>
   *
   * @return the total (non-negative) number of unprocessed bytes in input
   */
  @Override
  public int getRemaining() {
    checkStream();
    // userBuf + org.apache.hadoop.shaded.com.ressedDirectBuf
    return userBufferBytesToConsume + remaining;
  }

  /**
   * Resets everything including the input buffers (user and direct).
   */
  @Override
  public void reset() {
    checkStream();
    init(stream);
    remaining = 0;
    finished = false;
    org.apache.hadoop.shaded.com.ressedDirectBufOff = 0;
    bytesInCompressedBuffer = 0;
    uncompressedDirectBuf.limit(directBufferSize);
    uncompressedDirectBuf.position(directBufferSize);
    userBufOff = 0;
    userBufferBytesToConsume = 0;
  }

  @Override
  public void end() {
    if (stream != 0) {
      free(stream);
      stream = 0;
    }
  }

  @Override
  protected void finalize() {
    reset();
  }

  private void checkStream() {
    if (stream == 0) {
      throw new NullPointerException("Stream not initialized");
    }
  }

  private int populateUncompressedBuffer(byte[] b, int off, int len, int n) {
    n = Math.min(n, len);
    uncompressedDirectBuf.get(b, off, n);
    return n;
  }

  private native static void initIDs();
  private native static long create();
  private native static void init(long stream);
  private native int inflateBytesDirect(ByteBuffer src, int srcOffset,
      int srcLen, ByteBuffer dst, int dstOffset, int dstLen);
  private native static void free(long strm);
  private native static int getStreamSize();

  int inflateDirect(ByteBuffer src, ByteBuffer dst) throws IOException {
    assert
        (this instanceof ZStandardDecompressor.ZStandardDirectDecompressor);

    int originalPosition = dst.position();
    int n = inflateBytesDirect(
        src, src.position(), src.limit(), dst, dst.position(),
        dst.limit()
    );
    dst.position(originalPosition + n);
    if (bytesInCompressedBuffer > 0) {
      src.position(org.apache.hadoop.shaded.com.ressedDirectBufOff);
    } else {
      src.position(src.limit());
    }
    return n;
  }

  /**
   * A {@link DirectDecompressor} for ZStandard
   * https://github.org.apache.hadoop.shaded.com.facebook/zstd.
   */
  public static class ZStandardDirectDecompressor
      extends ZStandardDecompressor implements DirectDecompressor {

    public ZStandardDirectDecompressor(int directBufferSize) {
      super(directBufferSize);
    }

    @Override
    public boolean finished() {
      return (endOfInput && super.finished());
    }

    @Override
    public void reset() {
      super.reset();
      endOfInput = true;
    }

    private boolean endOfInput;

    @Override
    public void decompress(ByteBuffer src, ByteBuffer dst)
        throws IOException {
      assert dst.isDirect() : "dst.isDirect()";
      assert src.isDirect() : "src.isDirect()";
      assert dst.remaining() > 0 : "dst.remaining() > 0";
      this.inflateDirect(src, dst);
      endOfInput = !src.hasRemaining();
    }

    @Override
    public void setDictionary(byte[] b, int off, int len) {
      throw new UnsupportedOperationException(
          "byte[] arrays are not supported for DirectDecompressor");
    }

    @Override
    public int decompress(byte[] b, int off, int len) {
      throw new UnsupportedOperationException(
          "byte[] arrays are not supported for DirectDecompressor");
    }
  }
}
