/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.service.modules.orchestration;

import com.github.rholder.retry.AttemptTimeLimiters;
import com.github.rholder.retry.RetryException;
import com.github.rholder.retry.Retryer;
import com.github.rholder.retry.RetryerBuilder;
import com.github.rholder.retry.StopStrategies;
import com.github.rholder.retry.WaitStrategies;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.io.Closer;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.gobblin.service.modules.orchestration.AzkabanClientException;
import org.apache.gobblin.service.modules.orchestration.AzkabanClientStatus;
import org.apache.gobblin.service.modules.orchestration.AzkabanExecuteFlowStatus;
import org.apache.gobblin.service.modules.orchestration.AzkabanFetchExecuteFlowStatus;
import org.apache.gobblin.service.modules.orchestration.AzkabanGetProxyUsersStatus;
import org.apache.gobblin.service.modules.orchestration.AzkabanMultiCallables;
import org.apache.gobblin.service.modules.orchestration.AzkabanProjectFlowsStatus;
import org.apache.gobblin.service.modules.orchestration.AzkabanSessionManager;
import org.apache.gobblin.service.modules.orchestration.InvalidSessionException;
import org.apache.gobblin.service.modules.orchestration.SessionManager;
import org.apache.gobblin.service.modules.orchestration.UnreachableStatementException;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.conn.socket.LayeredConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.BasicHttpClientConnectionManager;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.ssl.TrustStrategy;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AzkabanClient
implements Closeable {
    protected final String username;
    protected final String password;
    protected final String url;
    protected final long sessionExpireInMin;
    protected SessionManager sessionManager;
    protected String sessionId;
    protected long sessionCreationTime = 0L;
    protected CloseableHttpClient httpClient;
    private ExecutorService executorService;
    private Closer closer = Closer.create();
    private Retryer<AzkabanClientStatus> retryer;
    private static Logger log = LoggerFactory.getLogger(AzkabanClient.class);
    private Duration requestTimeout;

    protected AzkabanClient(String username, String password, String url, long sessionExpireInMin, CloseableHttpClient httpClient, SessionManager sessionManager, ExecutorService executorService, Duration requestTimeout) throws AzkabanClientException {
        this.username = username;
        this.password = password;
        this.url = url;
        this.sessionExpireInMin = sessionExpireInMin;
        this.httpClient = httpClient;
        this.sessionManager = sessionManager;
        this.executorService = executorService;
        this.requestTimeout = (Duration)ObjectUtils.defaultIfNull((Object)requestTimeout, (Object)Duration.ofSeconds(10L));
        this.initializeClient();
        this.initializeSessionManager();
        this.intializeExecutorService();
        this.retryer = RetryerBuilder.newBuilder().retryIfExceptionOfType(InvalidSessionException.class).withAttemptTimeLimiter(AttemptTimeLimiters.fixedTimeLimit((long)this.requestTimeout.toMillis(), (TimeUnit)TimeUnit.MILLISECONDS, (ExecutorService)this.executorService)).withWaitStrategy(WaitStrategies.exponentialWait((long)60L, (TimeUnit)TimeUnit.SECONDS)).withStopStrategy(StopStrategies.stopAfterAttempt((int)3)).build();
        try {
            this.sessionId = this.sessionManager.fetchSession();
        }
        catch (Exception e) {
            this.sessionId = null;
            this.sessionCreationTime = -1L;
            log.error("Failed to fetch session in constructor due to: ", (Throwable)e);
            return;
        }
        this.sessionCreationTime = System.nanoTime();
    }

    private void initializeClient() throws AzkabanClientException {
        if (this.httpClient == null) {
            this.httpClient = this.createHttpClient();
            this.closer.register((Closeable)this.httpClient);
        }
    }

    private void initializeSessionManager() {
        if (this.sessionManager == null) {
            this.sessionManager = new AzkabanSessionManager(this.httpClient, this.url, this.username, this.password);
        }
    }

    private void intializeExecutorService() {
        if (this.executorService == null) {
            this.executorService = Executors.newFixedThreadPool(30);
        }
    }

    private CloseableHttpClient createHttpClient() throws AzkabanClientException {
        try {
            SSLContextBuilder sslcb = new SSLContextBuilder();
            sslcb.loadTrustMaterial(null, (TrustStrategy)new TrustSelfSignedStrategy());
            SSLConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(sslcb.build());
            HttpClientBuilder builder = HttpClientBuilder.create();
            RequestConfig requestConfig = RequestConfig.copy((RequestConfig)RequestConfig.DEFAULT).setSocketTimeout((int)this.requestTimeout.toMillis()).setConnectTimeout((int)this.requestTimeout.toMillis()).setConnectionRequestTimeout((int)this.requestTimeout.toMillis()).build();
            builder.disableCookieManagement().useSystemProperties().setDefaultRequestConfig(requestConfig).setConnectionManager((HttpClientConnectionManager)new BasicHttpClientConnectionManager()).setSSLSocketFactory((LayeredConnectionSocketFactory)sslsf);
            return builder.build();
        }
        catch (Exception e) {
            throw new AzkabanClientException("HttpClient cannot be created", e);
        }
    }

    void refreshSession(boolean forceRefresh) throws AzkabanClientException {
        boolean expired;
        Preconditions.checkArgument((this.sessionCreationTime != 0L ? 1 : 0) != 0);
        boolean bl = expired = System.nanoTime() - this.sessionCreationTime > Duration.ofMinutes(this.sessionExpireInMin).toNanos();
        if (expired) {
            log.debug("Session expired. Generating a new session.");
        } else if (forceRefresh) {
            log.info("Force to refresh session. Generating a new session.");
        }
        if (expired || forceRefresh) {
            this.sessionId = this.sessionManager.fetchSession();
            this.sessionCreationTime = System.nanoTime();
        }
    }

    protected static Map<String, String> handleResponse(HttpResponse response) throws IOException {
        AzkabanClient.verifyStatusCode(response);
        JsonObject json = AzkabanClient.getResponseJson(response);
        return AzkabanClient.getFlatMap(json);
    }

    protected static <T> T handleResponse(HttpResponse response, Class<T> responseClass) throws IOException {
        AzkabanClient.verifyStatusCode(response);
        JsonObject json = AzkabanClient.getResponseJson(response);
        Gson gson = new Gson();
        return (T)gson.fromJson((JsonElement)json, responseClass);
    }

    private static JsonObject getResponseJson(HttpResponse response) throws IOException {
        String jsonResponseString;
        HttpEntity entity = null;
        try {
            entity = response.getEntity();
            jsonResponseString = IOUtils.toString((InputStream)entity.getContent(), (String)"UTF-8");
            log.debug("Response string: {}", (Object)jsonResponseString);
        }
        catch (Exception e) {
            throw new AzkabanClientException("Cannot convert response to a string", e);
        }
        finally {
            if (entity != null) {
                EntityUtils.consume((HttpEntity)entity);
            }
        }
        return AzkabanClient.parseResponse(jsonResponseString);
    }

    protected static void verifyStatusCode(HttpResponse response) throws AzkabanClientException {
        int code = response.getStatusLine().getStatusCode();
        if (code != 201 && code != 200) {
            log.error("Failed : HTTP error code : " + response.getStatusLine().getStatusCode());
            throw new AzkabanClientException("Failed : HTTP error code : " + response.getStatusLine().getStatusCode());
        }
    }

    static Map<String, String> getFlatMap(JsonObject jsonObject) {
        if (jsonObject == null) {
            return null;
        }
        HashMap<String, String> responseMap = new HashMap<String, String>();
        for (Map.Entry entry : jsonObject.entrySet()) {
            responseMap.put((String)entry.getKey(), ((JsonElement)entry.getValue()).toString().replaceAll("\"", ""));
        }
        return responseMap;
    }

    static JsonObject parseResponse(String jsonResponseString) throws IOException {
        if (!StringUtils.isNotBlank((CharSequence)jsonResponseString)) {
            return null;
        }
        JsonObject jsonObject = new JsonParser().parse(jsonResponseString).getAsJsonObject();
        AzkabanClient.handleResponseError(jsonObject);
        return jsonObject;
    }

    private static void handleResponseError(JsonObject jsonObject) throws IOException {
        if (null != jsonObject.get("status") && "error".equalsIgnoreCase(jsonObject.get("status").toString().replaceAll("\"", ""))) {
            String message;
            String string = message = null != jsonObject.get("message") ? jsonObject.get("message").toString().replaceAll("\"", "") : "Unknown issue";
            if (message.contains("Invalid Session")) {
                throw new InvalidSessionException(message);
            }
            throw new IOException(message);
        }
        if (null != jsonObject.get("error")) {
            String error = jsonObject.get("error").toString().replaceAll("\"", "");
            throw new AzkabanClientException(error);
        }
    }

    public AzkabanClientStatus createProject(String projectName, String description) throws AzkabanClientException {
        AzkabanMultiCallables.CreateProjectCallable callable = AzkabanMultiCallables.CreateProjectCallable.builder().client(this).projectName(projectName).description(description).build();
        return this.runWithRetry(callable, AzkabanClientStatus.class);
    }

    public AzkabanClientStatus deleteProject(String projectName) throws AzkabanClientException {
        AzkabanMultiCallables.DeleteProjectCallable callable = AzkabanMultiCallables.DeleteProjectCallable.builder().client(this).projectName(projectName).build();
        return this.runWithRetry(callable, AzkabanClientStatus.class);
    }

    public Boolean projectExists(String projectName) throws AzkabanClientException {
        try {
            this.fetchProjectFlows(projectName);
            return true;
        }
        catch (AzkabanClientException e) {
            if (e.getCause().getMessage().contains("doesn't exist")) {
                return false;
            }
            throw e;
        }
    }

    public AzkabanClientStatus uploadProjectZip(String projectName, File zipFile) throws AzkabanClientException {
        AzkabanMultiCallables.UploadProjectCallable callable = AzkabanMultiCallables.UploadProjectCallable.builder().client(this).projectName(projectName).zipFile(zipFile).build();
        return this.runWithRetry(callable, AzkabanClientStatus.class);
    }

    public AzkabanExecuteFlowStatus executeFlowWithOptions(String projectName, String flowName, Map<String, String> flowOptions, Map<String, String> flowParameters) throws AzkabanClientException {
        AzkabanMultiCallables.ExecuteFlowCallable callable = AzkabanMultiCallables.ExecuteFlowCallable.builder().client(this).projectName(projectName).flowName(flowName).flowOptions(flowOptions).flowParameters(flowParameters).build();
        return this.runWithRetry(callable, AzkabanExecuteFlowStatus.class);
    }

    public AzkabanExecuteFlowStatus executeFlow(String projectName, String flowName, Map<String, String> flowParameters) throws AzkabanClientException {
        return this.executeFlowWithOptions(projectName, flowName, null, flowParameters);
    }

    public AzkabanClientStatus cancelFlow(String execId) throws AzkabanClientException {
        AzkabanMultiCallables.CancelFlowCallable callable = AzkabanMultiCallables.CancelFlowCallable.builder().client(this).execId(execId).build();
        return this.runWithRetry(callable, AzkabanClientStatus.class);
    }

    public AzkabanClientStatus fetchExecutionLog(String execId, String jobId, long offset, long length, OutputStream logStream) throws AzkabanClientException {
        AzkabanMultiCallables.FetchExecLogCallable callable = AzkabanMultiCallables.FetchExecLogCallable.builder().client(this).execId(execId).jobId(jobId).offset(offset).length(length).output(logStream).build();
        return this.runWithRetry(callable, AzkabanClientStatus.class);
    }

    public AzkabanFetchExecuteFlowStatus fetchFlowExecution(String execId) throws AzkabanClientException {
        AzkabanMultiCallables.FetchFlowExecCallable callable = AzkabanMultiCallables.FetchFlowExecCallable.builder().client(this).execId(execId).build();
        return this.runWithRetry(callable, AzkabanFetchExecuteFlowStatus.class);
    }

    public AzkabanProjectFlowsStatus fetchProjectFlows(String projectName) throws AzkabanClientException {
        AzkabanMultiCallables.FetchProjectFlowsCallable callable = AzkabanMultiCallables.FetchProjectFlowsCallable.builder().client(this).projectName(projectName).build();
        return this.runWithRetry(callable, AzkabanProjectFlowsStatus.class);
    }

    public AzkabanClientStatus addProxyUser(String projectName, String proxyUserName) throws AzkabanClientException {
        AzkabanMultiCallables.AddProxyUserCallable callable = AzkabanMultiCallables.AddProxyUserCallable.builder().client(this).projectName(projectName).proxyUserName(proxyUserName).build();
        return this.runWithRetry(callable, AzkabanClientStatus.class);
    }

    public AzkabanGetProxyUsersStatus getProxyUsers(String projectName) throws AzkabanClientException {
        AzkabanMultiCallables.GetProxyUserCallable callable = AzkabanMultiCallables.GetProxyUserCallable.builder().client(this).projectName(projectName).build();
        return this.runWithRetry(callable, AzkabanGetProxyUsersStatus.class);
    }

    private <T> T runWithRetry(Callable callable, Class<T> cls) throws AzkabanClientException {
        try {
            AzkabanClientStatus status = (AzkabanClientStatus)this.retryer.call(callable);
            if (cls.isAssignableFrom(status.getClass())) {
                return (T)status;
            }
            throw new AzkabanClientException(String.format("Unexpected response type, expected: %s actual: %s", cls, status.getClass()));
        }
        catch (ExecutionException e) {
            Throwables.propagateIfPossible((Throwable)e.getCause(), AzkabanClientException.class);
        }
        catch (RetryException e) {
            throw new AzkabanClientException("RetryException occurred ", e);
        }
        throw new UnreachableStatementException("Cannot reach here.");
    }

    @Override
    public void close() throws IOException {
        this.closer.close();
    }

    public static AzkabanClientBuilder builder() {
        return new AzkabanClientBuilder();
    }

    public static class AzkabanClientBuilder {
        private String username;
        private String password;
        private String url;
        private long sessionExpireInMin;
        private CloseableHttpClient httpClient;
        private SessionManager sessionManager;
        private ExecutorService executorService;
        private Duration requestTimeout;

        AzkabanClientBuilder() {
        }

        public AzkabanClientBuilder username(String username) {
            this.username = username;
            return this;
        }

        public AzkabanClientBuilder password(String password) {
            this.password = password;
            return this;
        }

        public AzkabanClientBuilder url(String url) {
            this.url = url;
            return this;
        }

        public AzkabanClientBuilder sessionExpireInMin(long sessionExpireInMin) {
            this.sessionExpireInMin = sessionExpireInMin;
            return this;
        }

        public AzkabanClientBuilder httpClient(CloseableHttpClient httpClient) {
            this.httpClient = httpClient;
            return this;
        }

        public AzkabanClientBuilder sessionManager(SessionManager sessionManager) {
            this.sessionManager = sessionManager;
            return this;
        }

        public AzkabanClientBuilder executorService(ExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        public AzkabanClientBuilder requestTimeout(Duration requestTimeout) {
            this.requestTimeout = requestTimeout;
            return this;
        }

        public AzkabanClient build() throws AzkabanClientException {
            return new AzkabanClient(this.username, this.password, this.url, this.sessionExpireInMin, this.httpClient, this.sessionManager, this.executorService, this.requestTimeout);
        }

        public String toString() {
            return "AzkabanClient.AzkabanClientBuilder(username=" + this.username + ", password=" + this.password + ", url=" + this.url + ", sessionExpireInMin=" + this.sessionExpireInMin + ", httpClient=" + this.httpClient + ", sessionManager=" + this.sessionManager + ", executorService=" + this.executorService + ", requestTimeout=" + this.requestTimeout + ")";
        }
    }
}

