/*
 * Decompiled with CFR 0.152.
 */
package org.apache.geode.management.internal.cli.commands;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.nio.charset.Charset;
import java.nio.file.Path;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.io.FileUtils;
import org.apache.geode.internal.cache.GemFireCacheImpl;
import org.apache.geode.internal.cache.InternalCache;
import org.apache.geode.internal.logging.LogService;
import org.apache.geode.management.internal.cli.functions.ExportLogsFunction;
import org.apache.geode.management.internal.cli.util.CommandStringBuilder;
import org.apache.geode.management.internal.configuration.utils.ZipUtils;
import org.apache.geode.test.dunit.IgnoredException;
import org.apache.geode.test.dunit.rules.ClusterStartupRule;
import org.apache.geode.test.dunit.rules.MemberVM;
import org.apache.geode.test.junit.rules.GfshCommandRule;
import org.apache.geode.test.junit.rules.LocatorStarterRule;
import org.apache.geode.test.junit.rules.Member;
import org.apache.logging.log4j.Logger;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ListAssert;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class ExportLogsDistributedTestBase {
    private static final String ERROR_LOG_PREFIX = "[IGNORE]";
    @Rule
    public ClusterStartupRule lsRule = new ClusterStartupRule().withLogFile();
    @Rule
    public GfshCommandRule gfshConnector = new GfshCommandRule();
    @Rule
    public TemporaryFolder temporaryFolder = new TemporaryFolder();
    protected MemberVM locator;
    private MemberVM server1;
    private MemberVM server2;
    private Map<MemberVM, List<LogLine>> expectedMessages;

    public File getWorkingDirectory() throws Exception {
        return this.locator.getWorkingDir();
    }

    @Before
    public void setup() throws Exception {
        Properties properties = new Properties();
        properties.setProperty("log-level", "debug");
        this.locator = this.lsRule.startLocatorVM(0, l -> (LocatorStarterRule)((LocatorStarterRule)l.withProperties(properties)).withHttpService());
        this.server1 = this.lsRule.startServerVM(1, properties, this.locator.getPort());
        this.server2 = this.lsRule.startServerVM(2, properties, this.locator.getPort());
        IgnoredException.addIgnoredException(ERROR_LOG_PREFIX);
        this.expectedMessages = new HashMap<MemberVM, List<LogLine>>();
        this.expectedMessages.put(this.locator, this.listOfLogLines(this.locator, "info", "error", "debug"));
        this.expectedMessages.put(this.server1, this.listOfLogLines(this.server1, "info", "error", "debug"));
        this.expectedMessages.put(this.server2, this.listOfLogLines(this.server2, "info", "error", "debug"));
        for (MemberVM member : this.expectedMessages.keySet()) {
            List<LogLine> logLines = this.expectedMessages.get(member);
            member.invoke(() -> {
                Logger logger = LogService.getLogger();
                logLines.forEach(logLine -> logLine.writeLog(logger));
            });
        }
        this.connect();
    }

    public void connect() throws Exception {
        this.gfshConnector.connectAndVerify(this.locator, new String[0]);
    }

    @After
    public void after() throws Exception {
        Stream.of(this.getWorkingDirectory().listFiles()).filter(f -> f.getName().endsWith(".zip")).forEach(file -> file.delete());
    }

    @Test
    public void withFiles_savedToLocatorWorkingDir() throws Exception {
        String[] extensions = new String[]{"zip"};
        this.gfshConnector.executeCommand("export logs");
        Assertions.assertThat((Iterable)FileUtils.listFiles((File)this.getWorkingDirectory(), (String[])extensions, (boolean)false)).isNotEmpty();
    }

    @Test
    public void withFiles_savedToLocatorSpecifiedRelativeDir() throws Exception {
        String[] extensions = new String[]{"zip"};
        Path workingDirPath = this.getWorkingDirectory().toPath();
        Path subdirPath = workingDirPath.resolve("some").resolve("test").resolve("directory");
        Path relativeDir = workingDirPath.relativize(subdirPath);
        this.gfshConnector.executeCommand("export logs --dir=" + relativeDir.toString());
        Assertions.assertThat((Iterable)FileUtils.listFiles((File)this.getWorkingDirectory(), (String[])extensions, (boolean)false)).isEmpty();
        Assertions.assertThat((Iterable)FileUtils.listFiles((File)this.getWorkingDirectory(), (String[])extensions, (boolean)true)).isNotEmpty();
        Assertions.assertThat((Iterable)FileUtils.listFiles((File)subdirPath.toFile(), (String[])extensions, (boolean)false)).isNotEmpty();
    }

    @Test
    public void withFiles_savedToLocatorSpecifiedAbsoluteDir() throws Exception {
        String[] extensions = new String[]{"zip"};
        Path workingDirPath = this.getWorkingDirectory().toPath();
        Path absoluteDirPath = workingDirPath.resolve("some").resolve("test").resolve("directory").toAbsolutePath();
        this.gfshConnector.executeCommand("export logs --dir=" + absoluteDirPath.toString());
        Assertions.assertThat((Iterable)FileUtils.listFiles((File)this.getWorkingDirectory(), (String[])extensions, (boolean)false)).isEmpty();
        Assertions.assertThat((Iterable)FileUtils.listFiles((File)this.getWorkingDirectory(), (String[])extensions, (boolean)true)).isNotEmpty();
        Assertions.assertThat((Iterable)FileUtils.listFiles((File)absoluteDirPath.toFile(), (String[])extensions, (boolean)false)).isNotEmpty();
    }

    @Test
    public void startAndEndDateCanIncludeLogs() throws Exception {
        ZonedDateTime now = LocalDateTime.now().atZone(ZoneId.systemDefault());
        ZonedDateTime yesterday = now.minusDays(1L);
        ZonedDateTime tomorrow = now.plusDays(1L);
        DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy/MM/dd");
        CommandStringBuilder commandStringBuilder = new CommandStringBuilder("export logs");
        commandStringBuilder.addOption("start-time", dateTimeFormatter.format(yesterday));
        commandStringBuilder.addOption("end-time", dateTimeFormatter.format(tomorrow));
        commandStringBuilder.addOption("log-level", "debug");
        this.gfshConnector.executeAndAssertThat(commandStringBuilder.toString()).statusIsSuccess();
        Set<String> acceptedLogLevels = Stream.of("info", "error", "debug").collect(Collectors.toSet());
        this.verifyZipFileContents(acceptedLogLevels);
    }

    @Test
    public void testExportWithStartAndEndDateTimeFiltering() throws Exception {
        ZonedDateTime cutoffTime = LocalDateTime.now().atZone(ZoneId.systemDefault());
        Thread.sleep(1L);
        String messageAfterCutoffTime = "[this message should not show up since it is after cutoffTime]";
        LogLine logLineAfterCutoffTime = new LogLine(messageAfterCutoffTime, "info", true);
        this.server1.invoke(() -> {
            Logger logger = LogService.getLogger();
            logLineAfterCutoffTime.writeLog(logger);
        });
        DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy/MM/dd/HH/mm/ss/SSS/z");
        String cutoffTimeString = dateTimeFormatter.format(cutoffTime);
        CommandStringBuilder commandStringBuilder = new CommandStringBuilder("export logs");
        commandStringBuilder.addOption("start-time", dateTimeFormatter.format(cutoffTime.minusDays(1L)));
        commandStringBuilder.addOption("end-time", cutoffTimeString);
        commandStringBuilder.addOption("log-level", "debug");
        this.gfshConnector.executeAndAssertThat(commandStringBuilder.toString()).statusIsSuccess();
        this.expectedMessages.get(this.server1).add(logLineAfterCutoffTime);
        Set<String> acceptedLogLevels = Stream.of("info", "error", "debug").collect(Collectors.toSet());
        this.verifyZipFileContents(acceptedLogLevels);
    }

    @Test
    public void testExportWithThresholdLogLevelFilter() throws Exception {
        this.gfshConnector.executeAndAssertThat("export logs --log-level=info --only-log-level=false").statusIsSuccess();
        Set<String> acceptedLogLevels = Stream.of("info", "error").collect(Collectors.toSet());
        this.verifyZipFileContents(acceptedLogLevels);
    }

    @Test
    public void testExportWithExactLogLevelFilter() throws Exception {
        this.gfshConnector.executeAndAssertThat("export logs --log-level=info --only-log-level=true").statusIsSuccess();
        Set<String> acceptedLogLevels = Stream.of("info").collect(Collectors.toSet());
        this.verifyZipFileContents(acceptedLogLevels);
    }

    @Test
    public void testExportWithNoOptionsGiven() throws Exception {
        this.gfshConnector.executeAndAssertThat("export logs").statusIsSuccess();
        Set<String> acceptedLogLevels = Stream.of("info", "error", "debug").collect(Collectors.toSet());
        this.verifyZipFileContents(acceptedLogLevels);
    }

    @Test
    public void testExportWithNoFilters() throws Exception {
        this.gfshConnector.executeAndAssertThat("export logs --log-level=all").statusIsSuccess();
        Set<String> acceptedLogLevels = Stream.of("info", "error", "debug").collect(Collectors.toSet());
        this.verifyZipFileContents(acceptedLogLevels);
        this.server1.invoke(ExportLogsDistributedTestBase::verifyExportLogsRegionWasDestroyed);
        this.server2.invoke(ExportLogsDistributedTestBase::verifyExportLogsRegionWasDestroyed);
        this.locator.invoke(ExportLogsDistributedTestBase::verifyExportLogsRegionWasDestroyed);
    }

    @Test
    public void exportLogsRegionIsCleanedUpProperly() throws IOException, ClassNotFoundException {
        this.locator.invoke(() -> {
            GemFireCacheImpl cache = GemFireCacheImpl.getInstance();
            ExportLogsFunction.createOrGetExistingExportLogsRegion((boolean)true, (InternalCache)cache);
            Assertions.assertThat((Map)cache.getRegion("__exportLogsRegion")).isNotNull();
        });
        this.server1.invoke(() -> {
            GemFireCacheImpl cache = GemFireCacheImpl.getInstance();
            ExportLogsFunction.createOrGetExistingExportLogsRegion((boolean)false, (InternalCache)cache);
            Assertions.assertThat((Map)cache.getRegion("__exportLogsRegion")).isNotNull();
        });
        this.locator.invoke(() -> {
            GemFireCacheImpl cache = GemFireCacheImpl.getInstance();
            ExportLogsFunction.destroyExportLogsRegion((InternalCache)cache);
            Assertions.assertThat((Map)cache.getRegion("__exportLogsRegion")).isNull();
        });
        this.server1.invoke(() -> {
            GemFireCacheImpl cache = GemFireCacheImpl.getInstance();
            Assertions.assertThat((Map)cache.getRegion("__exportLogsRegion")).isNull();
        });
    }

    private void verifyZipFileContents(Set<String> acceptedLogLevels) throws Exception {
        File unzippedLogFileDir = this.unzipExportedLogs();
        Set dirsFromZipFile = Stream.of(unzippedLogFileDir.listFiles()).filter(File::isDirectory).collect(Collectors.toSet());
        Assertions.assertThat(dirsFromZipFile).hasSize(this.expectedMessages.keySet().size());
        Set expectedDirNames = this.expectedMessages.keySet().stream().map(Member::getName).collect(Collectors.toSet());
        Set actualDirNames = dirsFromZipFile.stream().map(File::getName).collect(Collectors.toSet());
        Assertions.assertThat(actualDirNames).isEqualTo(expectedDirNames);
        System.out.println("Unzipped artifacts:");
        for (File dir : dirsFromZipFile) {
            this.verifyLogFileContents(acceptedLogLevels, dir);
        }
    }

    private void verifyLogFileContents(Set<String> acceptedLogLevels, File dirForMember) throws IOException {
        String memberName = dirForMember.getName();
        MemberVM member = (MemberVM)this.expectedMessages.keySet().stream().filter(aMember -> aMember.getName().equals(memberName)).findFirst().get();
        Assertions.assertThat((Object)member).isNotNull();
        Set fileNamesInDir = Stream.of(dirForMember.listFiles()).map(File::getName).collect(Collectors.toSet());
        System.out.println(dirForMember.getCanonicalPath() + " : " + fileNamesInDir);
        File logFileForMember = new File(dirForMember, memberName + ".log");
        Assertions.assertThat((File)logFileForMember).exists();
        String logFileContents = FileUtils.readLines((File)logFileForMember, (Charset)Charset.defaultCharset()).stream().collect(Collectors.joining("\n"));
        for (LogLine logLine : this.expectedMessages.get(member)) {
            boolean shouldExpectLogLine;
            boolean bl = shouldExpectLogLine = acceptedLogLevels.contains(logLine.level) && !logLine.shouldBeIgnoredDueToTimestamp;
            if (shouldExpectLogLine) {
                Assertions.assertThat((String)logFileContents).contains(new CharSequence[]{logLine.getMessage()});
                continue;
            }
            Assertions.assertThat((String)logFileContents).doesNotContain(new CharSequence[]{logLine.getMessage()});
        }
    }

    private File unzipExportedLogs() throws Exception {
        File locatorWorkingDir = this.getWorkingDirectory();
        List filesInDir = Stream.of(this.getWorkingDirectory().listFiles()).collect(Collectors.toList());
        Assertions.assertThat(filesInDir).isNotEmpty();
        List zipFilesInDir = Stream.of(this.getWorkingDirectory().listFiles()).filter(f -> f.getName().endsWith(".zip")).collect(Collectors.toList());
        ((ListAssert)Assertions.assertThat(zipFilesInDir).describedAs(filesInDir.stream().map(File::getAbsolutePath).collect(Collectors.joining(",")), new Object[0])).hasSize(1);
        File unzippedLogFileDir = this.temporaryFolder.newFolder("unzippedLogs");
        ZipUtils.unzip((String)((File)zipFilesInDir.get(0)).getCanonicalPath(), (String)unzippedLogFileDir.getCanonicalPath());
        return unzippedLogFileDir;
    }

    private List<LogLine> listOfLogLines(Member member, String ... levels) {
        return Stream.of(levels).map(level -> new LogLine(member, (String)level)).collect(Collectors.toList());
    }

    private static void verifyExportLogsRegionWasDestroyed() {
        GemFireCacheImpl cache = GemFireCacheImpl.getInstance();
        Assertions.assertThat((Map)cache.getRegion("__exportLogsRegion")).isNull();
    }

    public static class LogLine
    implements Serializable {
        String level;
        String message;
        boolean shouldBeIgnoredDueToTimestamp;

        LogLine(String message, String level, boolean shouldBeIgnoredDueToTimestamp) {
            this.message = message;
            this.level = level;
            this.shouldBeIgnoredDueToTimestamp = shouldBeIgnoredDueToTimestamp;
        }

        LogLine(Member member, String level) {
            this.level = level;
            this.message = this.buildMessage(member.getName());
        }

        public String getMessage() {
            return this.message;
        }

        private String buildMessage(String memberName) {
            StringBuilder stringBuilder = new StringBuilder();
            if (Objects.equals(this.level, "error")) {
                stringBuilder.append("[IGNORE]-");
            }
            stringBuilder.append(this.level).append("-");
            return stringBuilder.append(memberName).toString();
        }

        void writeLog(Logger logger) {
            switch (this.level) {
                case "info": {
                    logger.info(this.getMessage());
                    break;
                }
                case "error": {
                    logger.error(this.getMessage());
                    break;
                }
                case "debug": {
                    logger.debug(this.getMessage());
                }
            }
        }
    }
}

