/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.expressions;

import java.io.Serializable;
import java.math.BigDecimal;
import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.Period;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.flink.annotation.PublicEvolving;
import org.apache.flink.table.api.TableException;
import org.apache.flink.table.api.ValidationException;
import org.apache.flink.table.expressions.Expression;
import org.apache.flink.table.expressions.ExpressionVisitor;
import org.apache.flink.table.expressions.ResolvedExpression;
import org.apache.flink.table.expressions.SqlFactory;
import org.apache.flink.table.types.DataType;
import org.apache.flink.table.types.logical.DecimalType;
import org.apache.flink.table.types.logical.LogicalType;
import org.apache.flink.table.types.logical.LogicalTypeRoot;
import org.apache.flink.table.types.utils.ValueDataTypeConverter;
import org.apache.flink.table.utils.EncodingUtils;
import org.apache.flink.types.ColumnList;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.StringUtils;

@PublicEvolving
public final class ValueLiteralExpression
implements ResolvedExpression {
    @Nullable
    private final Object value;
    private final DataType dataType;

    public ValueLiteralExpression(@Nonnull Object value) {
        this(value, ValueLiteralExpression.deriveDataTypeFromValue(value));
    }

    public ValueLiteralExpression(@Nullable Object value, DataType dataType) {
        ValueLiteralExpression.validateValueDataType(value, Preconditions.checkNotNull(dataType, "Data type must not be null."));
        this.value = value;
        this.dataType = dataType;
    }

    public boolean isNull() {
        return this.value == null;
    }

    public <T> Optional<T> getValueAs(Class<T> clazz) {
        Preconditions.checkArgument(!clazz.isPrimitive());
        if (this.value == null) {
            return Optional.empty();
        }
        Serializable convertedValue = null;
        if (clazz.isInstance(this.value)) {
            convertedValue = (Serializable)clazz.cast(this.value);
        } else {
            Class<?> valueClass = this.value.getClass();
            if (clazz == Period.class) {
                convertedValue = this.convertToPeriod(this.value, valueClass);
            } else if (clazz == Duration.class) {
                convertedValue = this.convertToDuration(this.value, valueClass);
            } else if (clazz == LocalDate.class) {
                convertedValue = this.convertToLocalDate(this.value, valueClass);
            } else if (clazz == LocalTime.class) {
                convertedValue = this.convertToLocalTime(this.value, valueClass);
            } else if (clazz == LocalDateTime.class) {
                convertedValue = this.convertToLocalDateTime(this.value, valueClass);
            } else if (clazz == OffsetDateTime.class) {
                convertedValue = this.convertToOffsetDateTime(this.value, valueClass);
            } else if (clazz == Instant.class) {
                convertedValue = this.convertToInstant(this.value, valueClass);
            } else if (clazz == BigDecimal.class) {
                convertedValue = this.convertToBigDecimal(this.value);
            }
        }
        return Optional.ofNullable(convertedValue);
    }

    @Nullable
    private LocalDate convertToLocalDate(Object value, Class<?> valueClass) {
        if (valueClass == Date.class) {
            return ((Date)value).toLocalDate();
        }
        if (valueClass == Integer.class) {
            return LocalDate.ofEpochDay(((Integer)value).intValue());
        }
        return null;
    }

    @Nullable
    private LocalTime convertToLocalTime(Object value, Class<?> valueClass) {
        if (valueClass == Time.class) {
            return ((Time)value).toLocalTime();
        }
        if (valueClass == Integer.class) {
            return LocalTime.ofNanoOfDay((long)((Integer)value).intValue() * 1000000L);
        }
        if (valueClass == Long.class) {
            return LocalTime.ofNanoOfDay((Long)value);
        }
        return null;
    }

    @Nullable
    private LocalDateTime convertToLocalDateTime(Object value, Class<?> valueClass) {
        if (valueClass == Timestamp.class) {
            return ((Timestamp)value).toLocalDateTime();
        }
        return null;
    }

    @Nullable
    private OffsetDateTime convertToOffsetDateTime(Object value, Class<?> valueClass) {
        if (valueClass == ZonedDateTime.class) {
            return ((ZonedDateTime)value).toOffsetDateTime();
        }
        return null;
    }

    @Nullable
    private Instant convertToInstant(Object value, Class<?> valueClass) {
        if (valueClass == Integer.class) {
            return Instant.ofEpochSecond(((Integer)value).intValue());
        }
        if (valueClass == Long.class) {
            return Instant.ofEpochMilli((Long)value);
        }
        return null;
    }

    @Nullable
    private Duration convertToDuration(Object value, Class<?> valueClass) {
        if (valueClass == Long.class) {
            Long longValue = (Long)value;
            return Duration.ofMillis(longValue);
        }
        return null;
    }

    @Nullable
    private Period convertToPeriod(Object value, Class<?> valueClass) {
        if (valueClass == Integer.class) {
            Integer integer = (Integer)value;
            return Period.ofMonths(integer);
        }
        return null;
    }

    @Nullable
    private BigDecimal convertToBigDecimal(Object value) {
        if (Number.class.isAssignableFrom(value.getClass())) {
            return new BigDecimal(String.valueOf(value));
        }
        return null;
    }

    @Override
    public DataType getOutputDataType() {
        return this.dataType;
    }

    @Override
    public List<ResolvedExpression> getResolvedChildren() {
        return Collections.emptyList();
    }

    @Override
    public String asSummaryString() {
        return ValueLiteralExpression.stringifyValue(this.value);
    }

    @Override
    public String asSerializableString(SqlFactory sqlFactory) {
        if (this.value == null && !this.dataType.getLogicalType().is(LogicalTypeRoot.NULL)) {
            return String.format("CAST(NULL AS %s)", this.dataType.getLogicalType().copy(true).asSerializableString());
        }
        LogicalType logicalType = this.dataType.getLogicalType();
        switch (logicalType.getTypeRoot()) {
            case TINYINT: {
                return String.format("CAST(%s AS TINYINT)", this.value);
            }
            case SMALLINT: {
                return String.format("CAST(%s AS SMALLINT)", this.value);
            }
            case BIGINT: {
                return String.format("CAST(%s AS BIGINT)", this.value);
            }
            case FLOAT: {
                return String.format("CAST(%s AS FLOAT)", this.value);
            }
            case DOUBLE: {
                return String.format("CAST(%s AS DOUBLE)", this.value);
            }
            case DECIMAL: {
                DecimalType decimalType = (DecimalType)logicalType;
                BigDecimal decimal = this.getValueAs(BigDecimal.class).get();
                String decimalString = decimal.toString();
                if (decimal.precision() == decimalType.getPrecision() && decimal.scale() == decimalType.getScale()) {
                    return decimalString;
                }
                return String.format("CAST(%s AS DECIMAL(%d, %d))", decimalString, decimalType.getPrecision(), decimalType.getScale());
            }
            case CHAR: 
            case VARCHAR: {
                return "'" + ((String)this.value).replace("'", "''") + "'";
            }
            case INTEGER: {
                return String.valueOf(this.value);
            }
            case BOOLEAN: 
            case NULL: {
                return String.valueOf(this.value).toUpperCase(Locale.ROOT);
            }
            case BINARY: 
            case VARBINARY: {
                return String.format("X'%s'", StringUtils.byteToHexString((byte[])this.value));
            }
            case DATE: {
                return String.format("DATE '%s'", this.getValueAs(LocalDate.class).get());
            }
            case TIME_WITHOUT_TIME_ZONE: {
                LocalTime localTime = this.getValueAs(LocalTime.class).get();
                return String.format("TIME '%s'", localTime.format(DateTimeFormatter.ISO_LOCAL_TIME));
            }
            case TIMESTAMP_WITHOUT_TIME_ZONE: {
                LocalDateTime localDateTime = this.getValueAs(LocalDateTime.class).get();
                return String.format("TIMESTAMP '%s %s'", localDateTime.toLocalDate(), localDateTime.toLocalTime().format(DateTimeFormatter.ISO_LOCAL_TIME));
            }
            case TIMESTAMP_WITH_LOCAL_TIME_ZONE: {
                Instant instant = this.getValueAs(Instant.class).get();
                if (instant.getNano() % 1000000 != 0) {
                    throw new TableException("Maximum precision for TIMESTAMP_WITH_LOCAL_TIME_ZONE literals is '3'");
                }
                return String.format("TO_TIMESTAMP_LTZ(%d, %d)", instant.toEpochMilli(), 3);
            }
            case INTERVAL_YEAR_MONTH: {
                Period period = this.getValueAs(Period.class).get().normalized();
                return String.format("INTERVAL '%d-%d' YEAR TO MONTH", period.getYears(), period.getMonths());
            }
            case INTERVAL_DAY_TIME: {
                Duration duration = this.getValueAs(Duration.class).get();
                return String.format("INTERVAL '%d %02d:%02d:%02d.%d' DAY TO SECOND(3)", duration.toDays(), duration.toHours() % 24L, duration.toMinutes() % 60L, duration.getSeconds() % 60L, duration.getNano() / 1000000);
            }
            case DESCRIPTOR: {
                ColumnList columnList = this.getValueAs(ColumnList.class).get();
                if (!columnList.getDataTypes().isEmpty()) {
                    throw new TableException("Data types in DESCRIPTOR are not supported yet.");
                }
                return String.format("DESCRIPTOR(%s)", columnList.getNames().stream().map(EncodingUtils::escapeBackticks).map(c -> String.format("`%s`", c)).collect(Collectors.joining()));
            }
            case ARRAY: 
            case MULTISET: 
            case MAP: 
            case ROW: {
                throw new TableException("Constructed type literals are not SQL serializable. Please use respective constructor functions");
            }
        }
        throw new TableException("Literals with " + String.valueOf(this.dataType.getLogicalType()) + " are not SQL serializable.");
    }

    @Override
    public List<Expression> getChildren() {
        return Collections.emptyList();
    }

    @Override
    public <R> R accept(ExpressionVisitor<R> visitor) {
        return visitor.visit(this);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        ValueLiteralExpression that = (ValueLiteralExpression)o;
        return Objects.deepEquals(this.value, that.value) && this.dataType.equals(that.dataType);
    }

    public int hashCode() {
        return Objects.hash(this.value, this.dataType);
    }

    public String toString() {
        return this.asSummaryString();
    }

    private static DataType deriveDataTypeFromValue(Object value) {
        return ValueDataTypeConverter.extractDataType(value).orElseThrow(() -> new ValidationException("Cannot derive a data type for value '" + String.valueOf(value) + "'. The data type must be specified explicitly."));
    }

    private static void validateValueDataType(Object value, DataType dataType) {
        LogicalType logicalType = dataType.getLogicalType();
        if (value == null) {
            if (!logicalType.isNullable()) {
                throw new ValidationException(String.format("Data type '%s' does not support null values.", dataType));
            }
            return;
        }
        if (logicalType.isNullable()) {
            throw new ValidationException("Literals that have a non-null value must not have a nullable data type.");
        }
        Class<?> candidate = value.getClass();
        if (!dataType.getConversionClass().isAssignableFrom(candidate)) {
            throw new ValidationException(String.format("Data type '%s' with conversion class '%s' does not support a value literal of class '%s'.", dataType, dataType.getConversionClass().getName(), value.getClass().getName()));
        }
        if (!logicalType.supportsInputConversion(candidate)) {
            throw new ValidationException(String.format("Data type '%s' does not support a conversion from class '%s'.", dataType, candidate.getName()));
        }
    }

    private static String stringifyValue(Object value) {
        if (value instanceof String[]) {
            String[] array = (String[])value;
            return Stream.of(array).map(ValueLiteralExpression::stringifyValue).collect(Collectors.joining(", ", "[", "]"));
        }
        if (value instanceof Object[]) {
            Object[] array = (Object[])value;
            return Stream.of(array).map(ValueLiteralExpression::stringifyValue).collect(Collectors.joining(", ", "[", "]"));
        }
        if (value instanceof String) {
            return "'" + ((String)value).replace("'", "''") + "'";
        }
        return StringUtils.arrayAwareToString(value);
    }
}

