/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.core.security;

import java.security.Permission;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.configuration.ClusterOptions;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.IllegalConfigurationException;
import org.apache.flink.core.security.UserSystemExitException;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FlinkSecurityManager
extends SecurityManager {
    static final Logger LOG = LoggerFactory.getLogger(FlinkSecurityManager.class);
    private static FlinkSecurityManager flinkSecurityManager;
    private final SecurityManager originalSecurityManager;
    private final ThreadLocal<Boolean> monitorUserSystemExit = new InheritableThreadLocal<Boolean>();
    private final ClusterOptions.UserSystemExitMode userSystemExitMode;
    private final boolean haltOnSystemExit;

    @VisibleForTesting
    FlinkSecurityManager(ClusterOptions.UserSystemExitMode userSystemExitMode, boolean haltOnSystemExit) {
        this(userSystemExitMode, haltOnSystemExit, System.getSecurityManager());
    }

    @VisibleForTesting
    FlinkSecurityManager(ClusterOptions.UserSystemExitMode userSystemExitMode, boolean haltOnSystemExit, SecurityManager originalSecurityManager) {
        this.userSystemExitMode = Preconditions.checkNotNull(userSystemExitMode);
        this.haltOnSystemExit = haltOnSystemExit;
        this.originalSecurityManager = originalSecurityManager;
    }

    @VisibleForTesting
    static FlinkSecurityManager fromConfiguration(Configuration configuration) {
        ClusterOptions.UserSystemExitMode userSystemExitMode = configuration.get(ClusterOptions.INTERCEPT_USER_SYSTEM_EXIT);
        boolean haltOnSystemExit = configuration.get(ClusterOptions.HALT_ON_FATAL_ERROR);
        if (userSystemExitMode == ClusterOptions.UserSystemExitMode.DISABLED && !haltOnSystemExit) {
            return null;
        }
        LOG.info("FlinkSecurityManager is created with {} user system exit mode and {} exit", (Object)userSystemExitMode, (Object)(haltOnSystemExit ? "forceful" : "graceful"));
        return new FlinkSecurityManager(userSystemExitMode, haltOnSystemExit);
    }

    public static void setFromConfiguration(Configuration configuration) {
        FlinkSecurityManager flinkSecurityManager = FlinkSecurityManager.fromConfiguration(configuration);
        if (flinkSecurityManager != null) {
            try {
                System.setSecurityManager(flinkSecurityManager);
            }
            catch (Exception e) {
                throw new IllegalConfigurationException(String.format("Could not register security manager due to no permission to set a SecurityManager. Either update your existing SecurityManager to allow the permission or do not use security manager features (e.g., '%s: %s', '%s: %s')", ClusterOptions.INTERCEPT_USER_SYSTEM_EXIT.key(), ClusterOptions.INTERCEPT_USER_SYSTEM_EXIT.defaultValue(), ClusterOptions.HALT_ON_FATAL_ERROR.key(), ClusterOptions.HALT_ON_FATAL_ERROR.defaultValue()), e);
            }
        }
        FlinkSecurityManager.flinkSecurityManager = flinkSecurityManager;
    }

    public static void monitorUserSystemExitForCurrentThread() {
        if (flinkSecurityManager != null) {
            flinkSecurityManager.monitorUserSystemExit();
        }
    }

    public static void unmonitorUserSystemExitForCurrentThread() {
        if (flinkSecurityManager != null) {
            flinkSecurityManager.unmonitorUserSystemExit();
        }
    }

    @Override
    public void checkPermission(Permission perm) {
        if (this.originalSecurityManager != null) {
            this.originalSecurityManager.checkPermission(perm);
        }
    }

    @Override
    public void checkPermission(Permission perm, Object context) {
        if (this.originalSecurityManager != null) {
            this.originalSecurityManager.checkPermission(perm, context);
        }
    }

    @Override
    public void checkExit(int status) {
        if (this.userSystemExitMonitored()) {
            switch (this.userSystemExitMode) {
                case DISABLED: {
                    break;
                }
                case LOG: {
                    LOG.warn("Exiting JVM with status {} is monitored: The system will exit due to this call.", (Object)status, (Object)new UserSystemExitException());
                    break;
                }
                case THROW: {
                    throw new UserSystemExitException();
                }
                default: {
                    LOG.warn("No valid check exit mode configured: {}", (Object)this.userSystemExitMode);
                }
            }
        }
        if (this.originalSecurityManager != null) {
            this.originalSecurityManager.checkExit(status);
        }
        if (this.haltOnSystemExit) {
            System.setSecurityManager(null);
            Runtime.getRuntime().halt(status);
        }
    }

    @VisibleForTesting
    void monitorUserSystemExit() {
        this.monitorUserSystemExit.set(true);
    }

    @VisibleForTesting
    void unmonitorUserSystemExit() {
        this.monitorUserSystemExit.set(false);
    }

    @VisibleForTesting
    boolean userSystemExitMonitored() {
        return Boolean.TRUE.equals(this.monitorUserSystemExit.get());
    }

    public static void forceProcessExit(int exitCode) {
        System.setSecurityManager(null);
        if (flinkSecurityManager != null && FlinkSecurityManager.flinkSecurityManager.haltOnSystemExit) {
            Runtime.getRuntime().halt(exitCode);
        } else {
            System.exit(exitCode);
        }
    }
}

