/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.net;

import java.io.File;
import java.io.InputStream;
import java.net.ServerSocket;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import javax.net.ssl.SSLServerSocket;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.IllegalConfigurationException;
import org.apache.flink.configuration.SecurityOptions;
import org.apache.flink.runtime.io.network.netty.SSLHandlerFactory;
import org.apache.flink.runtime.net.SSLUtils;
import org.apache.flink.shaded.netty4.io.netty.buffer.ByteBufAllocator;
import org.apache.flink.shaded.netty4.io.netty.buffer.UnpooledByteBufAllocator;
import org.apache.flink.shaded.netty4.io.netty.handler.ssl.ClientAuth;
import org.apache.flink.shaded.netty4.io.netty.handler.ssl.JdkSslContext;
import org.apache.flink.shaded.netty4.io.netty.handler.ssl.OpenSsl;
import org.apache.flink.shaded.netty4.io.netty.handler.ssl.SslHandler;
import org.apache.flink.shaded.netty4.io.netty.handler.ssl.SslProvider;
import org.apache.flink.util.Preconditions;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;

public class SSLUtilsTest {
    private static final String TRUST_STORE_PATH = ((URL)Preconditions.checkNotNull((Object)SSLUtilsTest.class.getResource("/local127.truststore"))).getFile();
    private static final String KEY_STORE_PATH = ((URL)Preconditions.checkNotNull((Object)SSLUtilsTest.class.getResource("/local127.keystore"))).getFile();
    private static final String TRUST_STORE_PASSWORD = "password";
    private static final String KEY_STORE_PASSWORD = "password";
    private static final String KEY_PASSWORD = "password";
    public static final List<String> AVAILABLE_SSL_PROVIDERS;

    private static List<String> parameters() {
        return AVAILABLE_SSL_PROVIDERS;
    }

    @Test
    void testSocketFactoriesWhenSslDisabled() {
        Configuration config = new Configuration();
        Assertions.assertThatThrownBy(() -> SSLUtils.createSSLServerSocketFactory((Configuration)config)).isInstanceOf(IllegalConfigurationException.class);
        Assertions.assertThatThrownBy(() -> SSLUtils.createSSLClientSocketFactory((Configuration)config)).isInstanceOf(IllegalConfigurationException.class);
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testRESTClientSSL(String sslProvider) throws Exception {
        Configuration clientConfig = this.createRestSslConfigWithTrustStore(sslProvider);
        SSLHandlerFactory ssl = SSLUtils.createRestClientSSLEngineFactory((Configuration)clientConfig);
        Assertions.assertThat((Object)ssl).isNotNull();
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testRESTClientSSLDisabled(String sslProvider) {
        Configuration clientConfig = this.createRestSslConfigWithTrustStore(sslProvider);
        clientConfig.set(SecurityOptions.SSL_REST_ENABLED, (Object)false);
        Assertions.assertThatThrownBy(() -> SSLUtils.createRestClientSSLEngineFactory((Configuration)clientConfig)).isInstanceOf(IllegalConfigurationException.class);
    }

    @Test
    void testRESTClientSSLMissingTrustStore() {
        Configuration config = new Configuration();
        config.set(SecurityOptions.SSL_REST_ENABLED, (Object)true);
        config.set(SecurityOptions.SSL_REST_TRUSTSTORE_PASSWORD, (Object)"some password");
        Assertions.assertThatThrownBy(() -> SSLUtils.createRestClientSSLEngineFactory((Configuration)config)).isInstanceOf(IllegalConfigurationException.class);
    }

    @Test
    void testRESTClientSSLMissingPassword() {
        Configuration config = new Configuration();
        config.set(SecurityOptions.SSL_REST_ENABLED, (Object)true);
        config.set(SecurityOptions.SSL_REST_TRUSTSTORE, (Object)TRUST_STORE_PATH);
        Assertions.assertThatThrownBy(() -> SSLUtils.createRestClientSSLEngineFactory((Configuration)config)).isInstanceOf(IllegalConfigurationException.class);
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testRESTClientSSLWrongPassword(String sslProvider) {
        Configuration clientConfig = this.createRestSslConfigWithTrustStore(sslProvider);
        clientConfig.set(SecurityOptions.SSL_REST_TRUSTSTORE_PASSWORD, (Object)"badpassword");
        Assertions.assertThatThrownBy(() -> SSLUtils.createRestClientSSLEngineFactory((Configuration)clientConfig)).isInstanceOf(Exception.class);
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testRESTClientSSLBadTruststoreType(String sslProvider) {
        Configuration clientConfig = this.createRestSslConfigWithTrustStore(sslProvider);
        clientConfig.set(SecurityOptions.SSL_REST_TRUSTSTORE_TYPE, (Object)"bad-truststore-type");
        Assertions.assertThatThrownBy(() -> SSLUtils.createRestClientSSLEngineFactory((Configuration)clientConfig)).isInstanceOf(KeyStoreException.class);
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testRESTSSLConfigCipherAlgorithms(String sslProvider) throws Exception {
        String testSSLAlgorithms = "test_algorithm1,test_algorithm2";
        Configuration config = this.createRestSslConfigWithTrustStore(sslProvider);
        config.set(SecurityOptions.SSL_REST_ENABLED, (Object)true);
        config.setString(SecurityOptions.SSL_ALGORITHMS.key(), testSSLAlgorithms);
        JdkSslContext nettySSLContext = (JdkSslContext)SSLUtils.createRestNettySSLContext((Configuration)config, (boolean)true, (ClientAuth)ClientAuth.NONE, (SslProvider)SslProvider.JDK);
        List cipherSuites = ((JdkSslContext)Preconditions.checkNotNull((Object)nettySSLContext)).cipherSuites();
        Assertions.assertThat((List)cipherSuites).hasSize(2);
        Assertions.assertThat((List)cipherSuites).containsExactlyInAnyOrder((Object[])testSSLAlgorithms.split(","));
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testRESTServerSSL(String sslProvider) throws Exception {
        Configuration serverConfig = this.createRestSslConfigWithKeyStore(sslProvider);
        SSLHandlerFactory ssl = SSLUtils.createRestServerSSLEngineFactory((Configuration)serverConfig);
        Assertions.assertThat((Object)ssl).isNotNull();
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testRESTServerSSLDisabled(String sslProvider) {
        Configuration serverConfig = this.createRestSslConfigWithKeyStore(sslProvider);
        serverConfig.set(SecurityOptions.SSL_REST_ENABLED, (Object)false);
        Assertions.assertThatThrownBy(() -> SSLUtils.createRestServerSSLEngineFactory((Configuration)serverConfig)).isInstanceOf(IllegalConfigurationException.class);
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testRESTServerSSLBadKeystorePassword(String sslProvider) {
        Configuration serverConfig = this.createRestSslConfigWithKeyStore(sslProvider);
        serverConfig.set(SecurityOptions.SSL_REST_KEYSTORE_PASSWORD, (Object)"badpassword");
        Assertions.assertThatThrownBy(() -> SSLUtils.createRestServerSSLEngineFactory((Configuration)serverConfig)).isInstanceOf(Exception.class);
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testRESTServerSSLBadKeyPassword(String sslProvider) {
        Configuration serverConfig = this.createRestSslConfigWithKeyStore(sslProvider);
        serverConfig.set(SecurityOptions.SSL_REST_KEY_PASSWORD, (Object)"badpassword");
        Assertions.assertThatThrownBy(() -> SSLUtils.createRestServerSSLEngineFactory((Configuration)serverConfig)).isInstanceOf(Exception.class);
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testRESTServerSSLBadKeystoreType(String sslProvider) {
        Configuration serverConfig = this.createRestSslConfigWithKeyStore(sslProvider);
        serverConfig.set(SecurityOptions.SSL_REST_KEYSTORE_TYPE, (Object)"bad-keystore-type");
        Assertions.assertThatThrownBy(() -> SSLUtils.createRestServerSSLEngineFactory((Configuration)serverConfig)).isInstanceOf(KeyStoreException.class);
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testInternalSSL(String sslProvider) throws Exception {
        Configuration config = SSLUtilsTest.createInternalSslConfigWithKeyAndTrustStores(sslProvider);
        Assertions.assertThat((Object)SSLUtils.createInternalServerSSLEngineFactory((Configuration)config)).isNotNull();
        Assertions.assertThat((Object)SSLUtils.createInternalClientSSLEngineFactory((Configuration)config)).isNotNull();
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testInternalSSLWithSSLPinning(String sslProvider) throws Exception {
        Configuration config = SSLUtilsTest.createInternalSslConfigWithKeyAndTrustStores(sslProvider);
        config.set(SecurityOptions.SSL_INTERNAL_CERT_FINGERPRINT, (Object)SSLUtilsTest.getCertificateFingerprint(config, "flink.test"));
        Assertions.assertThat((Object)SSLUtils.createInternalServerSSLEngineFactory((Configuration)config)).isNotNull();
        Assertions.assertThat((Object)SSLUtils.createInternalClientSSLEngineFactory((Configuration)config)).isNotNull();
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testInternalSSLDisables(String sslProvider) {
        Configuration config = SSLUtilsTest.createInternalSslConfigWithKeyAndTrustStores(sslProvider);
        config.set(SecurityOptions.SSL_INTERNAL_ENABLED, (Object)false);
        Assertions.assertThatThrownBy(() -> SSLUtils.createInternalServerSSLEngineFactory((Configuration)config)).isInstanceOf(Exception.class);
        Assertions.assertThatThrownBy(() -> SSLUtils.createInternalClientSSLEngineFactory((Configuration)config)).isInstanceOf(Exception.class);
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testInternalSSLKeyStoreOnly(String sslProvider) {
        Configuration config = this.createInternalSslConfigWithKeyStore(sslProvider);
        Assertions.assertThatThrownBy(() -> SSLUtils.createInternalServerSSLEngineFactory((Configuration)config)).isInstanceOf(Exception.class);
        Assertions.assertThatThrownBy(() -> SSLUtils.createInternalClientSSLEngineFactory((Configuration)config)).isInstanceOf(Exception.class);
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testInternalSSLTrustStoreOnly(String sslProvider) {
        Configuration config = this.createInternalSslConfigWithTrustStore(sslProvider);
        Assertions.assertThatThrownBy(() -> SSLUtils.createInternalServerSSLEngineFactory((Configuration)config)).isInstanceOf(Exception.class);
        Assertions.assertThatThrownBy(() -> SSLUtils.createInternalClientSSLEngineFactory((Configuration)config)).isInstanceOf(Exception.class);
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testInternalSSLWrongKeystorePassword(String sslProvider) {
        Configuration config = SSLUtilsTest.createInternalSslConfigWithKeyAndTrustStores(sslProvider);
        config.set(SecurityOptions.SSL_INTERNAL_KEYSTORE_PASSWORD, (Object)"badpw");
        Assertions.assertThatThrownBy(() -> SSLUtils.createInternalServerSSLEngineFactory((Configuration)config)).isInstanceOf(Exception.class);
        Assertions.assertThatThrownBy(() -> SSLUtils.createInternalClientSSLEngineFactory((Configuration)config)).isInstanceOf(Exception.class);
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testInternalSSLWrongTruststorePassword(String sslProvider) {
        Configuration config = SSLUtilsTest.createInternalSslConfigWithKeyAndTrustStores(sslProvider);
        config.set(SecurityOptions.SSL_INTERNAL_TRUSTSTORE_PASSWORD, (Object)"badpw");
        Assertions.assertThatThrownBy(() -> SSLUtils.createInternalServerSSLEngineFactory((Configuration)config)).isInstanceOf(Exception.class);
        Assertions.assertThatThrownBy(() -> SSLUtils.createInternalClientSSLEngineFactory((Configuration)config)).isInstanceOf(Exception.class);
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testInternalSSLWrongTruststoreType(String sslProvider) {
        Configuration config = SSLUtilsTest.createInternalSslConfigWithKeyAndTrustStores(sslProvider);
        config.set(SecurityOptions.SSL_INTERNAL_TRUSTSTORE_TYPE, (Object)"bad-truststore-type");
        Assertions.assertThatThrownBy(() -> SSLUtils.createInternalServerSSLEngineFactory((Configuration)config)).isInstanceOf(KeyStoreException.class);
        Assertions.assertThatThrownBy(() -> SSLUtils.createInternalClientSSLEngineFactory((Configuration)config)).isInstanceOf(KeyStoreException.class);
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testInternalSSLWrongKeyPassword(String sslProvider) {
        Configuration config = SSLUtilsTest.createInternalSslConfigWithKeyAndTrustStores(sslProvider);
        config.set(SecurityOptions.SSL_INTERNAL_KEY_PASSWORD, (Object)"badpw");
        Assertions.assertThatThrownBy(() -> SSLUtils.createInternalServerSSLEngineFactory((Configuration)config)).isInstanceOf(Exception.class);
        Assertions.assertThatThrownBy(() -> SSLUtils.createInternalClientSSLEngineFactory((Configuration)config)).isInstanceOf(Exception.class);
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testInternalSSLWrongKeystoreType(String sslProvider) {
        Configuration config = SSLUtilsTest.createInternalSslConfigWithKeyAndTrustStores(sslProvider);
        config.set(SecurityOptions.SSL_INTERNAL_KEYSTORE_TYPE, (Object)"bad-keystore-type");
        Assertions.assertThatThrownBy(() -> SSLUtils.createInternalServerSSLEngineFactory((Configuration)config)).isInstanceOf(KeyStoreException.class);
        Assertions.assertThatThrownBy(() -> SSLUtils.createInternalClientSSLEngineFactory((Configuration)config)).isInstanceOf(KeyStoreException.class);
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testSetSSLVersionAndCipherSuitesForSSLServerSocket(String sslProvider) throws Exception {
        Configuration serverConfig = SSLUtilsTest.createInternalSslConfigWithKeyAndTrustStores(sslProvider);
        serverConfig.set(SecurityOptions.SSL_PROTOCOL, (Object)"TLSv1.1");
        serverConfig.set(SecurityOptions.SSL_ALGORITHMS, (Object)"TLS_RSA_WITH_AES_128_CBC_SHA,TLS_RSA_WITH_AES_128_CBC_SHA256");
        try (ServerSocket socket = SSLUtils.createSSLServerSocketFactory((Configuration)serverConfig).createServerSocket(0);){
            Assertions.assertThat((Object)socket).isInstanceOf(SSLServerSocket.class);
            SSLServerSocket sslSocket = (SSLServerSocket)socket;
            Object[] protocols = sslSocket.getEnabledProtocols();
            Object[] algorithms = sslSocket.getEnabledCipherSuites();
            Assertions.assertThat((Object[])protocols).hasSize(1);
            Assertions.assertThat((String)protocols[0]).isEqualTo("TLSv1.1");
            Assertions.assertThat((Object[])algorithms).hasSize(2);
            Assertions.assertThat((Object[])algorithms).contains((Object[])new String[]{"TLS_RSA_WITH_AES_128_CBC_SHA", "TLS_RSA_WITH_AES_128_CBC_SHA256"});
        }
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testCreateSSLEngineFactory(String sslProvider) throws Exception {
        Object[] expectedSslProtocols;
        Object[] sslAlgorithms;
        Configuration serverConfig = SSLUtilsTest.createInternalSslConfigWithKeyAndTrustStores(sslProvider);
        if (sslProvider.equalsIgnoreCase("OPENSSL")) {
            sslAlgorithms = new String[]{"TLS_RSA_WITH_AES_128_GCM_SHA256", "TLS_RSA_WITH_AES_256_GCM_SHA384"};
            expectedSslProtocols = new String[]{"SSLv2Hello", "TLSv1"};
        } else {
            sslAlgorithms = new String[]{"TLS_DHE_RSA_WITH_AES_128_CBC_SHA", "TLS_DHE_RSA_WITH_AES_128_CBC_SHA256"};
            expectedSslProtocols = new String[]{"TLSv1"};
        }
        serverConfig.set(SecurityOptions.SSL_PROTOCOL, (Object)"TLSv1");
        serverConfig.set(SecurityOptions.SSL_ALGORITHMS, (Object)String.join((CharSequence)",", (CharSequence[])sslAlgorithms));
        SSLHandlerFactory serverSSLHandlerFactory = SSLUtils.createInternalServerSSLEngineFactory((Configuration)serverConfig);
        SslHandler sslHandler = serverSSLHandlerFactory.createNettySSLHandler((ByteBufAllocator)UnpooledByteBufAllocator.DEFAULT);
        Assertions.assertThat((Object[])sslHandler.engine().getEnabledProtocols()).hasSameSizeAs((Object)expectedSslProtocols);
        Assertions.assertThat((Object[])sslHandler.engine().getEnabledProtocols()).contains(expectedSslProtocols);
        Assertions.assertThat((Object[])sslHandler.engine().getEnabledCipherSuites()).hasSameSizeAs((Object)sslAlgorithms);
        Assertions.assertThat((Object[])sslHandler.engine().getEnabledCipherSuites()).contains(sslAlgorithms);
    }

    @ParameterizedTest
    @MethodSource(value={"parameters"})
    void testInvalidFingerprintParsing(String sslProvider) throws Exception {
        Configuration config = SSLUtilsTest.createInternalSslConfigWithKeyAndTrustStores(sslProvider);
        String fingerprint = SSLUtilsTest.getCertificateFingerprint(config, "flink.test");
        config.set(SecurityOptions.SSL_INTERNAL_CERT_FINGERPRINT, (Object)fingerprint.substring(0, fingerprint.length() - 3));
        Assertions.assertThatThrownBy(() -> SSLUtils.createInternalServerSSLEngineFactory((Configuration)config)).isInstanceOf(IllegalArgumentException.class);
    }

    private Configuration createRestSslConfigWithKeyStore(String sslProvider) {
        Configuration config = new Configuration();
        config.set(SecurityOptions.SSL_REST_ENABLED, (Object)true);
        SSLUtilsTest.addSslProviderConfig(config, sslProvider);
        SSLUtilsTest.addRestKeyStoreConfig(config);
        return config;
    }

    private Configuration createRestSslConfigWithTrustStore(String sslProvider) {
        Configuration config = new Configuration();
        config.set(SecurityOptions.SSL_REST_ENABLED, (Object)true);
        SSLUtilsTest.addSslProviderConfig(config, sslProvider);
        SSLUtilsTest.addRestTrustStoreConfig(config);
        return config;
    }

    public static Configuration createRestSslConfigWithKeyAndTrustStores(String sslProvider) {
        Configuration config = new Configuration();
        config.set(SecurityOptions.SSL_REST_ENABLED, (Object)true);
        SSLUtilsTest.addSslProviderConfig(config, sslProvider);
        SSLUtilsTest.addRestKeyStoreConfig(config);
        SSLUtilsTest.addRestTrustStoreConfig(config);
        return config;
    }

    private Configuration createInternalSslConfigWithKeyStore(String sslProvider) {
        Configuration config = new Configuration();
        config.set(SecurityOptions.SSL_INTERNAL_ENABLED, (Object)true);
        SSLUtilsTest.addSslProviderConfig(config, sslProvider);
        SSLUtilsTest.addInternalKeyStoreConfig(config);
        return config;
    }

    private Configuration createInternalSslConfigWithTrustStore(String sslProvider) {
        Configuration config = new Configuration();
        config.set(SecurityOptions.SSL_INTERNAL_ENABLED, (Object)true);
        SSLUtilsTest.addSslProviderConfig(config, sslProvider);
        SSLUtilsTest.addInternalTrustStoreConfig(config);
        return config;
    }

    public static Configuration createInternalSslConfigWithKeyAndTrustStores(String sslProvider) {
        Configuration config = new Configuration();
        config.set(SecurityOptions.SSL_INTERNAL_ENABLED, (Object)true);
        SSLUtilsTest.addSslProviderConfig(config, sslProvider);
        SSLUtilsTest.addInternalKeyStoreConfig(config);
        SSLUtilsTest.addInternalTrustStoreConfig(config);
        return config;
    }

    public static String getCertificateFingerprint(Configuration config, String certificateAlias) throws Exception {
        KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
        try (InputStream keyStoreFile = Files.newInputStream(new File((String)config.get(SecurityOptions.SSL_INTERNAL_KEYSTORE)).toPath(), new OpenOption[0]);){
            keyStore.load(keyStoreFile, ((String)config.get(SecurityOptions.SSL_INTERNAL_KEYSTORE_PASSWORD)).toCharArray());
        }
        return SSLUtilsTest.getSha1Fingerprint(keyStore.getCertificate(certificateAlias));
    }

    public static String getRestCertificateFingerprint(Configuration config, String certificateAlias) throws Exception {
        KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
        try (InputStream keyStoreFile = Files.newInputStream(new File((String)config.get(SecurityOptions.SSL_REST_KEYSTORE)).toPath(), new OpenOption[0]);){
            keyStore.load(keyStoreFile, ((String)config.get(SecurityOptions.SSL_REST_KEYSTORE_PASSWORD)).toCharArray());
        }
        return SSLUtilsTest.getSha1Fingerprint(keyStore.getCertificate(certificateAlias));
    }

    private static void addSslProviderConfig(Configuration config, String sslProvider) {
        if (sslProvider.equalsIgnoreCase("OPENSSL")) {
            OpenSsl.ensureAvailability();
            config.set(SecurityOptions.SSL_ALGORITHMS, (Object)"TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256,TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384");
        }
        config.set(SecurityOptions.SSL_PROVIDER, (Object)sslProvider);
    }

    private static void addRestKeyStoreConfig(Configuration config) {
        config.set(SecurityOptions.SSL_REST_KEYSTORE, (Object)KEY_STORE_PATH);
        config.set(SecurityOptions.SSL_REST_KEYSTORE_PASSWORD, (Object)"password");
        config.set(SecurityOptions.SSL_REST_KEY_PASSWORD, (Object)"password");
    }

    private static void addRestTrustStoreConfig(Configuration config) {
        config.set(SecurityOptions.SSL_REST_TRUSTSTORE, (Object)TRUST_STORE_PATH);
        config.set(SecurityOptions.SSL_REST_TRUSTSTORE_PASSWORD, (Object)"password");
    }

    private static void addInternalKeyStoreConfig(Configuration config) {
        config.set(SecurityOptions.SSL_INTERNAL_KEYSTORE, (Object)KEY_STORE_PATH);
        config.set(SecurityOptions.SSL_INTERNAL_KEYSTORE_PASSWORD, (Object)"password");
        config.set(SecurityOptions.SSL_INTERNAL_KEY_PASSWORD, (Object)"password");
    }

    private static void addInternalTrustStoreConfig(Configuration config) {
        config.set(SecurityOptions.SSL_INTERNAL_TRUSTSTORE, (Object)TRUST_STORE_PATH);
        config.set(SecurityOptions.SSL_INTERNAL_TRUSTSTORE_PASSWORD, (Object)"password");
    }

    private static String getSha1Fingerprint(Certificate cert) {
        if (cert == null) {
            return null;
        }
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA1");
            return SSLUtilsTest.toHexadecimalString(digest.digest(cert.getEncoded()));
        }
        catch (NoSuchAlgorithmException | CertificateEncodingException generalSecurityException) {
            return null;
        }
    }

    private static String toHexadecimalString(byte[] value) {
        StringBuilder sb = new StringBuilder();
        int len = value.length;
        for (int i = 0; i < len; ++i) {
            int num = value[i] & 0xFF;
            if (num < 16) {
                sb.append('0');
            }
            sb.append(Integer.toHexString(num));
            if (i >= len - 1) continue;
            sb.append(':');
        }
        return sb.toString().toUpperCase(Locale.US);
    }

    static {
        if (System.getProperty("flink.tests.with-openssl") != null) {
            Assertions.assertThat((boolean)OpenSsl.isAvailable()).isTrue();
            AVAILABLE_SSL_PROVIDERS = Arrays.asList("JDK", "OPENSSL");
        } else {
            AVAILABLE_SSL_PROVIDERS = Collections.singletonList("JDK");
        }
    }
}

