/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.storage.s3;

import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.base.Throwables;
import com.google.inject.Inject;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Date;
import org.apache.druid.common.utils.CurrentTimeMillisSupplier;
import org.apache.druid.java.util.common.IOE;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.java.util.common.logger.Logger;
import org.apache.druid.storage.s3.S3InputDataConfig;
import org.apache.druid.storage.s3.S3TaskLogsConfig;
import org.apache.druid.storage.s3.S3Utils;
import org.apache.druid.storage.s3.ServerSideEncryptingAmazonS3;
import org.apache.druid.tasklogs.TaskLogs;

public class S3TaskLogs
implements TaskLogs {
    private static final Logger log = new Logger(S3TaskLogs.class);
    private final ServerSideEncryptingAmazonS3 service;
    private final S3TaskLogsConfig config;
    private final S3InputDataConfig inputDataConfig;
    private final CurrentTimeMillisSupplier timeSupplier;

    @Inject
    public S3TaskLogs(ServerSideEncryptingAmazonS3 service, S3TaskLogsConfig config, S3InputDataConfig inputDataConfig, CurrentTimeMillisSupplier timeSupplier) {
        this.service = service;
        this.config = config;
        this.inputDataConfig = inputDataConfig;
        this.timeSupplier = timeSupplier;
    }

    public Optional<InputStream> streamTaskLog(String taskid, long offset) throws IOException {
        String taskKey = this.getTaskLogKey(taskid, "log");
        return this.streamTaskFileWithRetry(offset, taskKey);
    }

    public Optional<InputStream> streamTaskReports(String taskid) throws IOException {
        String taskKey = this.getTaskLogKey(taskid, "report.json");
        return this.streamTaskFileWithRetry(0L, taskKey);
    }

    public Optional<InputStream> streamTaskStatus(String taskid) throws IOException {
        String taskKey = this.getTaskLogKey(taskid, "status.json");
        return this.streamTaskFileWithRetry(0L, taskKey);
    }

    public void pushTaskPayload(String taskid, File taskPayloadFile) throws IOException {
        String taskKey = this.getTaskLogKey(taskid, "task.json");
        log.info("Pushing task payload [%s] to location [%s]", new Object[]{taskPayloadFile, taskKey});
        this.pushTaskFile(taskPayloadFile, taskKey);
    }

    public Optional<InputStream> streamTaskPayload(String taskid) throws IOException {
        String taskKey = this.getTaskLogKey(taskid, "task.json");
        return this.streamTaskFileWithRetry(0L, taskKey);
    }

    private Optional<InputStream> streamTaskFileWithRetry(long offset, String taskKey) throws IOException {
        try {
            return (Optional)S3Utils.retryS3Operation(() -> this.streamTaskFile(offset, taskKey));
        }
        catch (Exception e) {
            throw new IOE((Throwable)e, "Failed to stream logs for task[%s] starting at offset[%d]", new Object[]{taskKey, offset});
        }
    }

    private Optional<InputStream> streamTaskFile(long offset, String taskKey) {
        try {
            ObjectMetadata objectMetadata = this.service.getObjectMetadata(this.config.getS3Bucket(), taskKey);
            long end = objectMetadata.getContentLength() - 1L;
            long contentLength = objectMetadata.getContentLength();
            long start = offset >= contentLength || offset <= -contentLength ? 0L : (offset >= 0L ? offset : contentLength + offset);
            GetObjectRequest request = new GetObjectRequest(this.config.getS3Bucket(), taskKey).withMatchingETagConstraint(S3TaskLogs.ensureQuotated(objectMetadata.getETag())).withRange(start, end);
            return Optional.of((Object)this.service.getObject(request).getObjectContent());
        }
        catch (AmazonS3Exception e) {
            if (404 == e.getStatusCode() || "NoSuchKey".equals(e.getErrorCode()) || "NoSuchBucket".equals(e.getErrorCode())) {
                return Optional.absent();
            }
            throw e;
        }
    }

    static String ensureQuotated(String eTag) {
        if (eTag != null && !eTag.startsWith("\"") && !eTag.endsWith("\"")) {
            return "\"" + eTag + "\"";
        }
        return eTag;
    }

    public void pushTaskLog(String taskid, File logFile) throws IOException {
        String taskKey = this.getTaskLogKey(taskid, "log");
        log.info("Pushing task log %s to: %s", new Object[]{logFile, taskKey});
        this.pushTaskFile(logFile, taskKey);
    }

    public void pushTaskReports(String taskid, File reportFile) throws IOException {
        String taskKey = this.getTaskLogKey(taskid, "report.json");
        log.info("Pushing task reports %s to: %s", new Object[]{reportFile, taskKey});
        this.pushTaskFile(reportFile, taskKey);
    }

    public void pushTaskStatus(String taskid, File statusFile) throws IOException {
        String taskKey = this.getTaskLogKey(taskid, "status.json");
        log.info("Pushing task status %s to: %s", new Object[]{statusFile, taskKey});
        this.pushTaskFile(statusFile, taskKey);
    }

    private void pushTaskFile(File logFile, String taskKey) throws IOException {
        try {
            S3Utils.retryS3Operation(() -> {
                S3Utils.uploadFileIfPossible(this.service, this.config.getDisableAcl(), this.config.getS3Bucket(), taskKey, logFile);
                return null;
            });
        }
        catch (Exception e) {
            Throwables.propagateIfInstanceOf((Throwable)e, IOException.class);
            throw new RuntimeException(e);
        }
    }

    String getTaskLogKey(String taskid, String filename) {
        return StringUtils.format((String)"%s/%s/%s", (Object[])new Object[]{this.config.getS3Prefix(), taskid, filename});
    }

    public void killAll() throws IOException {
        log.info("Deleting all task logs from s3 location [bucket: '%s' prefix: '%s'].", new Object[]{this.config.getS3Bucket(), this.config.getS3Prefix()});
        long now = this.timeSupplier.getAsLong();
        this.killOlderThan(now);
    }

    public void killOlderThan(long timestamp) throws IOException {
        log.info("Deleting all task logs from s3 location [bucket: '%s' prefix: '%s'] older than %s.", new Object[]{this.config.getS3Bucket(), this.config.getS3Prefix(), new Date(timestamp)});
        try {
            S3Utils.deleteObjectsInPath(this.service, this.inputDataConfig.getMaxListingLength(), this.config.getS3Bucket(), this.config.getS3Prefix(), (Predicate<S3ObjectSummary>)((Predicate)object -> object.getLastModified().getTime() < timestamp));
        }
        catch (Exception e) {
            log.error("Error occurred while deleting task log files from s3. Error: %s", new Object[]{e.getMessage()});
            throw new IOException(e);
        }
    }
}

