/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.protocol.datatransfer.sasl;

import com.google.common.base.Charsets;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import com.google.common.net.InetAddresses;
import com.google.protobuf.ByteString;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.codec.binary.Base64;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.crypto.CipherOption;
import org.apache.hadoop.crypto.CipherSuite;
import org.apache.hadoop.crypto.CryptoCodec;
import org.apache.hadoop.crypto.CryptoInputStream;
import org.apache.hadoop.crypto.CryptoOutputStream;
import org.apache.hadoop.hdfs.net.Peer;
import org.apache.hadoop.hdfs.protocol.datatransfer.IOStreamPair;
import org.apache.hadoop.hdfs.protocol.datatransfer.InvalidEncryptionKeyException;
import org.apache.hadoop.hdfs.protocol.datatransfer.sasl.SaslParticipant;
import org.apache.hadoop.hdfs.protocol.datatransfer.sasl.SaslResponseWithNegotiatedCipherOption;
import org.apache.hadoop.hdfs.protocol.proto.DataTransferProtos;
import org.apache.hadoop.hdfs.protocol.proto.HdfsProtos;
import org.apache.hadoop.hdfs.protocolPB.PBHelperClient;
import org.apache.hadoop.security.SaslPropertiesResolver;
import org.apache.hadoop.security.SaslRpcServer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public final class DataTransferSaslUtil {
    private static final Logger LOG = LoggerFactory.getLogger(DataTransferSaslUtil.class);
    public static final String NAME_DELIMITER = " ";
    public static final int SASL_TRANSFER_MAGIC_NUMBER = -559038737;

    public static void checkSaslComplete(SaslParticipant sasl, Map<String, String> saslProps) throws IOException {
        if (!sasl.isComplete()) {
            throw new IOException("Failed to complete SASL handshake");
        }
        ImmutableSet<String> requestedQop = ImmutableSet.copyOf(Arrays.asList(saslProps.get("javax.security.sasl.qop").split(",")));
        String negotiatedQop = sasl.getNegotiatedQop();
        LOG.debug("Verifying QOP, requested QOP = {}, negotiated QOP = {}", requestedQop, (Object)negotiatedQop);
        if (!requestedQop.contains(negotiatedQop)) {
            throw new IOException(String.format("SASL handshake completed, but channel does not have acceptable quality of protection, requested = %s, negotiated = %s", requestedQop, negotiatedQop));
        }
    }

    public static boolean requestedQopContainsPrivacy(Map<String, String> saslProps) {
        ImmutableSet<String> requestedQop = ImmutableSet.copyOf(Arrays.asList(saslProps.get("javax.security.sasl.qop").split(",")));
        return requestedQop.contains("auth-conf");
    }

    public static Map<String, String> createSaslPropertiesForEncryption(String encryptionAlgorithm) {
        HashMap<String, String> saslProps = Maps.newHashMapWithExpectedSize(3);
        saslProps.put("javax.security.sasl.qop", SaslRpcServer.QualityOfProtection.PRIVACY.getSaslQop());
        saslProps.put("javax.security.sasl.server.authentication", "true");
        saslProps.put("com.sun.security.sasl.digest.cipher", encryptionAlgorithm);
        return saslProps;
    }

    public static char[] encryptionKeyToPassword(byte[] encryptionKey) {
        return new String(Base64.encodeBase64((byte[])encryptionKey, (boolean)false), Charsets.UTF_8).toCharArray();
    }

    public static InetAddress getPeerAddress(Peer peer) {
        String remoteAddr = peer.getRemoteAddressString().split(":")[0];
        int slashIdx = remoteAddr.indexOf(47);
        return InetAddresses.forString(slashIdx != -1 ? remoteAddr.substring(slashIdx + 1, remoteAddr.length()) : remoteAddr);
    }

    public static SaslPropertiesResolver getSaslPropertiesResolver(Configuration conf) {
        String qops = conf.get("dfs.data.transfer.protection");
        if (qops == null || qops.isEmpty()) {
            LOG.debug("DataTransferProtocol not using SaslPropertiesResolver, no QOP found in configuration for {}", (Object)"dfs.data.transfer.protection");
            return null;
        }
        Configuration saslPropsResolverConf = new Configuration(conf);
        saslPropsResolverConf.set("hadoop.rpc.protection", qops);
        Class resolverClass = conf.getClass("hadoop.security.saslproperties.resolver.class", SaslPropertiesResolver.class, SaslPropertiesResolver.class);
        resolverClass = conf.getClass("dfs.data.transfer.saslproperties.resolver.class", resolverClass, SaslPropertiesResolver.class);
        saslPropsResolverConf.setClass("hadoop.security.saslproperties.resolver.class", resolverClass, SaslPropertiesResolver.class);
        SaslPropertiesResolver resolver = SaslPropertiesResolver.getInstance((Configuration)saslPropsResolverConf);
        LOG.debug("DataTransferProtocol using SaslPropertiesResolver, configured QOP {} = {}, configured class {} = {}", new Object[]{"dfs.data.transfer.protection", qops, "dfs.data.transfer.saslproperties.resolver.class", resolverClass});
        return resolver;
    }

    public static byte[] readSaslMessage(InputStream in) throws IOException {
        DataTransferProtos.DataTransferEncryptorMessageProto proto = DataTransferProtos.DataTransferEncryptorMessageProto.parseFrom(PBHelperClient.vintPrefixed(in));
        if (proto.getStatus() == DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.ERROR_UNKNOWN_KEY) {
            throw new InvalidEncryptionKeyException(proto.getMessage());
        }
        if (proto.getStatus() == DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.ERROR) {
            throw new IOException(proto.getMessage());
        }
        return proto.getPayload().toByteArray();
    }

    public static byte[] readSaslMessageAndNegotiationCipherOptions(InputStream in, List<CipherOption> cipherOptions) throws IOException {
        DataTransferProtos.DataTransferEncryptorMessageProto proto = DataTransferProtos.DataTransferEncryptorMessageProto.parseFrom(PBHelperClient.vintPrefixed(in));
        if (proto.getStatus() == DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.ERROR_UNKNOWN_KEY) {
            throw new InvalidEncryptionKeyException(proto.getMessage());
        }
        if (proto.getStatus() == DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.ERROR) {
            throw new IOException(proto.getMessage());
        }
        List<HdfsProtos.CipherOptionProto> optionProtos = proto.getCipherOptionList();
        if (optionProtos != null) {
            for (HdfsProtos.CipherOptionProto optionProto : optionProtos) {
                cipherOptions.add(PBHelperClient.convert(optionProto));
            }
        }
        return proto.getPayload().toByteArray();
    }

    public static SaslMessageWithHandshake readSaslMessageWithHandshakeSecret(InputStream in) throws IOException {
        DataTransferProtos.DataTransferEncryptorMessageProto proto = DataTransferProtos.DataTransferEncryptorMessageProto.parseFrom(PBHelperClient.vintPrefixed(in));
        if (proto.getStatus() == DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.ERROR_UNKNOWN_KEY) {
            throw new InvalidEncryptionKeyException(proto.getMessage());
        }
        if (proto.getStatus() == DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.ERROR) {
            throw new IOException(proto.getMessage());
        }
        byte[] payload = proto.getPayload().toByteArray();
        byte[] secret = null;
        String bpid = null;
        if (proto.hasHandshakeSecret()) {
            DataTransferProtos.HandshakeSecretProto handshakeSecret = proto.getHandshakeSecret();
            secret = handshakeSecret.getSecret().toByteArray();
            bpid = handshakeSecret.getBpid();
        }
        return new SaslMessageWithHandshake(payload, secret, bpid);
    }

    public static CipherOption negotiateCipherOption(Configuration conf, List<CipherOption> options) throws IOException {
        String cipherSuites = conf.get("dfs.encrypt.data.transfer.cipher.suites");
        if (cipherSuites == null || cipherSuites.isEmpty()) {
            return null;
        }
        if (!cipherSuites.equals(CipherSuite.AES_CTR_NOPADDING.getName())) {
            throw new IOException(String.format("Invalid cipher suite, %s=%s", "dfs.encrypt.data.transfer.cipher.suites", cipherSuites));
        }
        if (options != null) {
            for (CipherOption option : options) {
                CipherSuite suite = option.getCipherSuite();
                if (suite != CipherSuite.AES_CTR_NOPADDING) continue;
                int keyLen = conf.getInt("dfs.encrypt.data.transfer.cipher.key.bitlength", 128) / 8;
                CryptoCodec codec = CryptoCodec.getInstance((Configuration)conf, (CipherSuite)suite);
                byte[] inKey = new byte[keyLen];
                byte[] inIv = new byte[suite.getAlgorithmBlockSize()];
                byte[] outKey = new byte[keyLen];
                byte[] outIv = new byte[suite.getAlgorithmBlockSize()];
                assert (codec != null);
                codec.generateSecureRandom(inKey);
                codec.generateSecureRandom(inIv);
                codec.generateSecureRandom(outKey);
                codec.generateSecureRandom(outIv);
                codec.close();
                return new CipherOption(suite, inKey, inIv, outKey, outIv);
            }
        }
        return null;
    }

    public static void sendSaslMessageAndNegotiatedCipherOption(OutputStream out, byte[] payload, CipherOption option) throws IOException {
        DataTransferProtos.DataTransferEncryptorMessageProto.Builder builder = DataTransferProtos.DataTransferEncryptorMessageProto.newBuilder();
        builder.setStatus(DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.SUCCESS);
        if (payload != null) {
            builder.setPayload(ByteString.copyFrom(payload));
        }
        if (option != null) {
            builder.addCipherOption(PBHelperClient.convert(option));
        }
        DataTransferProtos.DataTransferEncryptorMessageProto proto = builder.build();
        proto.writeDelimitedTo(out);
        out.flush();
    }

    public static IOStreamPair createStreamPair(Configuration conf, CipherOption cipherOption, OutputStream out, InputStream in, boolean isServer) throws IOException {
        LOG.debug("Creating IOStreamPair of CryptoInputStream and CryptoOutputStream.");
        CryptoCodec codec = CryptoCodec.getInstance((Configuration)conf, (CipherSuite)cipherOption.getCipherSuite());
        byte[] inKey = cipherOption.getInKey();
        byte[] inIv = cipherOption.getInIv();
        byte[] outKey = cipherOption.getOutKey();
        byte[] outIv = cipherOption.getOutIv();
        CryptoInputStream cIn = new CryptoInputStream(in, codec, isServer ? inKey : outKey, isServer ? inIv : outIv);
        CryptoOutputStream cOut = new CryptoOutputStream(out, codec, isServer ? outKey : inKey, isServer ? outIv : inIv);
        return new IOStreamPair((InputStream)cIn, (OutputStream)cOut);
    }

    public static void sendGenericSaslErrorMessage(OutputStream out, String message) throws IOException {
        DataTransferSaslUtil.sendSaslMessage(out, DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.ERROR, null, message);
    }

    public static void sendSaslMessage(OutputStream out, byte[] payload) throws IOException {
        DataTransferSaslUtil.sendSaslMessage(out, DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.SUCCESS, payload, null);
    }

    public static void sendSaslMessageHandshakeSecret(OutputStream out, byte[] payload, byte[] secret, String bpid) throws IOException {
        DataTransferSaslUtil.sendSaslMessageHandshakeSecret(out, DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.SUCCESS, payload, null, secret, bpid);
    }

    public static void sendSaslMessageAndNegotiationCipherOptions(OutputStream out, byte[] payload, List<CipherOption> options) throws IOException {
        DataTransferProtos.DataTransferEncryptorMessageProto.Builder builder = DataTransferProtos.DataTransferEncryptorMessageProto.newBuilder();
        builder.setStatus(DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.SUCCESS);
        if (payload != null) {
            builder.setPayload(ByteString.copyFrom(payload));
        }
        if (options != null) {
            builder.addAllCipherOption(PBHelperClient.convertCipherOptions(options));
        }
        DataTransferProtos.DataTransferEncryptorMessageProto proto = builder.build();
        proto.writeDelimitedTo(out);
        out.flush();
    }

    public static SaslResponseWithNegotiatedCipherOption readSaslMessageAndNegotiatedCipherOption(InputStream in) throws IOException {
        DataTransferProtos.DataTransferEncryptorMessageProto proto = DataTransferProtos.DataTransferEncryptorMessageProto.parseFrom(PBHelperClient.vintPrefixed(in));
        if (proto.getStatus() == DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.ERROR_UNKNOWN_KEY) {
            throw new InvalidEncryptionKeyException(proto.getMessage());
        }
        if (proto.getStatus() == DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.ERROR) {
            throw new IOException(proto.getMessage());
        }
        byte[] response = proto.getPayload().toByteArray();
        List<CipherOption> options = PBHelperClient.convertCipherOptionProtos(proto.getCipherOptionList());
        CipherOption option = null;
        if (options != null && !options.isEmpty()) {
            option = options.get(0);
        }
        return new SaslResponseWithNegotiatedCipherOption(response, option);
    }

    public static CipherOption wrap(CipherOption option, SaslParticipant sasl) throws IOException {
        if (option != null) {
            byte[] outKey;
            byte[] inKey = option.getInKey();
            if (inKey != null) {
                inKey = sasl.wrap(inKey, 0, inKey.length);
            }
            if ((outKey = option.getOutKey()) != null) {
                outKey = sasl.wrap(outKey, 0, outKey.length);
            }
            return new CipherOption(option.getCipherSuite(), inKey, option.getInIv(), outKey, option.getOutIv());
        }
        return null;
    }

    public static CipherOption unwrap(CipherOption option, SaslParticipant sasl) throws IOException {
        if (option != null) {
            byte[] outKey;
            byte[] inKey = option.getInKey();
            if (inKey != null) {
                inKey = sasl.unwrap(inKey, 0, inKey.length);
            }
            if ((outKey = option.getOutKey()) != null) {
                outKey = sasl.unwrap(outKey, 0, outKey.length);
            }
            return new CipherOption(option.getCipherSuite(), inKey, option.getInIv(), outKey, option.getOutIv());
        }
        return null;
    }

    public static void sendSaslMessage(OutputStream out, DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus status, byte[] payload, String message) throws IOException {
        DataTransferSaslUtil.sendSaslMessage(out, status, payload, message, null);
    }

    public static void sendSaslMessage(OutputStream out, DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus status, byte[] payload, String message, DataTransferProtos.HandshakeSecretProto handshakeSecret) throws IOException {
        DataTransferProtos.DataTransferEncryptorMessageProto.Builder builder = DataTransferProtos.DataTransferEncryptorMessageProto.newBuilder();
        builder.setStatus(status);
        if (payload != null) {
            builder.setPayload(ByteString.copyFrom(payload));
        }
        if (message != null) {
            builder.setMessage(message);
        }
        if (handshakeSecret != null) {
            builder.setHandshakeSecret(handshakeSecret);
        }
        DataTransferProtos.DataTransferEncryptorMessageProto proto = builder.build();
        proto.writeDelimitedTo(out);
        out.flush();
    }

    public static void sendSaslMessageHandshakeSecret(OutputStream out, DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus status, byte[] payload, String message, byte[] secret, String bpid) throws IOException {
        DataTransferProtos.HandshakeSecretProto.Builder builder = DataTransferProtos.HandshakeSecretProto.newBuilder();
        builder.setSecret(ByteString.copyFrom(secret));
        builder.setBpid(bpid);
        DataTransferSaslUtil.sendSaslMessage(out, status, payload, message, builder.build());
    }

    private DataTransferSaslUtil() {
    }

    static class SaslMessageWithHandshake {
        private final byte[] payload;
        private final byte[] secret;
        private final String bpid;

        SaslMessageWithHandshake(byte[] payload, byte[] secret, String bpid) {
            this.payload = payload;
            this.secret = secret;
            this.bpid = bpid;
        }

        byte[] getPayload() {
            return this.payload;
        }

        byte[] getSecret() {
            return this.secret;
        }

        String getBpid() {
            return this.bpid;
        }
    }
}

