/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.common.property;

import com.google.common.collect.ImmutableMap;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Arrays;
import java.util.Date;
import java.util.Optional;
import org.apache.doris.common.io.Text;
import org.apache.doris.thrift.TPropertyVal;

public abstract class PropertySchema<T> {
    private final String name;
    private final boolean required;
    private Optional<T> defaultValue = Optional.empty();
    private Optional<T> maxValue = Optional.empty();
    private Optional<T> minValue = Optional.empty();

    protected PropertySchema(String name) {
        this(name, false);
    }

    public PropertySchema(String name, boolean required) {
        this.name = name;
        this.required = required;
    }

    public static ImmutableMap<String, PropertySchema> createSchemas(PropertySchema ... schemas) {
        ImmutableMap.Builder builder = ImmutableMap.builder();
        Arrays.stream(schemas).forEach(s -> builder.put((Object)s.getName(), s));
        return builder.build();
    }

    PropertySchema<T> setDefauleValue(T val) {
        this.defaultValue = Optional.of(val);
        return this;
    }

    PropertySchema<T> setMin(T min) {
        this.minValue = Optional.of(min);
        return this;
    }

    PropertySchema<T> setMax(T max) {
        this.maxValue = Optional.of(max);
        return this;
    }

    public String getName() {
        return this.name;
    }

    public boolean isRequired() {
        return this.required;
    }

    public Optional<T> getDefaultValue() {
        return this.defaultValue;
    }

    public Optional<T> getMinValue() {
        return this.minValue;
    }

    public Optional<T> getMaxValue() {
        return this.maxValue;
    }

    public abstract T read(String var1) throws IllegalArgumentException;

    public abstract T read(TPropertyVal var1) throws IllegalArgumentException;

    public abstract T read(DataInput var1) throws IOException;

    public abstract void write(T var1, TPropertyVal var2);

    public abstract void write(T var1, DataOutput var2) throws IOException;

    private static abstract class ComparableProperty<T extends Comparable>
    extends PropertySchema<T> {
        protected ComparableProperty(String name) {
            super(name);
        }

        protected ComparableProperty(String name, boolean isRequired) {
            super(name, isRequired);
        }

        protected void verifyRange(T val) throws IllegalArgumentException {
            if (this.getMinValue().isPresent() && (val == null || ((Comparable)this.getMinValue().get()).compareTo(val) > 0)) {
                throw new IllegalArgumentException(val + " should not be less than " + this.getMinValue().get());
            }
            if (this.getMaxValue().isPresent() && (val == null || ((Comparable)this.getMaxValue().get()).compareTo(val) < 0)) {
                throw new IllegalArgumentException(val + " should not be greater than " + this.getMaxValue().get());
            }
        }
    }

    public static final class EnumProperty<T extends Enum<T>>
    extends PropertySchema<T> {
        private final Class<T> enumClass;

        EnumProperty(String name, Class<T> enumClass) {
            super(name);
            this.enumClass = enumClass;
        }

        EnumProperty(String name, Class<T> enumClass, boolean isRequired) {
            super(name, isRequired);
            this.enumClass = enumClass;
        }

        @Override
        public T read(String rawVal) {
            if (rawVal == null || rawVal.length() == 0) {
                throw new IllegalArgumentException(this.formatError(rawVal));
            }
            try {
                return Enum.valueOf(this.enumClass, rawVal.toUpperCase());
            }
            catch (IllegalArgumentException e) {
                throw new IllegalArgumentException(this.formatError(rawVal));
            }
        }

        @Override
        public T read(TPropertyVal tVal) throws IllegalArgumentException {
            return Enum.valueOf(this.enumClass, tVal.getStrVal());
        }

        @Override
        public T read(DataInput input) throws IOException {
            return Enum.valueOf(this.enumClass, Text.readString(input));
        }

        @Override
        public void write(T val, TPropertyVal out) {
            out.setStrVal(((Enum)val).name());
        }

        @Override
        public void write(T val, DataOutput out) throws IOException {
            Text.writeString(out, ((Enum)val).name());
        }

        private String formatError(String rawVal) {
            String enumsStr = Arrays.stream(this.enumClass.getEnumConstants()).map(Enum::toString).reduce((sa, sb) -> sa + ", " + sb).orElse("");
            return String.format("Expected values are [%s], while [%s] provided", enumsStr, rawVal);
        }
    }

    public static final class DateProperty
    extends PropertySchema<Date> {
        DateTimeFormatter dateFormat;

        public DateProperty(String name, DateTimeFormatter dateFormat) {
            super(name);
            this.dateFormat = dateFormat;
        }

        DateProperty(String name, DateTimeFormatter dateFormat, boolean isRequired) {
            super(name, isRequired);
            this.dateFormat = dateFormat;
        }

        @Override
        public Date read(String rawVal) throws IllegalArgumentException {
            if (rawVal == null) {
                throw new IllegalArgumentException("Invalid time format, time param can not is null");
            }
            return this.readTimeFormat(rawVal);
        }

        @Override
        public Date read(TPropertyVal tVal) throws IllegalArgumentException {
            return this.readTimeFormat(tVal.getStrVal());
        }

        @Override
        public Date read(DataInput input) throws IOException {
            return this.readTimeFormat(Text.readString(input));
        }

        @Override
        public void write(Date val, TPropertyVal out) {
            out.setStrVal(this.writeTimeFormat(val));
        }

        @Override
        public void write(Date val, DataOutput out) throws IOException {
            Text.writeString(out, this.writeTimeFormat(val));
        }

        public Date readTimeFormat(String timeStr) throws IllegalArgumentException {
            try {
                return Date.from(LocalDateTime.parse(timeStr, this.dateFormat).atZone(ZoneId.systemDefault()).toInstant());
            }
            catch (DateTimeParseException e) {
                throw new IllegalArgumentException("Invalid time format, time param need to be " + this.dateFormat.toString());
            }
        }

        public String writeTimeFormat(Date timeDate) throws IllegalArgumentException {
            return LocalDateTime.ofInstant(timeDate.toInstant(), ZoneId.systemDefault()).format(this.dateFormat);
        }
    }

    public static final class BooleanProperty
    extends ComparableProperty<Boolean> {
        BooleanProperty(String name) {
            super(name);
        }

        BooleanProperty(String name, boolean isRequired) {
            super(name, isRequired);
        }

        @Override
        public Boolean read(String rawVal) {
            if (rawVal == null || !rawVal.equalsIgnoreCase("true") && !rawVal.equalsIgnoreCase("false")) {
                throw new IllegalArgumentException(String.format("Invalid boolean : %s, use true or false", rawVal));
            }
            try {
                return Boolean.parseBoolean(rawVal);
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException(String.format("Invalid boolean %s: %s", rawVal, e.getMessage()));
            }
        }

        @Override
        public Boolean read(TPropertyVal tVal) throws IllegalArgumentException {
            return tVal.isBoolVal();
        }

        @Override
        public Boolean read(DataInput input) throws IOException {
            return input.readBoolean();
        }

        @Override
        public void write(Boolean val, TPropertyVal out) {
            out.setBoolVal(val);
        }

        @Override
        public void write(Boolean val, DataOutput out) throws IOException {
            out.writeBoolean(val);
        }
    }

    public static final class LongProperty
    extends ComparableProperty<Long> {
        LongProperty(String name) {
            super(name);
        }

        LongProperty(String name, boolean isRequired) {
            super(name, isRequired);
        }

        @Override
        public Long read(String rawVal) {
            try {
                Long val = Long.parseLong(rawVal);
                this.verifyRange(val);
                return val;
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException(String.format("Invalid long %s: %s", rawVal, e.getMessage()));
            }
        }

        @Override
        public Long read(TPropertyVal tVal) throws IllegalArgumentException {
            this.verifyRange(tVal.getLongVal());
            return tVal.getLongVal();
        }

        @Override
        public Long read(DataInput input) throws IOException {
            return input.readLong();
        }

        @Override
        public void write(Long val, TPropertyVal out) {
            out.setLongVal(val);
        }

        @Override
        public void write(Long val, DataOutput out) throws IOException {
            out.writeLong(val);
        }
    }

    public static final class IntProperty
    extends ComparableProperty<Integer> {
        IntProperty(String name) {
            super(name);
        }

        IntProperty(String name, boolean isRequired) {
            super(name, isRequired);
        }

        @Override
        public Integer read(String rawVal) {
            try {
                Integer val = Integer.parseInt(rawVal);
                this.verifyRange(val);
                return val;
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException(String.format("Invalid integer %s: %s", rawVal, e.getMessage()));
            }
        }

        @Override
        public Integer read(TPropertyVal tVal) throws IllegalArgumentException {
            this.verifyRange(tVal.getIntVal());
            return tVal.getIntVal();
        }

        @Override
        public Integer read(DataInput input) throws IOException {
            return input.readInt();
        }

        @Override
        public void write(Integer val, TPropertyVal out) {
            out.setIntVal(val);
        }

        @Override
        public void write(Integer val, DataOutput out) throws IOException {
            out.writeInt(val);
        }
    }

    public static final class StringProperty
    extends ComparableProperty<String> {
        StringProperty(String name) {
            super(name);
        }

        StringProperty(String name, boolean isRequired) {
            super(name, isRequired);
        }

        @Override
        public String read(String rawVal) throws IllegalArgumentException {
            this.verifyRange(rawVal);
            return rawVal;
        }

        @Override
        public String read(TPropertyVal tVal) throws IllegalArgumentException {
            this.verifyRange(tVal.getStrVal());
            return tVal.getStrVal();
        }

        @Override
        public String read(DataInput input) throws IOException {
            return Text.readString(input);
        }

        @Override
        public void write(String val, TPropertyVal out) {
            out.setStrVal(val);
        }

        @Override
        public void write(String val, DataOutput out) throws IOException {
            Text.writeString(out, val);
        }
    }

    public static interface SchemaGroup {
        public ImmutableMap<String, PropertySchema> getSchemas();
    }
}

