/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.rng.examples.stress;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Formatter;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.rng.examples.stress.AlphaNumericComparator;
import org.apache.commons.rng.examples.stress.ApplicationException;
import org.apache.commons.rng.examples.stress.LogUtils;
import org.apache.commons.rng.examples.stress.StandardOptions;
import org.apache.commons.rng.simple.RandomSource;
import picocli.CommandLine;

@CommandLine.Command(name="results", description={"Collate results from stress test applications."})
class ResultsCommand
implements Callable<Void> {
    private static final Pattern RANDOM_SOURCE_PATTERN = Pattern.compile("^# RandomSource: (.*)");
    private static final Pattern RNG_PATTERN = Pattern.compile("^# RNG: (.*)");
    private static final Pattern TEST_EXIT_PATTERN = Pattern.compile("^# Exit value: (\\d+)");
    private static final Pattern DIEHARDER_PATTERN = Pattern.compile("^# *dieharder version");
    private static final Pattern DIEHARDER_FAILED_PATTERN = Pattern.compile("FAILED *$");
    private static final Pattern TESTU01_PATTERN = Pattern.compile("^ *Version: TestU01");
    private static final Pattern TESTU01_SUMMARY_PATTERN = Pattern.compile("^========= Summary results of (\\S*) ");
    private static final Pattern TESTU01_TEST_RESULT_PATTERN = Pattern.compile("^  ?(\\d+  .*)    ");
    private static final Pattern TESTU01_STARTING_PATTERN = Pattern.compile("^ *Starting (\\S*)");
    private static final Pattern PRACTRAND_PATTERN = Pattern.compile("PractRand version");
    private static final Pattern PRACTRAND_OUTPUT_SIZE_PATTERN = Pattern.compile("\\(2\\^(\\d+) bytes\\)");
    private static final Pattern PRACTRAND_FAILED_PATTERN = Pattern.compile("FAIL *!* *$");
    private static final String DIEHARDER_SUMS = "diehard_sums";
    private static final String BIT_REVERSED = "Bit-reversed";
    private static final char FORWARD_SLASH = '\\';
    private static final char BACK_SLASH = '\\';
    private static final char PIPE = '|';
    private static final String COLUMN_RNG = "RNG";
    private static final String[] BINARY_UNITS = new String[]{" B", " KiB", " MiB", " GiB", " TiB", " PiB", " EiB"};
    @CommandLine.Mixin
    private StandardOptions reusableOptions;
    @CommandLine.Parameters(arity="1..*", description={"The results files."}, paramLabel="<file>")
    private List<File> resultsFiles = new ArrayList<File>();
    @CommandLine.Option(names={"-o", "--out"}, description={"The output file (default: stdout)."})
    private File fileOutput;
    @CommandLine.Option(names={"-f", "--format"}, description={"Output format (default: ${DEFAULT-VALUE}).", "Valid values: ${COMPLETION-CANDIDATES}."})
    private OutputFormat outputFormat = OutputFormat.TXT;
    @CommandLine.Option(names={"--failed"}, description={"Output failed tests (support varies by format).", "CSV: List individual test failures.", "APT: Count of systematic test failures."})
    private boolean showFailedTests;
    @CommandLine.Option(names={"--include-sums"}, description={"Include Dieharder sums test."})
    private boolean includeDiehardSums;
    @CommandLine.Option(names={"--path-prefix"}, description={"Common path prefix.", "If specified this will replace the common prefix from all files when the path is output, e.g. for the APT report."})
    private String pathPrefix = "";
    @CommandLine.Option(names={"-i", "--ignore"}, description={"Ignore partial results."})
    private boolean ignorePartialResults;
    @CommandLine.Option(names={"--delete"}, description={"Delete partial results files.", "This is not reversible!"})
    private boolean deletePartialResults;

    ResultsCommand() {
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public Void call() {
        LogUtils.setLogLevel(this.reusableOptions.logLevel);
        List<TestResult> results = this.readResults();
        if (this.deletePartialResults) {
            ResultsCommand.deleteIfIncomplete(results);
            return null;
        }
        try (OutputStream out = this.createOutputStream();){
            switch (this.outputFormat) {
                case CSV: {
                    this.writeCSVData(out, results);
                    return null;
                }
                case APT: {
                    this.writeAPT(out, results);
                    return null;
                }
                case TXT: {
                    ResultsCommand.writeTXT(out, results);
                    return null;
                }
                case FAILURES: {
                    ResultsCommand.writeFailures(out, results);
                    return null;
                }
                default: {
                    throw new ApplicationException("Unknown output format: " + (Object)((Object)this.outputFormat));
                }
            }
        }
        catch (IOException ex) {
            throw new ApplicationException("IO error: " + ex.getMessage(), ex);
        }
    }

    private List<TestResult> readResults() {
        ArrayList<TestResult> results = new ArrayList<TestResult>();
        for (File resultFile : this.resultsFiles) {
            this.readResults(results, resultFile);
        }
        return results;
    }

    private void readResults(List<TestResult> results, File resultFile) {
        List<String> contents = ResultsCommand.readFileContents(resultFile);
        List<List<String>> outputs = ResultsCommand.splitContents(contents);
        if (outputs.isEmpty()) {
            LogUtils.error("No test output in file: %s", resultFile);
        } else {
            for (List<String> testOutput : outputs) {
                TestResult result = this.readResult(resultFile, testOutput);
                if (!result.isComplete()) {
                    LogUtils.info("Partial results in file: %s", resultFile);
                    if (this.ignorePartialResults) continue;
                }
                results.add(result);
            }
        }
    }

    private static List<String> readFileContents(File resultFile) {
        ArrayList<String> contents = new ArrayList<String>();
        try (BufferedReader reader = Files.newBufferedReader(resultFile.toPath());){
            String line = reader.readLine();
            while (line != null) {
                contents.add(line);
                line = reader.readLine();
            }
        }
        catch (IOException ex) {
            throw new ApplicationException("Failed to read file contents: " + resultFile, ex);
        }
        return contents;
    }

    private static List<List<String>> splitContents(List<String> contents) {
        ArrayList<List<String>> testOutputs = new ArrayList<List<String>>();
        int begin = -1;
        for (int i = 0; i < contents.size(); ++i) {
            if (!RANDOM_SOURCE_PATTERN.matcher(contents.get(i)).matches()) continue;
            if (begin >= 0) {
                testOutputs.add(contents.subList(begin, i));
            }
            begin = i;
        }
        if (begin >= 0) {
            testOutputs.add(contents.subList(begin, contents.size()));
        }
        return testOutputs;
    }

    private TestResult readResult(File resultFile, List<String> testOutput) {
        ListIterator<String> iter = testOutput.listIterator();
        RandomSource randomSource = ResultsCommand.getRandomSource(resultFile, iter);
        boolean bitReversed = ResultsCommand.getBitReversed(resultFile, iter);
        TestFormat testFormat = this.getTestFormat(resultFile, iter);
        TestResult testResult = ResultsCommand.createTestResult(resultFile, randomSource, bitReversed, testFormat);
        if (testFormat == TestFormat.DIEHARDER) {
            this.readDieharder(iter, testResult);
        } else if (testFormat == TestFormat.TESTU01) {
            this.readTestU01(resultFile, iter, testResult);
        } else {
            ResultsCommand.readPractRand(iter, (PractRandTestResult)testResult);
        }
        return testResult;
    }

    private static TestResult createTestResult(File resultFile, RandomSource randomSource, boolean bitReversed, TestFormat testFormat) {
        return testFormat == TestFormat.PRACTRAND ? new PractRandTestResult(resultFile, randomSource, bitReversed, testFormat) : new TestResult(resultFile, randomSource, bitReversed, testFormat);
    }

    private static RandomSource getRandomSource(File resultFile, Iterator<String> iter) {
        while (iter.hasNext()) {
            Matcher matcher = RANDOM_SOURCE_PATTERN.matcher(iter.next());
            if (!matcher.matches()) continue;
            return RandomSource.valueOf((String)matcher.group(1));
        }
        throw new ApplicationException("Failed to find RandomSource header line: " + resultFile);
    }

    private static boolean getBitReversed(File resultFile, Iterator<String> iter) {
        while (iter.hasNext()) {
            Matcher matcher = RNG_PATTERN.matcher(iter.next());
            if (!matcher.matches()) continue;
            return matcher.group(1).contains(BIT_REVERSED);
        }
        throw new ApplicationException("Failed to find RNG header line: " + resultFile);
    }

    private TestFormat getTestFormat(File resultFile, Iterator<String> iter) {
        while (iter.hasNext()) {
            String line = iter.next();
            if (DIEHARDER_PATTERN.matcher(line).find()) {
                return TestFormat.DIEHARDER;
            }
            if (TESTU01_PATTERN.matcher(line).find()) {
                return TestFormat.TESTU01;
            }
            if (!PRACTRAND_PATTERN.matcher(line).find()) continue;
            return TestFormat.PRACTRAND;
        }
        if (!this.ignorePartialResults) {
            throw new ApplicationException("Failed to identify the test application format: " + resultFile);
        }
        LogUtils.error("Failed to identify the test application format: %s", resultFile);
        return null;
    }

    private void readDieharder(Iterator<String> iter, TestResult testResult) {
        testResult.setTestApplicationName("Dieharder");
        while (iter.hasNext()) {
            String line = iter.next();
            if (DIEHARDER_FAILED_PATTERN.matcher(line).find()) {
                if (!this.includeDiehardSums && line.contains(DIEHARDER_SUMS)) continue;
                int index1 = line.indexOf(124);
                int index2 = line.indexOf(124, index1 + 1);
                testResult.addFailedTest(line.substring(0, index1).trim() + ":" + line.substring(index1 + 1, index2).trim());
                continue;
            }
            if (!ResultsCommand.findExitCode(testResult, line)) continue;
            return;
        }
    }

    private static boolean findExitCode(TestResult testResult, String line) {
        Matcher matcher = TEST_EXIT_PATTERN.matcher(line);
        if (matcher.find()) {
            testResult.setExitCode(Integer.parseInt(matcher.group(1)));
            return true;
        }
        return false;
    }

    private void readTestU01(File resultFile, ListIterator<String> iter, TestResult testResult) {
        String testSuiteName = this.skipToTestU01Summary(resultFile, iter);
        if (testSuiteName == null) {
            while (iter.hasPrevious()) {
                iter.previous();
            }
            ResultsCommand.updateTestU01ApplicationName(iter, testResult);
            return;
        }
        ResultsCommand.setTestU01ApplicationName(testResult, testSuiteName);
        while (iter.hasNext()) {
            String line = iter.next();
            Matcher matcher = TESTU01_TEST_RESULT_PATTERN.matcher(line);
            if (matcher.find()) {
                testResult.addFailedTest(matcher.group(1).trim());
                continue;
            }
            if (!ResultsCommand.findExitCode(testResult, line)) continue;
            return;
        }
    }

    private static void setTestU01ApplicationName(TestResult testResult, String testSuiteName) {
        testResult.setTestApplicationName("TestU01 (" + testSuiteName + ")");
    }

    private String skipToTestU01Summary(File resultFile, Iterator<String> iter) {
        String testSuiteName = ResultsCommand.findMatcherGroup1(iter, TESTU01_SUMMARY_PATTERN);
        if (testSuiteName != null || this.ignorePartialResults) {
            return testSuiteName;
        }
        throw new ApplicationException("Failed to identify the Test U01 result summary: " + resultFile);
    }

    private static void updateTestU01ApplicationName(Iterator<String> iter, TestResult testResult) {
        String testSuiteName = ResultsCommand.findMatcherGroup1(iter, TESTU01_STARTING_PATTERN);
        if (testSuiteName != null) {
            ResultsCommand.setTestU01ApplicationName(testResult, testSuiteName);
        }
    }

    private static String findMatcherGroup1(Iterator<String> iter, Pattern pattern) {
        while (iter.hasNext()) {
            String line = iter.next();
            Matcher matcher = pattern.matcher(line);
            if (!matcher.find()) continue;
            return matcher.group(1);
        }
        return null;
    }

    private static void readPractRand(Iterator<String> iter, PractRandTestResult testResult) {
        testResult.setTestApplicationName("PractRand");
        int exp = 0;
        while (iter.hasNext()) {
            String line = iter.next();
            Matcher matcher = PRACTRAND_OUTPUT_SIZE_PATTERN.matcher(line);
            if (matcher.find()) {
                exp = Integer.parseInt(matcher.group(1));
                continue;
            }
            if (PRACTRAND_FAILED_PATTERN.matcher(line).find()) {
                testResult.setLengthExponent(exp);
                line = line.trim();
                int index = line.indexOf(32);
                testResult.addFailedTest(line.substring(0, index));
                continue;
            }
            if (!ResultsCommand.findExitCode(testResult, line)) continue;
            return;
        }
    }

    private static void deleteIfIncomplete(List<TestResult> results) {
        results.forEach(ResultsCommand::deleteIfIncomplete);
    }

    private static void deleteIfIncomplete(TestResult result) {
        if (!result.isComplete()) {
            try {
                Files.delete(result.getResultFile().toPath());
                LogUtils.info("Deleted file: %s", result.getResultFile());
            }
            catch (IOException ex) {
                throw new ApplicationException("Failed to delete file: " + result.getResultFile(), ex);
            }
        }
    }

    private OutputStream createOutputStream() {
        if (this.fileOutput != null) {
            try {
                Files.newOutputStream(this.fileOutput.toPath(), new OpenOption[0]);
            }
            catch (IOException ex) {
                throw new ApplicationException("Failed to create output: " + this.fileOutput, ex);
            }
        }
        return new FilterOutputStream(System.out){

            @Override
            public void close() {
            }
        };
    }

    private void writeCSVData(OutputStream out, List<TestResult> results) throws IOException {
        Collections.sort(results, (o1, o2) -> {
            int result = Integer.compare(o1.getRandomSource().ordinal(), o2.getRandomSource().ordinal());
            if (result != 0) {
                return result;
            }
            result = Boolean.compare(o1.isBitReversed(), o2.isBitReversed());
            if (result != 0) {
                return result;
            }
            result = o1.getTestApplicationName().compareTo(o2.getTestApplicationName());
            if (result != 0) {
                return result;
            }
            return Integer.compare(o1.getFailureCount(), o2.getFailureCount());
        });
        try (BufferedWriter output = new BufferedWriter(new OutputStreamWriter(out, StandardCharsets.UTF_8));){
            output.write("RandomSource,Bit-reversed,Test,Failures");
            if (this.showFailedTests) {
                output.write(",Failed");
            }
            output.newLine();
            for (TestResult result : results) {
                output.write(result.getRandomSource().toString());
                output.write(44);
                output.write(Boolean.toString(result.isBitReversed()));
                output.write(44);
                output.write(result.getTestApplicationName());
                output.write(44);
                output.write(result.getFailureSummaryString());
                if (this.showFailedTests) {
                    output.write(44);
                    output.write(result.getFailedTests().stream().collect(Collectors.joining("|")));
                }
                output.newLine();
            }
        }
    }

    private void writeAPT(OutputStream out, List<TestResult> results) throws IOException {
        List<RandomSource> randomSources = ResultsCommand.getRandomSources(results);
        List<Boolean> bitReversed = ResultsCommand.getBitReversed(results);
        List<String> testNames = ResultsCommand.getTestNames(results);
        int prefixLength = this.pathPrefix.isEmpty() ? 0 : ResultsCommand.findCommonPathPrefixLength(results);
        Function<TestResult, String> toAPTString = result -> {
            String path = result.getResultFile().getPath();
            path = path.substring(prefixLength);
            StringBuilder sb = new StringBuilder().append("{{{").append(this.pathPrefix).append(path).append('}').append(result.getFailureSummaryString()).append("}}");
            for (int i = 0; i < sb.length(); ++i) {
                if (sb.charAt(i) != '\\') continue;
                sb.setCharAt(i, '\\');
            }
            return sb.toString();
        };
        boolean showBitReversedColumn = bitReversed.contains(Boolean.TRUE);
        String header = ResultsCommand.createAPTHeader(showBitReversedColumn, testNames);
        String separator = ResultsCommand.createAPTSeparator(header);
        try (BufferedWriter output = new BufferedWriter(new OutputStreamWriter(out, StandardCharsets.UTF_8));){
            output.write(separator.replace('+', '*'));
            output.write(header);
            output.newLine();
            output.write(separator);
            StringBuilder sb = new StringBuilder();
            for (RandomSource randomSource : randomSources) {
                for (boolean reversed : bitReversed) {
                    boolean highlight = true;
                    sb.setLength(0);
                    if (showBitReversedColumn) {
                        ResultsCommand.writeAPTColumn(sb, Boolean.toString(reversed), false);
                    }
                    for (String testName : testNames) {
                        List<TestResult> testResults = ResultsCommand.getTestResults(results, randomSource, reversed, testName);
                        String text = testResults.stream().map(toAPTString).collect(Collectors.joining(", "));
                        String summary = ResultsCommand.getFailuresSummary(testResults);
                        if (summary.length() != 0) {
                            highlight = false;
                            if (this.showFailedTests) {
                                text = text + " (" + summary + ")";
                            }
                        }
                        ResultsCommand.writeAPTColumn(sb, text, false);
                    }
                    output.write(124);
                    ResultsCommand.writeAPTColumn(output, randomSource.toString(), highlight);
                    output.write(sb.toString());
                    output.newLine();
                    output.write(separator);
                }
            }
        }
    }

    private static List<RandomSource> getRandomSources(List<TestResult> results) {
        EnumSet<RandomSource> set = EnumSet.noneOf(RandomSource.class);
        for (TestResult result : results) {
            set.add(result.getRandomSource());
        }
        ArrayList<RandomSource> list = new ArrayList<RandomSource>(set);
        Collections.sort(list);
        return list;
    }

    private static List<Boolean> getBitReversed(List<TestResult> results) {
        ArrayList<Boolean> list = new ArrayList<Boolean>(2);
        if (results.isEmpty()) {
            list.add(Boolean.FALSE);
        } else {
            boolean first = results.get(0).isBitReversed();
            list.add(first);
            for (TestResult result : results) {
                if (first == result.isBitReversed()) continue;
                list.add(!first);
                break;
            }
        }
        Collections.sort(list);
        return list;
    }

    private static List<String> getTestNames(List<TestResult> results) {
        LinkedHashSet<String> set = new LinkedHashSet<String>();
        for (TestResult result : results) {
            set.add(result.getTestApplicationName());
        }
        return new ArrayList<String>(set);
    }

    private static int findCommonPathPrefixLength(List<TestResult> results) {
        if (results.isEmpty()) {
            return 0;
        }
        String prefix1 = ResultsCommand.getPathPrefix(results.get(0));
        int length = prefix1.length();
        for (int i = 1; i < results.size() && length != 0; ++i) {
            String prefix2 = ResultsCommand.getPathPrefix(results.get(i));
            int size = Math.min(prefix2.length(), length);
            for (length = 0; length < size && prefix1.charAt(length) == prefix2.charAt(length); ++length) {
            }
        }
        return length;
    }

    private static String getPathPrefix(TestResult testResult) {
        String parent = testResult.getResultFile().getParent();
        return parent == null ? "" : parent;
    }

    private static String createAPTHeader(boolean showBitReversedColumn, List<String> testNames) {
        StringBuilder sb = new StringBuilder(100).append("|| RNG identifier ||");
        if (showBitReversedColumn) {
            sb.append(" Bit-reversed ||");
        }
        for (String name : testNames) {
            sb.append(' ').append(name).append(" ||");
        }
        return sb.toString();
    }

    private static String createAPTSeparator(String header) {
        StringBuilder sb = new StringBuilder(header);
        for (int i = 0; i < header.length(); ++i) {
            if (sb.charAt(i) == '|') {
                sb.setCharAt(i, i == 0 ? (char)'*' : '+');
                sb.setCharAt(i + 1, '-');
                continue;
            }
            sb.setCharAt(i, '-');
        }
        sb.setCharAt(header.length() - 2, '-');
        sb.setCharAt(header.length() - 1, '+');
        sb.append(System.lineSeparator());
        return sb.toString();
    }

    private static void writeAPTColumn(Appendable output, String text, boolean highlight) throws IOException {
        output.append(' ');
        if (highlight) {
            output.append("<<");
        }
        output.append(text);
        if (highlight) {
            output.append(">>");
        }
        output.append(" |");
    }

    private static List<TestResult> getTestResults(List<TestResult> results, RandomSource randomSource, boolean bitReversed, String testName) {
        ArrayList<TestResult> list = new ArrayList<TestResult>();
        for (TestResult result : results) {
            if (result.getRandomSource() != randomSource || result.bitReversed != bitReversed || !result.getTestApplicationName().equals(testName)) continue;
            list.add(result);
        }
        return list;
    }

    private static List<String> getSystematicFailures(List<TestResult> results) {
        HashMap<String, Integer> map = new HashMap<String, Integer>();
        for (TestResult result : results) {
            if (!result.isComplete()) continue;
            HashSet<String> unique = new HashSet<String>(result.getFailedTests());
            for (String test : unique) {
                map.merge(test, 1, (i, j) -> i + j);
            }
        }
        int completeCount = (int)results.stream().filter(TestResult::isComplete).count();
        List list = map.entrySet().stream().filter(e -> (Integer)e.getValue() == completeCount).map(Map.Entry::getKey).collect(Collectors.toCollection(ArrayList::new));
        int max = ResultsCommand.getMaxLengthExponent(results);
        if (max != 0) {
            list.add(ResultsCommand.bytesToString(max));
        }
        return list;
    }

    private static int getMaxLengthExponent(List<TestResult> results) {
        if (results.isEmpty()) {
            return 0;
        }
        int[] data = new int[2];
        results.stream().filter(TestResult::isComplete).filter(r -> r instanceof PractRandTestResult).mapToInt(r -> ((PractRandTestResult)r).getLengthExponent()).forEach(i -> {
            if (i == 0) {
                data[0] = data[0] + 1;
            } else {
                data[1] = Math.max(i, data[1]);
            }
        });
        return data[0] == 0 ? data[1] : 0;
    }

    private static String getFailuresSummary(List<TestResult> results) {
        if (results.isEmpty()) {
            return "";
        }
        if (results.get(0).getTestFormat() == TestFormat.PRACTRAND) {
            int max = ResultsCommand.getMaxLengthExponent(results);
            return max == 0 ? "" : ResultsCommand.bytesToString(max);
        }
        int count = ResultsCommand.getSystematicFailures(results).size();
        return count == 0 ? "" : Integer.toString(count);
    }

    private static void writeTXT(OutputStream out, List<TestResult> results) throws IOException {
        List<RandomSource> randomSources = ResultsCommand.getRandomSources(results);
        List<Boolean> bitReversed = ResultsCommand.getBitReversed(results);
        List<String> testNames = ResultsCommand.getTestNames(results);
        boolean showBitReversedColumn = bitReversed.contains(Boolean.TRUE);
        List<List<String>> columns = ResultsCommand.createTXTColumns(testNames, showBitReversedColumn);
        for (RandomSource randomSource : randomSources) {
            for (boolean reversed : bitReversed) {
                int i = 0;
                columns.get(i++).add(randomSource.toString());
                if (showBitReversedColumn) {
                    columns.get(i++).add(Boolean.toString(reversed));
                }
                for (String testName : testNames) {
                    List<TestResult> testResults = ResultsCommand.getTestResults(results, randomSource, reversed, testName);
                    columns.get(i++).add(testResults.stream().map(TestResult::getFailureSummaryString).collect(Collectors.joining(",")));
                    columns.get(i++).add(ResultsCommand.getFailuresSummary(testResults));
                }
            }
        }
        ResultsCommand.writeColumns(out, columns);
    }

    private static List<List<String>> createTXTColumns(List<String> testNames, boolean showBitReversedColumn) {
        ArrayList<List<String>> columns = new ArrayList<List<String>>();
        columns.add(ResultsCommand.createColumn(COLUMN_RNG));
        if (showBitReversedColumn) {
            columns.add(ResultsCommand.createColumn(BIT_REVERSED));
        }
        for (String testName : testNames) {
            columns.add(ResultsCommand.createColumn(testName));
            columns.add(ResultsCommand.createColumn("\u2229"));
        }
        return columns;
    }

    private static List<String> createColumn(String columnName) {
        ArrayList<String> list = new ArrayList<String>();
        list.add(columnName);
        return list;
    }

    private static String createTextFormatFromColumnWidths(List<List<String>> columns) {
        StringBuilder sb = new StringBuilder();
        try (Formatter formatter = new Formatter(sb);){
            for (int i = 0; i < columns.size(); ++i) {
                if (i != 0) {
                    sb.append('\t');
                }
                formatter.format("%%-%ds", ResultsCommand.getColumnWidth(columns.get(i)));
            }
        }
        sb.append(System.lineSeparator());
        return sb.toString();
    }

    private static int getColumnWidth(List<String> column) {
        int width = 0;
        for (String text : column) {
            width = Math.max(width, text.length());
        }
        return width;
    }

    private static void writeColumns(OutputStream out, List<List<String>> columns) throws IOException {
        String format = ResultsCommand.createTextFormatFromColumnWidths(columns);
        try (BufferedWriter output = new BufferedWriter(new OutputStreamWriter(out, StandardCharsets.UTF_8));
             Formatter formatter = new Formatter(output);){
            int rows = columns.get(0).size();
            Object[] args = new Object[columns.size()];
            for (int row = 0; row < rows; ++row) {
                for (int i = 0; i < args.length; ++i) {
                    args[i] = columns.get(i).get(row);
                }
                formatter.format(format, args);
            }
        }
    }

    private static void writeFailures(OutputStream out, List<TestResult> results) throws IOException {
        List<RandomSource> randomSources = ResultsCommand.getRandomSources(results);
        List<Boolean> bitReversed = ResultsCommand.getBitReversed(results);
        List<String> testNames = ResultsCommand.getTestNames(results);
        boolean showBitReversedColumn = bitReversed.contains(Boolean.TRUE);
        List<List<String>> columns = ResultsCommand.createFailuresColumns(testNames, showBitReversedColumn);
        AlphaNumericComparator cmp = new AlphaNumericComparator();
        for (RandomSource randomSource : randomSources) {
            for (boolean reversed : bitReversed) {
                for (String testName : testNames) {
                    List<TestResult> testResults = ResultsCommand.getTestResults(results, randomSource, reversed, testName);
                    List<String> failures = ResultsCommand.getSystematicFailures(testResults);
                    if (failures.isEmpty()) continue;
                    Collections.sort(failures, cmp);
                    for (String failed : failures) {
                        int i = 0;
                        columns.get(i++).add(randomSource.toString());
                        if (showBitReversedColumn) {
                            columns.get(i++).add(Boolean.toString(reversed));
                        }
                        columns.get(i++).add(testName);
                        columns.get(i).add(failed);
                    }
                }
            }
        }
        ResultsCommand.writeColumns(out, columns);
    }

    private static List<List<String>> createFailuresColumns(List<String> testNames, boolean showBitReversedColumn) {
        ArrayList<List<String>> columns = new ArrayList<List<String>>();
        columns.add(ResultsCommand.createColumn(COLUMN_RNG));
        if (showBitReversedColumn) {
            columns.add(ResultsCommand.createColumn(BIT_REVERSED));
        }
        columns.add(ResultsCommand.createColumn("Test Suite"));
        columns.add(ResultsCommand.createColumn("Test"));
        return columns;
    }

    static String bytesToString(int exponent) {
        int unit = exponent / 10;
        int size = 1 << exponent - 10 * unit;
        return size + BINARY_UNITS[unit];
    }

    private static class PractRandTestResult
    extends TestResult {
        private int lengthExponent;

        PractRandTestResult(File resultFile, RandomSource randomSource, boolean bitReversed, TestFormat testFormat) {
            super(resultFile, randomSource, bitReversed, testFormat);
        }

        int getLengthExponent() {
            return this.lengthExponent;
        }

        void setLengthExponent(int lengthExponent) {
            this.lengthExponent = lengthExponent;
        }

        @Override
        String getFailureSummaryString() {
            return this.lengthExponent == 0 ? "-" : Integer.toString(this.lengthExponent);
        }
    }

    private static class TestResult {
        private final File resultFile;
        private final RandomSource randomSource;
        private final boolean bitReversed;
        private final TestFormat testFormat;
        private final List<String> failedTests = new ArrayList<String>();
        private String testApplicationName;
        private int exitCode = Integer.MIN_VALUE;

        TestResult(File resultFile, RandomSource randomSource, boolean bitReversed, TestFormat testFormat) {
            this.resultFile = resultFile;
            this.randomSource = randomSource;
            this.bitReversed = bitReversed;
            this.testFormat = testFormat;
        }

        void addFailedTest(String testId) {
            this.failedTests.add(testId);
        }

        File getResultFile() {
            return this.resultFile;
        }

        RandomSource getRandomSource() {
            return this.randomSource;
        }

        boolean isBitReversed() {
            return this.bitReversed;
        }

        TestFormat getTestFormat() {
            return this.testFormat;
        }

        List<String> getFailedTests() {
            return this.failedTests;
        }

        int getFailureCount() {
            return this.failedTests.size();
        }

        String getFailureSummaryString() {
            return this.isComplete() ? Integer.toString(this.failedTests.size()) : "-" + this.failedTests.size();
        }

        void setTestApplicationName(String testApplicationName) {
            this.testApplicationName = testApplicationName;
        }

        String getTestApplicationName() {
            return this.testApplicationName == null ? String.valueOf((Object)this.getTestFormat()) : this.testApplicationName;
        }

        boolean isComplete() {
            return this.exitCode == 0;
        }

        void setExitCode(int exitCode) {
            this.exitCode = exitCode;
        }
    }

    static enum TestFormat {
        DIEHARDER,
        TESTU01,
        PRACTRAND;

    }

    static enum OutputFormat {
        CSV,
        APT,
        TXT,
        FAILURES;

    }
}

