/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.schema;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.Futures;
import java.lang.management.ManagementFactory;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Deque;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.LongSupplier;
import org.apache.cassandra.concurrent.ScheduledExecutors;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.exceptions.RequestFailureReason;
import org.apache.cassandra.gms.ApplicationState;
import org.apache.cassandra.gms.EndpointState;
import org.apache.cassandra.gms.FailureDetector;
import org.apache.cassandra.gms.Gossiper;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.net.Message;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.net.NoPayload;
import org.apache.cassandra.net.RequestCallback;
import org.apache.cassandra.net.Verb;
import org.apache.cassandra.schema.Schema;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.concurrent.WaitQueue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MigrationCoordinator {
    private static final Logger logger = LoggerFactory.getLogger(MigrationCoordinator.class);
    private static final Future<Void> FINISHED_FUTURE = Futures.immediateFuture(null);
    private static LongSupplier getUptimeFn = () -> ManagementFactory.getRuntimeMXBean().getUptime();
    private static final int MIGRATION_DELAY_IN_MS = 60000;
    private static final int MAX_OUTSTANDING_VERSION_REQUESTS = 3;
    public static final MigrationCoordinator instance = new MigrationCoordinator();
    private final Map<UUID, VersionInfo> versionInfo = new HashMap<UUID, VersionInfo>();
    private final Map<InetAddressAndPort, UUID> endpointVersions = new HashMap<InetAddressAndPort, UUID>();
    private final AtomicInteger inflightTasks = new AtomicInteger();

    @VisibleForTesting
    public static void setUptimeFn(LongSupplier supplier) {
        getUptimeFn = supplier;
    }

    public void start() {
        ScheduledExecutors.scheduledTasks.scheduleWithFixedDelay(this::pullUnreceivedSchemaVersions, 1L, 1L, TimeUnit.MINUTES);
    }

    public synchronized void reset() {
        this.versionInfo.clear();
    }

    synchronized List<Future<Void>> pullUnreceivedSchemaVersions() {
        ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>();
        for (VersionInfo info : this.versionInfo.values()) {
            Future<Void> future;
            if (info.wasReceived() || info.outstandingRequests.size() > 0 || (future = this.maybePullSchema(info)) == null || future == FINISHED_FUTURE) continue;
            futures.add(future);
        }
        return futures;
    }

    synchronized Future<Void> maybePullSchema(VersionInfo info) {
        if (info.endpoints.isEmpty() || info.wasReceived() || !this.shouldPullSchema(info.version)) {
            return FINISHED_FUTURE;
        }
        if (info.outstandingRequests.size() >= this.getMaxOutstandingVersionRequests()) {
            return FINISHED_FUTURE;
        }
        int isize = info.requestQueue.size();
        for (int i = 0; i < isize; ++i) {
            InetAddressAndPort endpoint = info.requestQueue.remove();
            if (!info.endpoints.contains(endpoint)) continue;
            if (this.shouldPullFromEndpoint(endpoint) && info.outstandingRequests.add(endpoint)) {
                return this.scheduleSchemaPull(endpoint, info);
            }
            info.requestQueue.offer(endpoint);
        }
        return null;
    }

    public synchronized Map<UUID, Set<InetAddressAndPort>> outstandingVersions() {
        HashMap<UUID, Set<InetAddressAndPort>> map = new HashMap<UUID, Set<InetAddressAndPort>>();
        for (VersionInfo info : this.versionInfo.values()) {
            if (info.wasReceived()) continue;
            map.put(info.version, (Set<InetAddressAndPort>)ImmutableSet.copyOf(info.endpoints));
        }
        return map;
    }

    @VisibleForTesting
    protected VersionInfo getVersionInfoUnsafe(UUID version) {
        return this.versionInfo.get(version);
    }

    @VisibleForTesting
    protected int getMaxOutstandingVersionRequests() {
        return 3;
    }

    @VisibleForTesting
    protected boolean isAlive(InetAddressAndPort endpoint) {
        return FailureDetector.instance.isAlive(endpoint);
    }

    @VisibleForTesting
    protected boolean shouldPullSchema(UUID version) {
        if (Schema.instance.getVersion() == null) {
            logger.debug("Not pulling schema for version {}, because local schama version is not known yet", (Object)version);
            return false;
        }
        if (Schema.instance.isSameVersion(version)) {
            logger.debug("Not pulling schema for version {}, because schema versions match: local={}, remote={}", new Object[]{version, Schema.schemaVersionToString(Schema.instance.getVersion()), Schema.schemaVersionToString(version)});
            return false;
        }
        return true;
    }

    private static boolean is30Compatible(int version) {
        return version == 12 || version == 11;
    }

    @VisibleForTesting
    protected boolean shouldPullFromEndpoint(InetAddressAndPort endpoint) {
        if (endpoint.equals(FBUtilities.getBroadcastAddressAndPort())) {
            return false;
        }
        EndpointState state = Gossiper.instance.getEndpointStateForEndpoint(endpoint);
        if (state == null) {
            return false;
        }
        String releaseVersion = state.getApplicationState((ApplicationState)ApplicationState.RELEASE_VERSION).value;
        String ourMajorVersion = FBUtilities.getReleaseVersionMajor();
        if (!releaseVersion.startsWith(ourMajorVersion)) {
            logger.debug("Not pulling schema from {} because release version in Gossip is not major version {}, it is {}", new Object[]{endpoint, ourMajorVersion, releaseVersion});
            return false;
        }
        if (!MessagingService.instance().versions.knows(endpoint)) {
            logger.debug("Not pulling schema from {} because their messaging version is unknown", (Object)endpoint);
            return false;
        }
        if (MessagingService.instance().versions.getRaw(endpoint) != 12) {
            logger.debug("Not pulling schema from {} because their schema format is incompatible", (Object)endpoint);
            return false;
        }
        if (Gossiper.instance.isGossipOnlyMember(endpoint)) {
            logger.debug("Not pulling schema from {} because it's a gossip only member", (Object)endpoint);
            return false;
        }
        return true;
    }

    @VisibleForTesting
    protected boolean shouldPullImmediately(InetAddressAndPort endpoint, UUID version) {
        if (Schema.instance.isEmpty() || getUptimeFn.getAsLong() < 60000L) {
            logger.debug("Immediately submitting migration task for {}, schema versions: local={}, remote={}", new Object[]{endpoint, Schema.schemaVersionToString(Schema.instance.getVersion()), Schema.schemaVersionToString(version)});
            return true;
        }
        return false;
    }

    @VisibleForTesting
    protected boolean isLocalVersion(UUID version) {
        return Schema.instance.isSameVersion(version);
    }

    synchronized boolean shouldApplySchemaFor(VersionInfo info) {
        if (info.wasReceived()) {
            return false;
        }
        return !this.isLocalVersion(info.version);
    }

    public synchronized Future<Void> reportEndpointVersion(InetAddressAndPort endpoint, UUID version) {
        UUID current = this.endpointVersions.put(endpoint, version);
        if (current != null && current.equals(version)) {
            return FINISHED_FUTURE;
        }
        VersionInfo info = this.versionInfo.computeIfAbsent(version, VersionInfo::new);
        if (this.isLocalVersion(version)) {
            info.markReceived();
        }
        info.endpoints.add(endpoint);
        info.requestQueue.addFirst(endpoint);
        this.removeEndpointFromVersion(endpoint, current);
        return this.maybePullSchema(info);
    }

    public Future<Void> reportEndpointVersion(InetAddressAndPort endpoint, EndpointState state) {
        if (state == null) {
            return FINISHED_FUTURE;
        }
        UUID version = state.getSchemaVersion();
        if (version == null) {
            return FINISHED_FUTURE;
        }
        return this.reportEndpointVersion(endpoint, version);
    }

    private synchronized void removeEndpointFromVersion(InetAddressAndPort endpoint, UUID version) {
        if (version == null) {
            return;
        }
        VersionInfo info = this.versionInfo.get(version);
        if (info == null) {
            return;
        }
        info.endpoints.remove(endpoint);
        if (info.endpoints.isEmpty()) {
            info.waitQueue.signalAll();
            this.versionInfo.remove(version);
        }
    }

    Future<Void> scheduleSchemaPull(InetAddressAndPort endpoint, VersionInfo info) {
        FutureTask<Object> task = new FutureTask<Object>(() -> this.pullSchema(new Callback(endpoint, info)), null);
        if (this.shouldPullImmediately(endpoint, info.version)) {
            Stage.MIGRATION.submit(task);
        } else {
            ScheduledExecutors.nonPeriodicTasks.schedule(() -> Stage.MIGRATION.submit(task), 60000L, TimeUnit.MILLISECONDS);
        }
        return task;
    }

    @VisibleForTesting
    protected void mergeSchemaFrom(InetAddressAndPort endpoint, Collection<Mutation> mutations) {
        Schema.instance.mergeAndAnnounceVersion(mutations);
    }

    private void pullSchema(Callback callback) {
        if (!this.isAlive(callback.endpoint)) {
            logger.warn("Can't send schema pull request: node {} is down.", (Object)callback.endpoint);
            callback.fail();
            return;
        }
        if (!this.shouldPullFromEndpoint(callback.endpoint)) {
            logger.info("Skipped sending a migration request: node {} has a higher major version now.", (Object)callback.endpoint);
            callback.fail();
            return;
        }
        logger.debug("Requesting schema from {}", (Object)callback.endpoint);
        this.sendMigrationMessage(callback);
    }

    protected void sendMigrationMessage(Callback callback) {
        this.inflightTasks.getAndIncrement();
        Message<NoPayload> message = Message.out(Verb.SCHEMA_PULL_REQ, NoPayload.noPayload);
        logger.info("Sending schema pull request to {}", (Object)callback.endpoint);
        MessagingService.instance().sendWithCallback(message, callback.endpoint, callback);
    }

    private synchronized Future<Void> pullComplete(InetAddressAndPort endpoint, VersionInfo info, boolean wasSuccessful) {
        this.inflightTasks.decrementAndGet();
        if (wasSuccessful) {
            info.markReceived();
        }
        info.outstandingRequests.remove(endpoint);
        info.requestQueue.add(endpoint);
        return this.maybePullSchema(info);
    }

    public int getInflightTasks() {
        return this.inflightTasks.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean awaitSchemaRequests(long waitMillis) {
        if (!FBUtilities.getBroadcastAddressAndPort().equals(InetAddressAndPort.getLoopbackAddress())) {
            Gossiper.waitToSettle();
        }
        WaitQueue.Signal signal = null;
        try {
            MigrationCoordinator migrationCoordinator = this;
            synchronized (migrationCoordinator) {
                ArrayList<WaitQueue.Signal> signalList;
                block13: {
                    signalList = new ArrayList<WaitQueue.Signal>(this.versionInfo.size());
                    for (VersionInfo version : this.versionInfo.values()) {
                        if (version.wasReceived()) continue;
                        signalList.add(version.register());
                    }
                    if (!signalList.isEmpty()) break block13;
                    boolean bl = true;
                    return bl;
                }
                WaitQueue.Signal[] signals = new WaitQueue.Signal[signalList.size()];
                signalList.toArray(signals);
                signal = WaitQueue.all(signals);
            }
            boolean bl = signal.awaitUntil(System.nanoTime() + TimeUnit.MILLISECONDS.toNanos(waitMillis));
            return bl;
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        finally {
            if (signal != null) {
                signal.cancel();
            }
        }
    }

    class Callback
    implements RequestCallback<Collection<Mutation>> {
        final InetAddressAndPort endpoint;
        final VersionInfo info;

        public Callback(InetAddressAndPort endpoint, VersionInfo info) {
            this.endpoint = endpoint;
            this.info = info;
        }

        @Override
        public void onFailure(InetAddressAndPort from, RequestFailureReason failureReason) {
            this.fail();
        }

        Future<Void> fail() {
            return MigrationCoordinator.this.pullComplete(this.endpoint, this.info, false);
        }

        @Override
        public void onResponse(Message<Collection<Mutation>> message) {
            this.response((Collection)message.payload);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        Future<Void> response(Collection<Mutation> mutations) {
            VersionInfo versionInfo = this.info;
            synchronized (versionInfo) {
                if (MigrationCoordinator.this.shouldApplySchemaFor(this.info)) {
                    try {
                        MigrationCoordinator.this.mergeSchemaFrom(this.endpoint, mutations);
                    }
                    catch (Exception e) {
                        logger.error(String.format("Unable to merge schema from %s", this.endpoint), (Throwable)e);
                        return this.fail();
                    }
                }
                return MigrationCoordinator.this.pullComplete(this.endpoint, this.info, true);
            }
        }

        public boolean isLatencyForSnitch() {
            return false;
        }
    }

    static class VersionInfo {
        final UUID version;
        final Set<InetAddressAndPort> endpoints = Sets.newConcurrentHashSet();
        final Set<InetAddressAndPort> outstandingRequests = Sets.newConcurrentHashSet();
        final Deque<InetAddressAndPort> requestQueue = new ArrayDeque<InetAddressAndPort>();
        private final WaitQueue waitQueue = new WaitQueue();
        volatile boolean receivedSchema;

        VersionInfo(UUID version) {
            this.version = version;
        }

        WaitQueue.Signal register() {
            return this.waitQueue.register();
        }

        void markReceived() {
            if (this.receivedSchema) {
                return;
            }
            this.receivedSchema = true;
            this.waitQueue.signalAll();
        }

        boolean wasReceived() {
            return this.receivedSchema;
        }
    }
}

