/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.codahale.metrics.Timer;
import com.google.common.annotations.VisibleForTesting;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.LockSupport;
import org.apache.cassandra.concurrent.NamedThreadFactory;
import org.apache.cassandra.db.commitlog.CommitLog;
import org.apache.cassandra.db.commitlog.CommitLogSegment;
import org.apache.cassandra.utils.MonotonicClock;
import org.apache.cassandra.utils.NoSpamLogger;
import org.apache.cassandra.utils.concurrent.WaitQueue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractCommitLogService {
    static final long DEFAULT_MARKER_INTERVAL_MILLIS = 100L;
    private volatile Thread thread;
    private volatile boolean shutdown = false;
    protected volatile long lastSyncedAt = System.currentTimeMillis();
    private final AtomicLong written = new AtomicLong(0L);
    protected final AtomicLong pending = new AtomicLong(0L);
    protected final WaitQueue syncComplete = new WaitQueue();
    final CommitLog commitLog;
    private final String name;
    final long syncIntervalNanos;
    final long markerIntervalNanos;
    private volatile boolean syncRequested;
    private static final Logger logger = LoggerFactory.getLogger(AbstractCommitLogService.class);

    AbstractCommitLogService(CommitLog commitLog, String name, long syncIntervalMillis) {
        this(commitLog, name, syncIntervalMillis, false);
    }

    AbstractCommitLogService(CommitLog commitLog, String name, long syncIntervalMillis, boolean markHeadersFaster) {
        long markerIntervalMillis;
        this.commitLog = commitLog;
        this.name = name;
        if (markHeadersFaster && syncIntervalMillis > 100L) {
            markerIntervalMillis = 100L;
            long modulo = syncIntervalMillis % markerIntervalMillis;
            if (modulo != 0L) {
                syncIntervalMillis -= modulo;
                if (modulo >= markerIntervalMillis / 2L) {
                    syncIntervalMillis += markerIntervalMillis;
                }
            }
            logger.debug("Will update the commitlog markers every {}ms and flush every {}ms", (Object)markerIntervalMillis, (Object)syncIntervalMillis);
        } else {
            markerIntervalMillis = syncIntervalMillis;
        }
        assert (syncIntervalMillis % markerIntervalMillis == 0L);
        this.markerIntervalNanos = TimeUnit.NANOSECONDS.convert(markerIntervalMillis, TimeUnit.MILLISECONDS);
        this.syncIntervalNanos = TimeUnit.NANOSECONDS.convert(syncIntervalMillis, TimeUnit.MILLISECONDS);
    }

    void start() {
        if (this.syncIntervalNanos < 1L) {
            throw new IllegalArgumentException(String.format("Commit log flush interval must be positive: %fms", (double)this.syncIntervalNanos * 1.0E-6));
        }
        this.shutdown = false;
        this.thread = NamedThreadFactory.createThread(new SyncRunnable(MonotonicClock.preciseTime), this.name);
        this.thread.start();
    }

    public void finishWriteFor(CommitLogSegment.Allocation alloc) {
        this.maybeWaitForSync(alloc);
        this.written.incrementAndGet();
    }

    protected abstract void maybeWaitForSync(CommitLogSegment.Allocation var1);

    void requestExtraSync() {
        this.syncRequested = true;
        LockSupport.unpark(this.thread);
    }

    public void shutdown() {
        this.shutdown = true;
        this.requestExtraSync();
    }

    public void syncBlocking() {
        long requestTime = System.nanoTime();
        this.requestExtraSync();
        this.awaitSyncAt(requestTime, null);
    }

    void awaitSyncAt(long syncTime, Timer.Context context) {
        do {
            WaitQueue.Signal signal;
            WaitQueue.Signal signal2 = signal = context != null ? this.syncComplete.register(context) : this.syncComplete.register();
            if (this.lastSyncedAt < syncTime) {
                signal.awaitUninterruptibly();
                continue;
            }
            signal.cancel();
        } while (this.lastSyncedAt < syncTime);
    }

    public void awaitTermination() throws InterruptedException {
        if (this.thread != null) {
            this.thread.join();
        }
    }

    public long getCompletedTasks() {
        return this.written.get();
    }

    public long getPendingTasks() {
        return this.pending.get();
    }

    class SyncRunnable
    implements Runnable {
        private final MonotonicClock clock;
        private long firstLagAt = 0L;
        private long totalSyncDuration = 0L;
        private long syncExceededIntervalBy = 0L;
        private int lagCount = 0;
        private int syncCount = 0;

        SyncRunnable(MonotonicClock clock) {
            this.clock = clock;
        }

        @Override
        public void run() {
            while (this.sync()) {
            }
        }

        boolean sync() {
            boolean shutdownRequested = AbstractCommitLogService.this.shutdown;
            try {
                boolean flushToDisk;
                long pollStarted = this.clock.now();
                boolean bl = flushToDisk = AbstractCommitLogService.this.lastSyncedAt + AbstractCommitLogService.this.syncIntervalNanos <= pollStarted || shutdownRequested || AbstractCommitLogService.this.syncRequested;
                if (flushToDisk) {
                    AbstractCommitLogService.this.syncRequested = false;
                    AbstractCommitLogService.this.commitLog.sync(true);
                    AbstractCommitLogService.this.lastSyncedAt = pollStarted;
                    AbstractCommitLogService.this.syncComplete.signalAll();
                    ++this.syncCount;
                } else {
                    AbstractCommitLogService.this.commitLog.sync(false);
                }
                long now = this.clock.now();
                if (flushToDisk) {
                    this.maybeLogFlushLag(pollStarted, now);
                }
                if (shutdownRequested) {
                    return false;
                }
                long wakeUpAt = pollStarted + AbstractCommitLogService.this.markerIntervalNanos;
                if (wakeUpAt > now) {
                    LockSupport.parkNanos(wakeUpAt - now);
                }
            }
            catch (Throwable t) {
                if (!CommitLog.handleCommitError("Failed to persist commits to disk", t)) {
                    return false;
                }
                LockSupport.parkNanos(AbstractCommitLogService.this.markerIntervalNanos);
            }
            return true;
        }

        @VisibleForTesting
        boolean maybeLogFlushLag(long pollStarted, long now) {
            boolean logged;
            long flushDuration = now - pollStarted;
            this.totalSyncDuration += flushDuration;
            long maxFlushTimestamp = pollStarted + AbstractCommitLogService.this.syncIntervalNanos;
            if (maxFlushTimestamp > now) {
                return false;
            }
            if (this.firstLagAt == 0L) {
                this.firstLagAt = now;
                this.lagCount = 0;
                this.syncExceededIntervalBy = 0;
                this.syncCount = 1;
                this.totalSyncDuration = flushDuration;
            }
            this.syncExceededIntervalBy += now - maxFlushTimestamp;
            ++this.lagCount;
            if (this.firstLagAt > 0L && (logged = NoSpamLogger.log(logger, NoSpamLogger.Level.WARN, 5L, TimeUnit.MINUTES, "Out of {} commit log syncs over the past {}s with average duration of {}ms, {} have exceeded the configured commit interval by an average of {}ms", this.syncCount, String.format("%.2f", (double)(now - this.firstLagAt) * 1.0E-9), String.format("%.2f", (double)this.totalSyncDuration * 1.0E-6 / (double)this.syncCount), this.lagCount, String.format("%.2f", (double)this.syncExceededIntervalBy * 1.0E-6 / (double)this.lagCount)))) {
                this.firstLagAt = 0L;
            }
            return true;
        }

        @VisibleForTesting
        long getTotalSyncDuration() {
            return this.totalSyncDuration;
        }
    }
}

