/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.utils.memory;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import io.netty.util.concurrent.FastThreadLocal;
import java.lang.ref.PhantomReference;
import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayDeque;
import java.util.Collections;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicLongFieldUpdater;
import java.util.function.BiPredicate;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.apache.cassandra.concurrent.InfiniteLoopExecutor;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.io.compress.BufferType;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.metrics.BufferPoolMetrics;
import org.apache.cassandra.utils.ExecutorUtils;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.NoSpamLogger;
import org.apache.cassandra.utils.concurrent.Ref;
import org.apache.cassandra.utils.memory.MemoryUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BufferPool {
    public static final int NORMAL_CHUNK_SIZE = 131072;
    public static final int NORMAL_ALLOCATION_UNIT = 2048;
    public static final int TINY_CHUNK_SIZE = 2048;
    public static final int TINY_ALLOCATION_UNIT = 32;
    public static final int TINY_ALLOCATION_LIMIT = 1024;
    private static final BufferPoolMetrics metrics = new BufferPoolMetrics();
    @VisibleForTesting
    public static long MEMORY_USAGE_THRESHOLD = (long)DatabaseDescriptor.getFileCacheSizeInMB() * 1024L * 1024L;
    private static Debug debug;
    private static final Logger logger;
    private static final NoSpamLogger noSpamLogger;
    private static final ByteBuffer EMPTY_BUFFER;
    private static final GlobalPool globalPool;
    private static final FastThreadLocal<LocalPool> localPool;
    private static final Set<LocalPoolRef> localPoolReferences;
    private static final ReferenceQueue<Object> localPoolRefQueue;
    private static final InfiniteLoopExecutor EXEC;

    public static ByteBuffer get(int size, BufferType bufferType) {
        if (bufferType == BufferType.ON_HEAP) {
            return BufferPool.allocate(size, bufferType);
        }
        return ((LocalPool)localPool.get()).get(size);
    }

    public static ByteBuffer getAtLeast(int size, BufferType bufferType) {
        if (bufferType == BufferType.ON_HEAP) {
            return BufferPool.allocate(size, bufferType);
        }
        return ((LocalPool)localPool.get()).getAtLeast(size);
    }

    public static ByteBuffer tryGet(int size) {
        return ((LocalPool)BufferPool.localPool.get()).tryGet(size, false);
    }

    public static ByteBuffer tryGetAtLeast(int size) {
        return ((LocalPool)BufferPool.localPool.get()).tryGet(size, true);
    }

    private static ByteBuffer allocate(int size, BufferType bufferType) {
        return bufferType == BufferType.ON_HEAP ? ByteBuffer.allocate(size) : ByteBuffer.allocateDirect(size);
    }

    public static void put(ByteBuffer buffer) {
        if (MemoryUtil.isExactlyDirect(buffer)) {
            ((LocalPool)localPool.get()).put(buffer);
        }
    }

    public static void putUnusedPortion(ByteBuffer buffer) {
        if (MemoryUtil.isExactlyDirect(buffer)) {
            LocalPool pool = (LocalPool)localPool.get();
            if (buffer.limit() > 0) {
                pool.putUnusedPortion(buffer);
            } else {
                pool.put(buffer);
            }
        }
    }

    public static void setRecycleWhenFreeForCurrentThread(boolean recycleWhenFree) {
        ((LocalPool)localPool.get()).recycleWhenFree(recycleWhenFree);
    }

    public static long sizeInBytes() {
        return globalPool.sizeInBytes();
    }

    public static void debug(Debug setDebug) {
        debug = setDebug;
    }

    private static void cleanupOneReference() throws InterruptedException {
        Reference<Object> obj = localPoolRefQueue.remove(100L);
        if (obj instanceof LocalPoolRef) {
            ((LocalPoolRef)obj).release();
            localPoolReferences.remove(obj);
        }
    }

    private static ByteBuffer allocateDirectAligned(int capacity) {
        int align = MemoryUtil.pageSize();
        if (Integer.bitCount(align) != 1) {
            throw new IllegalArgumentException("Alignment must be a power of 2");
        }
        ByteBuffer buffer = ByteBuffer.allocateDirect(capacity + align);
        long address = MemoryUtil.getAddress(buffer);
        long offset = address & (long)(align - 1);
        if (offset == 0L) {
            buffer.limit(capacity);
        } else {
            int pos = (int)((long)align - offset);
            buffer.position(pos);
            buffer.limit(pos + capacity);
        }
        return buffer.slice();
    }

    @VisibleForTesting
    public static int roundUp(int size) {
        if (size <= 1024) {
            return BufferPool.roundUp(size, 32);
        }
        return BufferPool.roundUp(size, 2048);
    }

    @VisibleForTesting
    public static int roundUp(int size, int unit) {
        int mask = unit - 1;
        return size + mask & ~mask;
    }

    @VisibleForTesting
    public static void shutdownLocalCleaner(long timeout, TimeUnit unit) throws InterruptedException, TimeoutException {
        ExecutorUtils.shutdownNow(ImmutableList.of((Object)EXEC));
        ExecutorUtils.awaitTermination(timeout, unit, ImmutableList.of((Object)EXEC));
    }

    public static long unsafeGetBytesInUse() {
        long totalMemory = globalPool.memoryUsage.get();
        class L {
            long v;

            L() {
            }
        }
        L availableMemory = new L();
        for (Chunk chunk2 : globalPool.chunks) {
            availableMemory.v += (long)chunk2.capacity();
        }
        for (LocalPoolRef ref : localPoolReferences) {
            ref.chunks.forEach(chunk -> availableMemory.v += (long)chunk.free());
        }
        return totalMemory - availableMemory.v;
    }

    @VisibleForTesting
    static void unsafeReset() {
        ((LocalPool)localPool.get()).unsafeRecycle();
        globalPool.unsafeFree();
    }

    @VisibleForTesting
    static Chunk unsafeCurrentChunk() {
        return ((LocalPool)localPool.get()).chunks.chunk0;
    }

    @VisibleForTesting
    static int unsafeNumChunks() {
        LocalPool pool = (LocalPool)localPool.get();
        return (pool.chunks.chunk0 != null ? 1 : 0) + (pool.chunks.chunk1 != null ? 1 : 0) + (pool.chunks.chunk2 != null ? 1 : 0);
    }

    static {
        logger = LoggerFactory.getLogger(BufferPool.class);
        noSpamLogger = NoSpamLogger.getLogger(logger, 15L, TimeUnit.MINUTES);
        EMPTY_BUFFER = ByteBuffer.allocateDirect(0);
        globalPool = new GlobalPool();
        localPool = new FastThreadLocal<LocalPool>(){

            protected LocalPool initialValue() {
                return new LocalPool();
            }

            protected void onRemoval(LocalPool value) {
                value.release();
            }
        };
        localPoolReferences = Collections.newSetFromMap(new ConcurrentHashMap());
        localPoolRefQueue = new ReferenceQueue();
        EXEC = new InfiniteLoopExecutor("LocalPool-Cleaner", BufferPool::cleanupOneReference).start();
    }

    static final class Chunk {
        private final ByteBuffer slab;
        private final long baseAddress;
        private final int shift;
        private volatile long freeSlots;
        private static final AtomicLongFieldUpdater<Chunk> freeSlotsUpdater = AtomicLongFieldUpdater.newUpdater(Chunk.class, "freeSlots");
        private volatile LocalPool owner;
        private final Recycler recycler;
        @VisibleForTesting
        Object debugAttachment;

        Chunk(Chunk recycle) {
            assert (recycle.freeSlots == 0L);
            this.slab = recycle.slab;
            this.baseAddress = recycle.baseAddress;
            this.shift = recycle.shift;
            this.freeSlots = -1L;
            this.recycler = recycle.recycler;
        }

        Chunk(Recycler recycler, ByteBuffer slab) {
            assert (MemoryUtil.isExactlyDirect(slab));
            this.recycler = recycler;
            this.slab = slab;
            this.baseAddress = MemoryUtil.getAddress(slab);
            this.shift = 0x1F & Integer.numberOfTrailingZeros(slab.capacity() / 64);
            this.freeSlots = slab.capacity() == 0 ? 0L : -1L;
        }

        void acquire(LocalPool owner) {
            assert (this.owner == null);
            this.owner = owner;
        }

        void release() {
            this.owner = null;
            this.tryRecycle();
        }

        void tryRecycle() {
            assert (this.owner == null);
            if (this.isFree() && freeSlotsUpdater.compareAndSet(this, -1L, 0L)) {
                this.recycle();
            }
        }

        void recycle() {
            assert (this.freeSlots == 0L);
            this.recycler.recycle(this);
        }

        static Chunk getParentChunk(ByteBuffer buffer) {
            Object attachment = MemoryUtil.getAttachment(buffer);
            if (attachment instanceof Chunk) {
                return (Chunk)attachment;
            }
            if (attachment instanceof Ref) {
                return (Chunk)((Ref)attachment).get();
            }
            return null;
        }

        void setAttachment(ByteBuffer buffer) {
            if (Ref.DEBUG_ENABLED) {
                MemoryUtil.setAttachment(buffer, new Ref<Chunk>(this, null));
            } else {
                MemoryUtil.setAttachment(buffer, this);
            }
        }

        boolean releaseAttachment(ByteBuffer buffer) {
            Object attachment = MemoryUtil.getAttachment(buffer);
            if (attachment == null) {
                return false;
            }
            if (Ref.DEBUG_ENABLED) {
                ((Ref)attachment).release();
            }
            return true;
        }

        @VisibleForTesting
        long setFreeSlots(long val) {
            long ret = this.freeSlots;
            this.freeSlots = val;
            return ret;
        }

        int capacity() {
            return 64 << this.shift;
        }

        final int unit() {
            return 1 << this.shift;
        }

        final boolean isFree() {
            return this.freeSlots == -1L;
        }

        int free() {
            return Long.bitCount(this.freeSlots) * this.unit();
        }

        int freeSlotCount() {
            return Long.bitCount(this.freeSlots);
        }

        ByteBuffer get(int size) {
            return this.get(size, false, null);
        }

        ByteBuffer get(int size, boolean sizeIsLowerBound, ByteBuffer into) {
            long cur;
            int index;
            long candidate;
            int slotCount = size - 1 + this.unit() >>> this.shift;
            if (sizeIsLowerBound) {
                size = slotCount << this.shift;
            }
            if (slotCount > 64) {
                return null;
            }
            long slotBits = -1L >>> 64 - slotCount;
            long searchMask = 0x1111111111111111L;
            searchMask *= 15L >>> (slotCount - 1 & 3);
            searchMask &= -1L >>> slotCount - 1;
            do {
                if ((index = Long.numberOfTrailingZeros((cur = this.freeSlots) & searchMask)) == 64) {
                    return null;
                }
                searchMask ^= 1L << index;
            } while (((candidate = slotBits << index) & cur) != candidate);
            while (!freeSlotsUpdater.compareAndSet(this, cur, cur & (candidate ^ 0xFFFFFFFFFFFFFFFFL))) {
                cur = this.freeSlots;
                assert ((candidate & cur) == candidate);
            }
            return this.set(index << this.shift, size, into);
        }

        private ByteBuffer set(int offset, int size, ByteBuffer into) {
            if (into == null) {
                into = MemoryUtil.getHollowDirectByteBuffer(ByteOrder.BIG_ENDIAN);
            }
            MemoryUtil.sliceDirectByteBuffer(this.slab, into, offset, size);
            this.setAttachment(into);
            return into;
        }

        int roundUp(int v) {
            return BufferPool.roundUp(v, this.unit());
        }

        long free(ByteBuffer buffer, boolean tryRelease) {
            long next;
            long cur;
            if (!this.releaseAttachment(buffer)) {
                return 1L;
            }
            int size = this.roundUp(buffer.capacity());
            long address = MemoryUtil.getAddress(buffer);
            assert (address >= this.baseAddress & address + (long)size <= this.baseAddress + (long)this.capacity());
            int position = (int)(address - this.baseAddress) >> this.shift;
            int slotCount = size >> this.shift;
            long slotBits = -1L >>> 64 - slotCount;
            long shiftedSlotBits = slotBits << position;
            do {
                cur = this.freeSlots;
                next = cur | shiftedSlotBits;
                assert (next == (cur ^ shiftedSlotBits));
                if (!tryRelease || next != -1L) continue;
                next = 0L;
            } while (!freeSlotsUpdater.compareAndSet(this, cur, next));
            return next;
        }

        void freeUnusedPortion(ByteBuffer buffer) {
            long next;
            long cur;
            int capacity;
            int size = this.roundUp(buffer.limit());
            if (size == (capacity = this.roundUp(buffer.capacity()))) {
                return;
            }
            long address = MemoryUtil.getAddress(buffer);
            assert (address >= this.baseAddress & address + (long)size <= this.baseAddress + (long)this.capacity());
            int position = (int)(address + (long)size - this.baseAddress) >> this.shift;
            int slotCount = capacity - size >> this.shift;
            long slotBits = -1L >>> 64 - slotCount;
            long shiftedSlotBits = slotBits << position;
            do {
                cur = this.freeSlots;
                next = cur | shiftedSlotBits;
                assert (next == (cur ^ shiftedSlotBits));
            } while (!freeSlotsUpdater.compareAndSet(this, cur, next));
            MemoryUtil.setByteBufferCapacity(buffer, size);
        }

        public String toString() {
            return String.format("[slab %s, slots bitmap %s, capacity %d, free %d]", this.slab, Long.toBinaryString(this.freeSlots), this.capacity(), this.free());
        }

        @VisibleForTesting
        void unsafeFree() {
            Chunk parent = Chunk.getParentChunk(this.slab);
            if (parent != null) {
                parent.free(this.slab, false);
            } else {
                FileUtils.clean(this.slab);
            }
        }

        static void unsafeRecycle(Chunk chunk) {
            if (chunk != null) {
                chunk.owner = null;
                chunk.freeSlots = 0L;
                chunk.recycle();
            }
        }
    }

    private static final class LocalPoolRef
    extends PhantomReference<LocalPool> {
        private final MicroQueueOfChunks chunks;

        public LocalPoolRef(LocalPool localPool, ReferenceQueue<? super LocalPool> q) {
            super(localPool, q);
            this.chunks = localPool.chunks;
        }

        public void release() {
            this.chunks.release();
        }
    }

    public static final class LocalPool
    implements Recycler {
        private final Queue<ByteBuffer> reuseObjects;
        private final Supplier<Chunk> parent;
        private final LocalPoolRef leakRef;
        private final MicroQueueOfChunks chunks = new MicroQueueOfChunks();
        private LocalPool tinyPool;
        private final int tinyLimit;
        private boolean recycleWhenFree = true;

        public LocalPool() {
            this.parent = globalPool;
            this.tinyLimit = 1024;
            this.reuseObjects = new ArrayDeque<ByteBuffer>();
            this.leakRef = new LocalPoolRef(this, localPoolRefQueue);
            localPoolReferences.add(this.leakRef);
        }

        private LocalPool(LocalPool parent) {
            this.parent = () -> {
                ByteBuffer buffer = parent.tryGetInternal(2048, false);
                if (buffer == null) {
                    return null;
                }
                return new Chunk(parent, buffer);
            };
            this.tinyLimit = 0;
            this.reuseObjects = parent.reuseObjects;
            this.leakRef = new LocalPoolRef(this, localPoolRefQueue);
            localPoolReferences.add(this.leakRef);
        }

        private LocalPool tinyPool() {
            if (this.tinyPool == null) {
                this.tinyPool = new LocalPool(this).recycleWhenFree(this.recycleWhenFree);
            }
            return this.tinyPool;
        }

        public void put(ByteBuffer buffer) {
            Chunk chunk = Chunk.getParentChunk(buffer);
            if (chunk == null) {
                FileUtils.clean(buffer);
            } else {
                this.put(buffer, chunk);
            }
        }

        public void put(ByteBuffer buffer, Chunk chunk) {
            LocalPool owner = chunk.owner;
            if (owner != null && owner == this.tinyPool) {
                this.tinyPool.put(buffer, chunk);
                return;
            }
            long free = chunk.free(buffer, owner == null || owner == this && this.recycleWhenFree);
            if (free == 0L) {
                if (owner == this) {
                    this.remove(chunk);
                }
                chunk.recycle();
            } else if (free == -1L && owner != this && chunk.owner == null) {
                chunk.tryRecycle();
            }
            if (owner == this) {
                MemoryUtil.setAttachment(buffer, null);
                MemoryUtil.setDirectByteBuffer(buffer, 0L, 0);
                this.reuseObjects.add(buffer);
            }
        }

        public void putUnusedPortion(ByteBuffer buffer) {
            Chunk chunk = Chunk.getParentChunk(buffer);
            if (chunk == null) {
                return;
            }
            chunk.freeUnusedPortion(buffer);
        }

        public ByteBuffer get(int size) {
            return this.get(size, false);
        }

        public ByteBuffer getAtLeast(int size) {
            return this.get(size, true);
        }

        private ByteBuffer get(int size, boolean sizeIsLowerBound) {
            ByteBuffer ret = this.tryGet(size, sizeIsLowerBound);
            if (ret != null) {
                return ret;
            }
            if (size > 131072) {
                if (logger.isTraceEnabled()) {
                    logger.trace("Requested buffer size {} is bigger than {}; allocating directly", (Object)FBUtilities.prettyPrintMemory(size), (Object)FBUtilities.prettyPrintMemory(131072L));
                }
            } else if (logger.isTraceEnabled()) {
                logger.trace("Requested buffer size {} has been allocated directly due to lack of capacity", (Object)FBUtilities.prettyPrintMemory(size));
            }
            metrics.misses.mark();
            return BufferPool.allocate(size, BufferType.OFF_HEAP);
        }

        public ByteBuffer tryGet(int size) {
            return this.tryGet(size, false);
        }

        public ByteBuffer tryGetAtLeast(int size) {
            return this.tryGet(size, true);
        }

        private ByteBuffer tryGet(int size, boolean sizeIsLowerBound) {
            LocalPool pool = this;
            if (size <= this.tinyLimit) {
                if (size <= 0) {
                    if (size == 0) {
                        return EMPTY_BUFFER;
                    }
                    throw new IllegalArgumentException("Size must be non-negative (" + size + ')');
                }
                pool = this.tinyPool();
            } else if (size > 131072) {
                return null;
            }
            return pool.tryGetInternal(size, sizeIsLowerBound);
        }

        private ByteBuffer tryGetInternal(int size, boolean sizeIsLowerBound) {
            ByteBuffer result;
            ByteBuffer reuse = this.reuseObjects.poll();
            ByteBuffer buffer = this.chunks.get(size, sizeIsLowerBound, reuse);
            if (buffer != null) {
                return buffer;
            }
            Chunk chunk = this.addChunkFromParent();
            if (chunk != null && (result = chunk.get(size, sizeIsLowerBound, reuse)) != null) {
                return result;
            }
            if (reuse != null) {
                this.reuseObjects.add(reuse);
            }
            return null;
        }

        @Override
        public void recycle(Chunk chunk) {
            ByteBuffer buffer = chunk.slab;
            Chunk parentChunk = Chunk.getParentChunk(buffer);
            this.put(buffer, parentChunk);
        }

        private void remove(Chunk chunk) {
            this.chunks.remove(chunk);
            if (this.tinyPool != null) {
                this.tinyPool.chunks.removeIf((child, parent) -> Chunk.getParentChunk(((Chunk)child).slab) == parent, chunk);
            }
        }

        private Chunk addChunkFromParent() {
            Chunk chunk = this.parent.get();
            if (chunk == null) {
                return null;
            }
            this.addChunk(chunk);
            return chunk;
        }

        private void addChunk(Chunk chunk) {
            chunk.acquire(this);
            Chunk evict = this.chunks.add(chunk);
            if (evict != null) {
                if (this.tinyPool != null) {
                    this.tinyPool.chunks.removeIf((child, parent) -> Chunk.getParentChunk(((Chunk)child).slab) == parent, evict);
                }
                evict.release();
            }
        }

        public void release() {
            this.chunks.release();
            this.reuseObjects.clear();
            localPoolReferences.remove(this.leakRef);
            this.leakRef.clear();
            if (this.tinyPool != null) {
                this.tinyPool.release();
            }
        }

        @VisibleForTesting
        void unsafeRecycle() {
            this.chunks.unsafeRecycle();
        }

        public LocalPool recycleWhenFree(boolean recycleWhenFree) {
            this.recycleWhenFree = recycleWhenFree;
            if (this.tinyPool != null) {
                this.tinyPool.recycleWhenFree = recycleWhenFree;
            }
            return this;
        }
    }

    private static class MicroQueueOfChunks {
        private Chunk chunk0;
        private Chunk chunk1;
        private Chunk chunk2;
        private int count;

        private MicroQueueOfChunks() {
        }

        private Chunk add(Chunk chunk) {
            switch (this.count) {
                case 0: {
                    this.chunk0 = chunk;
                    this.count = 1;
                    break;
                }
                case 1: {
                    this.chunk1 = chunk;
                    this.count = 2;
                    break;
                }
                case 2: {
                    this.chunk2 = chunk;
                    this.count = 3;
                    break;
                }
                case 3: {
                    Chunk release;
                    int chunk0Free = this.chunk0.freeSlotCount();
                    int chunk1Free = this.chunk1.freeSlotCount();
                    int chunk2Free = this.chunk2.freeSlotCount();
                    if (chunk0Free < chunk1Free) {
                        if (chunk0Free < chunk2Free) {
                            release = this.chunk0;
                            this.chunk0 = chunk;
                        } else {
                            release = this.chunk2;
                            this.chunk2 = chunk;
                        }
                    } else if (chunk1Free < chunk2Free) {
                        release = this.chunk1;
                        this.chunk1 = chunk;
                    } else {
                        release = this.chunk2;
                        this.chunk2 = chunk;
                    }
                    return release;
                }
                default: {
                    throw new IllegalStateException();
                }
            }
            return null;
        }

        private void remove(Chunk chunk) {
            if (this.chunk0 == chunk) {
                this.chunk0 = this.chunk1;
                this.chunk1 = this.chunk2;
            } else if (this.chunk1 == chunk) {
                this.chunk1 = this.chunk2;
            } else if (this.chunk2 != chunk) {
                return;
            }
            this.chunk2 = null;
            --this.count;
        }

        ByteBuffer get(int size, boolean sizeIsLowerBound, ByteBuffer reuse) {
            if (null != this.chunk0) {
                ByteBuffer buffer = this.chunk0.get(size, sizeIsLowerBound, reuse);
                if (null != buffer) {
                    return buffer;
                }
                if (null != this.chunk1) {
                    buffer = this.chunk1.get(size, sizeIsLowerBound, reuse);
                    if (null != buffer) {
                        return buffer;
                    }
                    if (null != this.chunk2 && null != (buffer = this.chunk2.get(size, sizeIsLowerBound, reuse))) {
                        return buffer;
                    }
                }
            }
            return null;
        }

        private void forEach(Consumer<Chunk> consumer) {
            MicroQueueOfChunks.forEach(consumer, this.count, this.chunk0, this.chunk1, this.chunk2);
        }

        private void clearForEach(Consumer<Chunk> consumer) {
            Chunk chunk0 = this.chunk0;
            Chunk chunk1 = this.chunk1;
            Chunk chunk2 = this.chunk2;
            this.chunk2 = null;
            this.chunk1 = null;
            this.chunk0 = null;
            MicroQueueOfChunks.forEach(consumer, this.count, chunk0, chunk1, chunk2);
            this.count = 0;
        }

        private static void forEach(Consumer<Chunk> consumer, int count, Chunk chunk0, Chunk chunk1, Chunk chunk2) {
            switch (count) {
                case 3: {
                    consumer.accept(chunk2);
                }
                case 2: {
                    consumer.accept(chunk1);
                }
                case 1: {
                    consumer.accept(chunk0);
                }
            }
        }

        private <T> void removeIf(BiPredicate<Chunk, T> predicate, T value) {
            switch (this.count) {
                case 3: {
                    Chunk chunk;
                    if (predicate.test(this.chunk2, (Chunk)value)) {
                        --this.count;
                        chunk = this.chunk2;
                        this.chunk2 = null;
                        chunk.release();
                    }
                }
                case 2: {
                    Chunk chunk;
                    if (predicate.test(this.chunk1, (Chunk)value)) {
                        --this.count;
                        chunk = this.chunk1;
                        this.chunk1 = null;
                        chunk.release();
                    }
                }
                case 1: {
                    if (!predicate.test(this.chunk0, (Chunk)value)) break;
                    --this.count;
                    Chunk chunk = this.chunk0;
                    this.chunk0 = null;
                    chunk.release();
                    break;
                }
                case 0: {
                    return;
                }
            }
            switch (this.count) {
                case 2: {
                    if (this.chunk0 == null) {
                        this.chunk0 = this.chunk1;
                        this.chunk1 = this.chunk2;
                        this.chunk2 = null;
                        break;
                    }
                    if (this.chunk1 != null) break;
                    this.chunk1 = this.chunk2;
                    this.chunk2 = null;
                    break;
                }
                case 1: {
                    if (this.chunk1 != null) {
                        this.chunk0 = this.chunk1;
                        this.chunk1 = null;
                        break;
                    }
                    if (this.chunk2 == null) break;
                    this.chunk0 = this.chunk2;
                    this.chunk2 = null;
                }
            }
        }

        private void release() {
            this.clearForEach(Chunk::release);
        }

        private void unsafeRecycle() {
            this.clearForEach(Chunk::unsafeRecycle);
        }
    }

    static final class GlobalPool
    implements Supplier<Chunk>,
    Recycler {
        static final int MACRO_CHUNK_SIZE = 0x800000;
        private final Queue<Chunk> macroChunks = new ConcurrentLinkedQueue<Chunk>();
        private final Queue<Chunk> chunks = new ConcurrentLinkedQueue<Chunk>();
        private final AtomicLong memoryUsage = new AtomicLong();
        static final /* synthetic */ boolean $assertionsDisabled;

        GlobalPool() {
        }

        @Override
        public Chunk get() {
            Chunk chunk = this.chunks.poll();
            if (chunk != null) {
                return chunk;
            }
            chunk = this.allocateMoreChunks();
            if (chunk != null) {
                return chunk;
            }
            return this.chunks.poll();
        }

        private Chunk allocateMoreChunks() {
            Chunk chunk;
            long cur;
            do {
                if ((cur = this.memoryUsage.get()) + 0x800000L <= MEMORY_USAGE_THRESHOLD) continue;
                if (MEMORY_USAGE_THRESHOLD > 0L) {
                    noSpamLogger.info("Maximum memory usage reached ({}), cannot allocate chunk of {}", FBUtilities.prettyPrintMemory(MEMORY_USAGE_THRESHOLD), FBUtilities.prettyPrintMemory(0x800000L));
                }
                return null;
            } while (!this.memoryUsage.compareAndSet(cur, cur + 0x800000L));
            try {
                chunk = new Chunk(null, BufferPool.allocateDirectAligned(0x800000));
            }
            catch (OutOfMemoryError oom) {
                noSpamLogger.error("Buffer pool failed to allocate chunk of {}, current size {} ({}). Attempting to continue; buffers will be allocated in on-heap memory which can degrade performance. Make sure direct memory size (-XX:MaxDirectMemorySize) is large enough to accommodate off-heap memtables and caches.", FBUtilities.prettyPrintMemory(0x800000L), FBUtilities.prettyPrintMemory(this.sizeInBytes()), oom.toString());
                return null;
            }
            chunk.acquire(null);
            this.macroChunks.add(chunk);
            Chunk callerChunk = new Chunk(this, chunk.get(131072));
            if (debug != null) {
                debug.registerNormal(callerChunk);
            }
            for (int i = 131072; i < 0x800000; i += 131072) {
                Chunk add = new Chunk(this, chunk.get(131072));
                this.chunks.add(add);
                if (debug == null) continue;
                debug.registerNormal(add);
            }
            return callerChunk;
        }

        @Override
        public void recycle(Chunk chunk) {
            Chunk recycleAs = new Chunk(chunk);
            if (debug != null) {
                debug.recycleNormal(chunk, recycleAs);
            }
            this.chunks.add(recycleAs);
        }

        public long sizeInBytes() {
            return this.memoryUsage.get();
        }

        @VisibleForTesting
        void unsafeFree() {
            while (!this.chunks.isEmpty()) {
                this.chunks.poll().unsafeFree();
            }
            while (!this.macroChunks.isEmpty()) {
                this.macroChunks.poll().unsafeFree();
            }
            this.memoryUsage.set(0L);
        }

        static {
            boolean bl = $assertionsDisabled = !BufferPool.class.desiredAssertionStatus();
            if (!$assertionsDisabled && Integer.bitCount(131072) != 1) {
                throw new AssertionError();
            }
            if (!$assertionsDisabled && Integer.bitCount(0x800000) != 1) {
                throw new AssertionError();
            }
            logger.info("Global buffer pool limit is {}", (Object)FBUtilities.prettyPrintMemory(MEMORY_USAGE_THRESHOLD));
        }
    }

    static interface Recycler {
        public void recycle(Chunk var1);
    }

    static interface Debug {
        public void registerNormal(Chunk var1);

        public void recycleNormal(Chunk var1, Chunk var2);
    }
}

