/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.streaming;

import com.google.common.collect.Iterators;
import io.netty.util.concurrent.FastThreadLocalThread;
import java.io.EOFException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.util.Iterator;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.DoubleSupplier;
import org.apache.cassandra.db.streaming.CompressionInfo;
import org.apache.cassandra.io.compress.CompressionMetadata;
import org.apache.cassandra.io.util.DataInputPlus;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.RebufferingInputStream;
import org.apache.cassandra.utils.ChecksumType;
import org.apache.cassandra.utils.WrappedRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompressedInputStream
extends RebufferingInputStream
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(CompressedInputStream.class);
    private final CompressionInfo info;
    private final BlockingQueue<ByteBuffer> dataBuffer;
    private final DoubleSupplier crcCheckChanceSupplier;
    private long bufferOffset = 0L;
    private long current = 0L;
    private final ChecksumType checksumType;
    private static final int CHECKSUM_LENGTH = 4;
    private static final ByteBuffer POISON_PILL = ByteBuffer.wrap(new byte[0]);
    private volatile IOException readException = null;
    private long totalCompressedBytesRead;

    public CompressedInputStream(DataInputPlus source, CompressionInfo info, ChecksumType checksumType, DoubleSupplier crcCheckChanceSupplier) {
        super(ByteBuffer.allocateDirect(info.parameters.chunkLength()));
        this.buffer.limit(this.buffer.position());
        this.info = info;
        this.dataBuffer = new ArrayBlockingQueue<ByteBuffer>(Math.min(info.chunks.length, 1024));
        this.crcCheckChanceSupplier = crcCheckChanceSupplier;
        this.checksumType = checksumType;
        new FastThreadLocalThread((Runnable)new Reader(source, info, this.dataBuffer)).start();
    }

    public void position(long position) throws IOException {
        if (this.readException != null) {
            throw this.readException;
        }
        assert (position >= this.current) : "stream can only read forward.";
        this.current = position;
        if (this.current > this.bufferOffset + (long)this.buffer.limit()) {
            this.reBuffer(false);
        }
        this.buffer.position((int)(this.current - this.bufferOffset));
    }

    @Override
    protected void reBuffer() throws IOException {
        this.reBuffer(true);
    }

    private void reBuffer(boolean updateCurrent) throws IOException {
        if (this.readException != null) {
            FileUtils.clean(this.buffer);
            this.buffer = null;
            throw this.readException;
        }
        if (updateCurrent) {
            this.current += (long)this.buffer.position();
        }
        try {
            ByteBuffer compressedWithCRC = this.dataBuffer.take();
            if (compressedWithCRC == POISON_PILL) {
                assert (this.readException != null);
                throw this.readException;
            }
            this.decompress(compressedWithCRC);
        }
        catch (InterruptedException e) {
            throw new EOFException("No chunk available");
        }
    }

    private void decompress(ByteBuffer compressed) throws IOException {
        boolean releaseCompressedBuffer;
        int length = compressed.remaining();
        if (length - 4 < this.info.parameters.maxCompressedLength()) {
            this.buffer.clear();
            compressed.limit(length - 4);
            this.info.parameters.getSstableCompressor().uncompress(compressed, this.buffer);
            this.buffer.flip();
            releaseCompressedBuffer = true;
        } else {
            FileUtils.clean(this.buffer);
            this.buffer = compressed;
            this.buffer.limit(length - 4);
            releaseCompressedBuffer = false;
        }
        this.totalCompressedBytesRead += (long)length;
        double crcCheckChance = this.crcCheckChanceSupplier.getAsDouble();
        if (crcCheckChance >= 1.0 || crcCheckChance > 0.0 && crcCheckChance > ThreadLocalRandom.current().nextDouble()) {
            ByteBuffer crcBuf = compressed.duplicate();
            crcBuf.limit(length - 4).position(0);
            int checksum = (int)this.checksumType.of(crcBuf);
            crcBuf.limit(length);
            if (crcBuf.getInt() != checksum) {
                throw new IOException("CRC unmatched");
            }
        }
        if (releaseCompressedBuffer) {
            FileUtils.clean(compressed);
        }
        int compressedChunkLength = this.info.parameters.chunkLength();
        this.bufferOffset = this.current & (long)(~(compressedChunkLength - 1));
    }

    public long getTotalCompressedBytesRead() {
        return this.totalCompressedBytesRead;
    }

    @Override
    public void close() {
        if (this.buffer != null) {
            FileUtils.clean(this.buffer);
            this.buffer = null;
        }
    }

    class Reader
    extends WrappedRunnable {
        private final DataInputPlus source;
        private final Iterator<CompressionMetadata.Chunk> chunks;
        private final BlockingQueue<ByteBuffer> dataBuffer;

        Reader(DataInputPlus source, CompressionInfo info, BlockingQueue<ByteBuffer> dataBuffer) {
            this.source = source;
            this.chunks = Iterators.forArray((Object[])info.chunks);
            this.dataBuffer = dataBuffer;
        }

        @Override
        protected void runMayThrow() throws Exception {
            byte[] tmp = null;
            while (this.chunks.hasNext()) {
                CompressionMetadata.Chunk chunk = this.chunks.next();
                int readLength = chunk.length + 4;
                ByteBuffer compressedWithCRC = null;
                try {
                    int r;
                    if (this.source instanceof ReadableByteChannel) {
                        compressedWithCRC = ByteBuffer.allocateDirect(readLength);
                        r = ((ReadableByteChannel)((Object)this.source)).read(compressedWithCRC);
                        compressedWithCRC.flip();
                    } else {
                        if (tmp == null || tmp.length < ((CompressedInputStream)CompressedInputStream.this).info.parameters.chunkLength() + 4) {
                            tmp = new byte[((CompressedInputStream)CompressedInputStream.this).info.parameters.chunkLength() + 4];
                        }
                        this.source.readFully(tmp, 0, readLength);
                        compressedWithCRC = ByteBuffer.allocateDirect(readLength);
                        compressedWithCRC.put(tmp, 0, readLength);
                        compressedWithCRC.position(0);
                        r = readLength;
                    }
                    if (r < 0) {
                        FileUtils.clean(compressedWithCRC);
                        CompressedInputStream.this.readException = new EOFException("No chunk available");
                        this.dataBuffer.put(POISON_PILL);
                        return;
                    }
                }
                catch (IOException e) {
                    if (!(e instanceof EOFException)) {
                        logger.warn("Error while reading compressed input stream.", (Throwable)e);
                    }
                    if (compressedWithCRC != null) {
                        FileUtils.clean(compressedWithCRC);
                    }
                    CompressedInputStream.this.readException = e;
                    this.dataBuffer.put(POISON_PILL);
                    return;
                }
                this.dataBuffer.put(compressedWithCRC);
            }
        }
    }
}

